package be.ehealth.businessconnector.ehboxV2.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.businessconnector.ehbox.api.domain.Addressee;
import be.ehealth.businessconnector.ehbox.api.domain.Document;
import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.domain.NewsMessage;
import be.ehealth.businessconnector.ehbox.api.utils.QualityType;
import be.ehealth.businessconnector.ehboxV2.AbstractEhboxV2IntegrationTest;
import be.ehealth.businessconnector.ehboxV2.builders.BuilderFactory;
import be.ehealth.businessconnector.ehboxV2.builders.SendMessageBuilder;
import be.ehealth.businessconnector.ehboxV2.testutil.EhboxTestUtil;
import be.ehealth.technicalconnector.service.sts.security.SAMLToken;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.ehealth.technicalconnector.utils.IdentifierType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.Message;
import be.fgov.ehealth.ehbox.core.v2.CustomMetaType;
import be.fgov.ehealth.ehbox.publication.protocol.v2.EhBoxPublicationPortType;
import be.fgov.ehealth.ehbox.publication.protocol.v2.SendMessageRequest;
import be.fgov.ehealth.ehbox.publication.protocol.v2.SendMessageResponse;


/**
 * eHealthBox Service Integration Tests This test shows the use of the eHealthBox Service through the use of the Business Connector.
 * 
 * The tests below use the Session Management Service to manage the session and SAML/STS token. The eHealthBox of the test user specified in
 * the business connector property file is used, these tests expect that this is set to a valid INSS of a citizen.
 */
public class PublicationServiceIntegrationTest extends AbstractEhboxV2IntegrationTest {

    /**
     * Properties file
     */
    private static final String FILE_AS_DOC = "/LOREM_IPSUM.txt";

    private static final String ZIP_FILE_FOR_ANNEX = "/500Kb.zip";

    /**
     * Test Session user
     */
    private static final String TEST_SESSION_USER = "82051234978";

    private static final Logger LOG = LoggerFactory.getLogger(PublicationServiceIntegrationTest.class);

    private static SAMLToken token;


    /**
     * Test the sendMessage operation from the eHealthBox Service through the use of the Business Connector. This test creates and sends a
     * message to a predefined eHealthBox, based on a person's INSS
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testSendNewsEncryptedMessage() throws Exception {
        /*
         * Create the parameters of the new message
         */
        Assert.assertNotNull("crypto in session should not be null", Session.getInstance().getSession().getEncryptionCrypto());
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();
        Assert.assertNotNull(builder);


        NewsMessage<Message> news = new NewsMessage<Message>();
        news.setPublicationId(EhboxTestUtil.generatePublicationId());
        Document newsDocument = new Document();
        news.setDocument(newsDocument);
        newsDocument.setTitle("Encrypted news " + new Date());
        newsDocument.setContent("eureka".getBytes());
        newsDocument.setMimeType("text/xml");

        news.setDestinations(getDestination());
        news.setImportant(true);
        news.setEncrypted(true);

        SendMessageRequest request = builder.buildMessage(news);

        CustomMetaType type = new CustomMetaType();
        type.setValue("");
        type.setKey("");


        /*
         * Invoke the business connector framework's eHealthBox's sendMessage operation
         */
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxPublicationPortType service = ServiceFactory.getPublicationService(token);
        SendMessageResponse response = service.sendMessage(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getId());

        LOG.debug("Message Sent: " + response.getId());

    }

    /**
     * Sens a newsMessages and verifies the response
     * 
     * @throws Exception
     */
    @Before
    public void testSendNewsMessage() throws Exception {
        /*
         * Create the parameters of the new message
         */
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();


        NewsMessage<Message> news = new NewsMessage<Message>();
        news.setPublicationId(EhboxTestUtil.generatePublicationId());

        Document newsDocument = new Document();
        news.setDocument(newsDocument);
        newsDocument.setTitle("News " + new Date());
        newsDocument.setContent("eureka".getBytes());
        newsDocument.setMimeType("text/xml");

        news.setDestinations(getDestination());
        news.setImportant(true);
        news.setEncrypted(false);

        SendMessageRequest request = builder.buildMessage(news);
        ConnectorXmlUtils.logXmlObject(request);


        /*
         * Invoke the business connector framework's eHealthBox's sendMessage operation
         */

        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxPublicationPortType service = ServiceFactory.getPublicationService(token);
        SendMessageResponse response = service.sendMessage(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getId());

        LOG.debug("Message Sent: " + response.getId());

    }

    /**
     * Sends an encrypted document with annex and verifies the response
     * 
     * @throws Exception
     */
    @Test
    public void testSendEncryptedDocumentWithAnnex() throws Exception {
        /*
         * Create the parameters of the new message
         */
        Assert.assertNotNull("crypto in session should not be null", Session.getInstance().getSession().getEncryptionCrypto());
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();
        Assert.assertNotNull(builder);

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();
        documentMsg.setPublicationId(EhboxTestUtil.generatePublicationId());
        documentMsg.setDestinations(getDestination());
        documentMsg.setImportant(false);
        documentMsg.setEncrypted(true);

        documentMsg.setFreeText("Free text:" + UUID.randomUUID());
        documentMsg.setPatientInss(TEST_SESSION_USER);

        Document doc = new Document();
        doc.setTitle("testSendEncryptedDocumentWithAnnex : Encrypted document " + new Date());
        doc.setContent(PublicationServiceIntegrationTest.class.getResourceAsStream(FILE_AS_DOC));
        doc.setMimeType("text/plain");
        doc.setFilename("be.ehealth.businessconnector.properties");
        documentMsg.setDocument(doc);

        Document annex = new Document();
        annex.setTitle("properties");
        annex.setContent(PublicationServiceIntegrationTest.class.getResourceAsStream(ZIP_FILE_FOR_ANNEX));
        annex.setFilename("be.ehealth.technicalconnector.properties");
        annex.setMimeType("application/octet-stream");
        documentMsg.getAnnexList().add(annex);

        SendMessageRequest request = builder.buildMessage(documentMsg);
        LOG.debug("created request : \n" + ConnectorXmlUtils.toString(request));
        /*
         * Invoke the business connector framework's eHealthBox's sendMessage operation
         */

        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxPublicationPortType service = ServiceFactory.getPublicationService(token);
        SendMessageResponse response = service.sendMessage(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getId());

        LOG.debug("Message Sent: " + response.getId());


    }

    /**
     * @throws Exception
     */
    @Test
    public void testSendDocumentWithAnnex() throws Exception {
        /*
         * Create the parameters of the new message
         */
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();
        documentMsg.setPublicationId(EhboxTestUtil.generatePublicationId());
        documentMsg.setDestinations(getDestination());
        documentMsg.setImportant(false);

        documentMsg.setFreeText("Free text:" + UUID.randomUUID());


        Document doc = new Document();
        doc.setTitle("Document " + new Date());
        doc.setContent(PublicationServiceIntegrationTest.class.getResourceAsStream(FILE_AS_DOC));
        doc.setMimeType("text/plain");
        doc.setFilename("be.ehealth.businessconnector.properties");
        documentMsg.setDocument(doc);

        Document annex = new Document();
        annex.setTitle("properties");
        annex.setContent(PublicationServiceIntegrationTest.class.getResourceAsStream(FILE_AS_DOC));
        annex.setFilename("be.ehealth.technicalconnector.properties");
        annex.setMimeType("text/plain");
        documentMsg.getAnnexList().add(annex);

        SendMessageRequest request = builder.buildMessage(documentMsg);
        /*
         * Invoke the business connector framework's eHealthBox's sendMessage operation
         */
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxPublicationPortType service = ServiceFactory.getPublicationService(token);
        SendMessageResponse response = service.sendMessage(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getId());

        LOG.debug("Message Sent: " + response.getId());


    }

    /**
     * Returns the destination
     * 
     * @return
     */
    private List<Addressee> getDestination() {
        List<Addressee> addresseeList = new ArrayList<Addressee>();
        Addressee addressee = new Addressee(IdentifierType.SSIN);
        addressee.setId(TEST_SESSION_USER);
        addressee.setQuality(QualityType.GENERAL_PRACTIONER_SSIN);
        addresseeList.add(addressee);

        return addresseeList;
    }

    /**
     * @throws Exception
     */
    @Test
    public void testSendDocumentWithNullAnnex() throws Exception {
        /*
         * Create the parameters of the new message
         */
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();
        documentMsg.setPublicationId(EhboxTestUtil.generatePublicationId());
        documentMsg.setDestinations(getDestination());
        documentMsg.setImportant(false);

        documentMsg.setFreeText("Free text:" + UUID.randomUUID());

        Document doc = new Document();
        doc.setTitle(documentMsg.getPublicationId() + " Döé\"'(§è!çà)-ument   |; ¢¾ñ");
        doc.setContent(PublicationServiceIntegrationTest.class.getResourceAsStream(FILE_AS_DOC));
        doc.setMimeType("text/plain");
        doc.setFilename("be.ehealth.businessconnector.properties");
        documentMsg.setDocument(doc);

        Document annex = new Document();
        annex.setTitle("properties");
        annex.setContent(new byte[]{});
        annex.setFilename("be.ehealth.technicalconnector.properties");
        annex.setMimeType("text/plain");
        documentMsg.getAnnexList().add(annex);

        SendMessageRequest request = builder.buildMessage(documentMsg);

        /*
         * Invoke the business connector framework's eHealthBox's sendMessage operation
         */
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxPublicationPortType service = ServiceFactory.getPublicationService(token);
        SendMessageResponse response = service.sendMessage(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getId());

        LOG.debug("Message Sent: " + response.getId());


    }

}
