package be.ehealth.businessconnector.ehboxV2.session;

import java.util.List;
import java.util.Properties;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.domain.NewsMessage;
import be.ehealth.businessconnector.ehboxV2.AbstractEhboxV2IntegrationTest;
import be.ehealth.businessconnector.ehboxV2.builders.BuilderFactory;
import be.ehealth.businessconnector.ehboxV2.testutil.EhboxTestUtil;
import be.fgov.ehealth.commons.protocol.v1.ResponseType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetBoxInfoRequest;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetBoxInfoResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetFullMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetHistoryResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetMessageListResponseType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetMessagesListRequest;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.Message;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.MessageRequestType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.MoveMessageRequest;
import be.fgov.ehealth.ehbox.publication.protocol.v2.SendMessageResponse;


/**
 * eHealthBox Service Integration Tests This test shows the use of the eHealthBox Service through the use of the Business Connector.
 * 
 * The tests below use the Session Management Service to manage the session and SAML/STS token. The eHealthBox of the test user specified in
 * the business connector property file is used, these tests expect that this is set to a valid INSS of a citizen.
 */
public class EhboxV2ServiceConsultationIntegrationTest extends AbstractEhboxV2IntegrationTest {

    /**
     * Property key test.session.user
     */
    private static final String PROP_TEST_SESSION_USER = "test.session.user";

    private static final Logger LOG = LoggerFactory.getLogger(EhboxV2ServiceConsultationIntegrationTest.class);

    private static Properties props;

    /**
     * Response ID
     */
    private String messageId;

    /**
     * Initialise messageID if not already done
     * 
     * @throws Exception
     */
    @Before
    public void init() throws Exception {

        /*
         * Create the parameters of the new message
         */
        // set the folder (INBOX | SENTBOX | BININBOX | BINSENTBOX)
        String source = "INBOX";

        GetMessagesListRequest request = new GetMessagesListRequest();
        request.setSource(source);


        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */

        EhealthBoxServiceV2 service = ServiceFactory.getEhealthBoxServiceV2();
        GetMessageListResponseType response = service.getMessageList(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getMessages());
        // print out the messageId's for the given folder
        List<Message> list = response.getMessages();
        LOG.info("Messages in " + source);
        if (list.size() > 0) {
            messageId = list.get(0).getMessageId();
        }
    }

    /**
     * Test the getBoxInfo operation from the eHealthBox Service through the use of the Business Connector. The getBoxInfo retrieves general
     * info about an eHealthBox
     * 
     * The following main steps are done: - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testGetBoxInfo() throws Exception {
        /*
         * Invoke the business connector framework's eHealthBox's getBoxInfo operation
         */
        EhealthBoxServiceV2 service = ServiceFactory.getEhealthBoxServiceV2();
        GetBoxInfoRequest getBoxInfoRequest = new GetBoxInfoRequest();
        GetBoxInfoResponse response = service.getBoxInfo(getBoxInfoRequest);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if there is any content
        Assert.assertNotNull(response.getBoxId());
    }


    /**
     * Test the getMessageList operation from the eHealthBox Service through the use of the Business Connector. This test retrieves the
     * messageId's of the message's in a given folder, correct values for the folder are: INBOX | SENT | BIN
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testGetMessageList() throws Exception {
        /*
         * Create the parameters of the new message
         */
        // set the folder (INBOX | SENTBOX | BININBOX | BINSENTBOX)
        String source = "INBOX";

        GetMessagesListRequest request = new GetMessagesListRequest();
        request.setSource(source);


        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */

        EhealthBoxServiceV2 service = ServiceFactory.getEhealthBoxServiceV2();
        GetMessageListResponseType response = service.getMessageList(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getMessages());
        // print out the messageId's for the given folder
        List<Message> list = response.getMessages();
        LOG.info("Messages in " + source);

        for (Message message : list) {
            // JAXB.marshal(message, System.out);
            be.ehealth.businessconnector.ehbox.api.domain.Message<Message> msg = BuilderFactory.getConsultationMessageBuilder().buildMessage(message);

            if (msg instanceof NewsMessage) {
                NewsMessage<Message> news = (NewsMessage<Message>) msg;
                LOG.info("\t" + news);

            } else if (msg instanceof DocumentMessage) {
                DocumentMessage<Message> doc = (DocumentMessage<Message>) msg;
                LOG.debug(doc.getDocumentTitle());
                LOG.info("\t" + doc);
            } else {
                LOG.info("\t" + message.getMessageId());
            }
        }
    }

    /**
     * Test the getFullMessage operation from the eHealthBox Service through the use of the Business Connector. This test retrieves a full
     * message based on a given messageId's. Usually the messageId is retrieved first from the getMessageList operation.
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testGetFullMessage() throws Exception {
        /*
         * Create the parameters of the new message
         */
        MessageRequestType request = new MessageRequestType();
        // set the messageId, use the messageId of the message that has been sent in testSendMessage

        request.setMessageId(messageId);
        request.setSource("INBOX");

        /*
         * Invoke the business connector framework's eHealthBox's getFullMessage operation
         */

        EhealthBoxServiceV2 service = ServiceFactory.getEhealthBoxServiceV2();
        GetFullMessageResponse response = service.getFullMessage(request);

        /*
         * Verify the response
         */

        // check if the message has been returned
        Assert.assertNotNull(response.getMessage().getPublicationId());
        be.ehealth.businessconnector.ehbox.api.domain.Message<GetFullMessageResponse> msg = BuilderFactory.getConsultationMessageBuilder().buildFullMessage(response);
        if (msg instanceof DocumentMessage) {
            DocumentMessage<GetFullMessageResponse> message = (DocumentMessage<GetFullMessageResponse>) msg;
            LOG.debug(message.getDocumentTitle());
            LOG.debug(message.toString());
        } else if (msg instanceof NewsMessage) {
            NewsMessage<GetFullMessageResponse> message = (NewsMessage<GetFullMessageResponse>) msg;
            LOG.debug(message.toString());
        }
    }


    /**
     * Test the getHistory operation from the eHealthBox Service through the use of the Business Connector. This test retrieves the history
     * based on a given messageId. the list that is returned is a list of Strings containing messageId's.
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testGetHistory() throws Exception {
        /*
         * Create the parameters of the new message
         */
        MessageRequestType request = new MessageRequestType();
        // set the messageId, use the messageId of the message that has been sent in testSendMessage
        request.setMessageId(messageId);
        request.setSource("INBOX");

        /*
         * Invoke the business connector framework's eHealthBox's getFullMessage operation
         */
        EhealthBoxServiceV2 service = ServiceFactory.getEhealthBoxServiceV2();
        GetHistoryResponse response = service.getMessageHistory(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if the message has been returned
        Assert.assertNotNull(response.getMessageIds());
    }

    /**
     * Test the moveToInbox operation from the eHealthBox Service through the use of the Business Connector. This moves a message from the
     * BIN folder to the INBOX folder based on a given messageId.
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    @Ignore
    public void testMoveToInbox() throws Exception {

        EhealthBoxServiceV2 service = ServiceFactory.getEhealthBoxServiceV2();
        SendMessageResponse sendMessage = service.sendMessage(BuilderFactory.getSendMessageBuilder().buildMessage(EhboxTestUtil.createFilledDocumentMessage(this.getClass().getCanonicalName() + ".testMoveToInbox")));

        Thread.sleep(2500); // allow the service to process the message.

        GetMessagesListRequest messagesListRequest = new GetMessagesListRequest();
        messagesListRequest.setSource("SENTBOX");
        GetMessageListResponseType messageList = service.getMessageList(messagesListRequest);
        Assert.assertEquals("100", messageList.getStatus().getCode());
        Assert.assertTrue("there should be at least one message in the sendmessage list", messageList.getMessages().size() > 1);
        String messageIdToUse = messageList.getMessages().get(0).getMessageId();


        /*
         * Create the parameters of the new message
         */
        MoveMessageRequest request = new MoveMessageRequest();
        // set the messageId, use the messageId of the message that has been sent in testSendMessage
        request.getMessageIds().add(messageIdToUse);
        request.setSource("SENTBOX");
        request.setDestination("BINSENTBOX");

        /*
         * Invoke the business connector framework's eHealthBox's getFullMessage operation
         */
        ResponseType response = service.moveMessage(request);


        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
    }


}
