package be.ehealth.businessconnector.ehboxV2.service;

import java.util.List;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.domain.NewsMessage;
import be.ehealth.businessconnector.ehboxV2.AbstractEhboxV2IntegrationTest;
import be.ehealth.businessconnector.ehboxV2.builders.BuilderFactory;
import be.ehealth.businessconnector.ehboxV2.testutil.EhboxTestUtil;
import be.ehealth.technicalconnector.service.sts.security.SAMLToken;
import be.ehealth.technicalconnector.session.Session;
import be.fgov.ehealth.commons.protocol.v1.ResponseType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.DeleteMessageRequest;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.DeleteMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.EhBoxConsultationPortType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetBoxInfoRequest;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetBoxInfoResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetFullMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetHistoryResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetMessageListResponseType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetMessagesListRequest;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.Message;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.MessageRequestType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.MoveMessageRequest;
import be.fgov.ehealth.ehbox.publication.protocol.v2.EhBoxPublicationPortType;
import be.fgov.ehealth.ehbox.publication.protocol.v2.SendMessageResponse;


/**
 * eHealthBox Service Integration Tests This test shows the use of the eHealthBox Service through the use of the Business Connector.
 * 
 * The tests below use the Session Management Service to manage the session and SAML/STS token. The eHealthBox of the test user specified in
 * the business connector property file is used, these tests expect that this is set to a valid INSS of a citizen.
 */
public class ConsultationServiceImplIntegrationTest extends AbstractEhboxV2IntegrationTest {


    private static final Logger LOG = LoggerFactory.getLogger(ConsultationServiceImplIntegrationTest.class);

    private static SAMLToken token;

    /**
     * Id of the response
     */
    private static String messageId;

    /**
     * Test the getBoxInfo operation from the eHealthBox Service through the use of the Business Connector. The getBoxInfo retrieves general
     * info about an eHealthBox
     * 
     * The following main steps are done: - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testGetBoxInfo() throws Exception {
        /*
         * Invoke the business connector framework's eHealthBox's getBoxInfo operation
         */
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        GetBoxInfoRequest getBoxInfoRequest = new GetBoxInfoRequest();
        GetBoxInfoResponse response = service.getBoxInfo(getBoxInfoRequest);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if there is any content
        Assert.assertNotNull(response.getBoxId());
    }

    /**
     * Initialize messageId if not already done
     * 
     * @throws Exception
     */
    @Before
    public void init() throws Exception {

        /*
         * Create the parameters of the new message
         */
        // set the folder (INBOX | SENTBOX | BININBOX | BINSENTBOX)
        String source = "INBOX";

        GetMessagesListRequest request = new GetMessagesListRequest();
        request.setSource(source);


        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        GetMessageListResponseType response = service.getMessagesList(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getMessages());
        // print out the messageId's for the given folder
        List<Message> list = response.getMessages();
        LOG.info("Messages in " + source);

        if (list.size() > 0) {
            messageId = list.get(0).getMessageId();
        }

    }

    /**
     * Test the getMessageList operation from the eHealthBox Service through the use of the Business Connector. This test retrieves the
     * messageId's of the message's in a given folder, correct values for the folder are: INBOX | SENT | BIN
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testGetMessageList() throws Exception {
        /*
         * Create the parameters of the new message
         */
        // set the folder (INBOX | SENTBOX | BININBOX | BINSENTBOX)
        String source = "INBOX";

        GetMessagesListRequest request = new GetMessagesListRequest();
        request.setSource(source);


        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        GetMessageListResponseType response = service.getMessagesList(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getMessages());
        // print out the messageId's for the given folder
        List<Message> list = response.getMessages();
        LOG.info("Messages in " + source);
        for (Message message : list) {
            // JAXB.marshal(message, System.out);
            be.ehealth.businessconnector.ehbox.api.domain.Message<Message> msg = BuilderFactory.getConsultationMessageBuilder().buildMessage(message);
            if (msg instanceof NewsMessage) {
                NewsMessage<Message> news = (NewsMessage<Message>) msg;
                LOG.info("\t" + news);

            } else if (msg instanceof DocumentMessage) {
                DocumentMessage<Message> doc = (DocumentMessage<Message>) msg;
                LOG.debug(doc.getDocumentTitle());
                LOG.info("\t" + doc);
            } else {
                LOG.info("\t" + message.getMessageId());
            }
        }
    }

    /**
     * Test the getFullMessage operation from the eHealthBox Service through the use of the Business Connector. This test retrieves a full
     * message based on a given messageId's. Usually the messageId is retrieved first from the getMessageList operation.
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    @Ignore("add message id manually to test look in log from getMessageList for a messageId ")
    public void testGetFullMessage() throws Exception {
        /*
         * Create the parameters of the new message
         */
        MessageRequestType request = new MessageRequestType();
        // set the messageId, use the messageId of the message that has been sent in testSendMessage
        // request.setMessageId("9Y0000CTPK00P");
        request.setSource("INBOX");

        /*
         * Invoke the business connector framework's eHealthBox's getFullMessage operation
         */

        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        GetFullMessageResponse response = service.getFullMessage(request);

        /*
         * Verify the response
         */

        // check if the message has been returned
        Assert.assertNotNull(response.getMessage().getPublicationId());
        be.ehealth.businessconnector.ehbox.api.domain.Message<GetFullMessageResponse> msg = BuilderFactory.getConsultationMessageBuilder().buildFullMessage(response);
        if (msg instanceof DocumentMessage) {
            DocumentMessage<GetFullMessageResponse> message = (DocumentMessage<GetFullMessageResponse>) msg;
            LOG.debug(message.getDocumentTitle());
            LOG.debug(message.toString());
        } else if (msg instanceof NewsMessage) {
            NewsMessage<GetFullMessageResponse> message = (NewsMessage<GetFullMessageResponse>) msg;
            LOG.debug(message.toString());
        }
    }


    /**
     * Test the getHistory operation from the eHealthBox Service through the use of the Business Connector. This test retrieves the history
     * based on a given messageId. the list that is returned is a list of Strings containing messageId's.
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testGetHistory() throws Exception {
        /*
         * Create the parameters of the new message
         */
        MessageRequestType request = new MessageRequestType();
        // set the messageId, use the messageId of the message that has been sent in testSendMessage
        request.setMessageId(messageId);
        request.setSource("INBOX");

        /*
         * Invoke the business connector framework's eHealthBox's getFullMessage operation
         */
        token = Session.getInstance().getSession().getSAMLToken();

        EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        GetHistoryResponse response = service.getMessageHistory(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if the message has been returned
        Assert.assertNotNull(response.getMessageIds());
    }

    /**
     * Test the moveToInbox operation from the eHealthBox Service through the use of the Business Connector. This moves a message from the
     * BIN folder to the INBOX folder based on a given messageId.
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testMoveToInbox() throws Exception {
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxPublicationPortType publicationService = ServiceFactory.getPublicationService(token);
        SendMessageResponse sendMessage = publicationService.sendMessage(BuilderFactory.getSendMessageBuilder().buildMessage(EhboxTestUtil.createFilledDocumentMessage(this.getClass().getCanonicalName() + ".testMoveToInbox")));
        String sendMessageId = sendMessage.getId();
        Assert.assertEquals("100", sendMessage.getStatus().getCode());
        LOG.debug("testMoveToInbox : sleeping ");
        Thread.sleep(2500); // allow the service to process the message.
        EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        GetMessagesListRequest messagesListRequest = new GetMessagesListRequest();
        messagesListRequest.setSource("SENTBOX");
        GetMessageListResponseType messageList = service.getMessagesList(messagesListRequest);
        Assert.assertEquals("100", messageList.getStatus().getCode());
        Assert.assertTrue("there should be at least one message in the sendmessage list", messageList.getMessages().size() > 1);
        String messageIdToUse = messageList.getMessages().get(0).getMessageId();

        /*
         * Create the parameters of the new message
         */
        MoveMessageRequest request = new MoveMessageRequest();
        // set the messageId, use the messageId of the message that has been sent in testSendMessage
        request.getMessageIds().add(messageIdToUse);
        request.setSource("SENTBOX");
        request.setDestination("BINSENTBOX");

        /*
         * Invoke the business connector framework's eHealthBox's getFullMessage operation
         */

        token = Session.getInstance().getSession().getSAMLToken();
        // EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        ResponseType response = service.moveMessage(request);


        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
    }

    /**
     * Test the getMessageList operation from the eHealthBox Service through the use of the Business Connector. This test retrieves the
     * messageId's of the message's in a given folder, correct values for the folder are: INBOX | SENT | BIN
     * 
     * The following main steps are done: - Create the parameters of the new message - Invoke the business connector - Verify the response
     * 
     * @throws Exception
     */
    @Test
    public void testDeleteMessage() throws Exception {
        /*
         * Create the parameters of the message to delete
         */
        // set the folder (INBOX | SENTBOX | BININBOX | BINSENTBOX)
        String source = "INBOX";

        DeleteMessageRequest request = new DeleteMessageRequest();
        request.setSource(source);
        request.getMessageIds().add(messageId);


        /*
         * Invoke the business connector framework's eHealthBox's deleteMessage operation
         */
        token = Session.getInstance().getSession().getSAMLToken();
        EhBoxConsultationPortType service = ServiceFactory.getConsultationService(token);
        DeleteMessageResponse response = service.deleteMessage(request);

        /*
         * Verify the response
         */
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
    }


}
