/*
t * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.ehboxV2.builders.impl;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.security.cert.X509Certificate;
import java.util.HashSet;
import java.util.Set;

import javax.security.auth.x500.X500Principal;

import org.easymock.EasyMock;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import be.ehealth.businessconnector.ehbox.api.domain.Addressee;
import be.ehealth.businessconnector.ehbox.api.domain.Document;
import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.domain.NewsMessage;
import be.ehealth.businessconnector.ehboxV2.builders.SendMessageBuilder;
import be.ehealth.businessconnector.ehboxV2.testutil.EhboxTestUtil;
import be.ehealth.technicalconnector.service.etee.Crypto;
import be.ehealth.technicalconnector.service.etee.domain.EncryptionToken;
import be.ehealth.technicalconnector.service.keydepot.KeyDepotManagerFactory;
import be.ehealth.technicalconnector.service.keydepot.KeyDepotService;
import be.ehealth.technicalconnector.service.keydepot.impl.KeyDepotServiceImpl;
import be.ehealth.technicalconnector.service.sts.security.Credential;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.session.SessionItem;
import be.ehealth.technicalconnector.session.SessionManager;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.IdentifierType;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.Message;
import be.fgov.ehealth.ehbox.publication.protocol.v2.SendMessageRequest;


/**
 * Test class for {@link SendMessageBuilderImpl}
 * 
 * @author EHP
 * 
 */
public class SendMessageBuilderImplTest {

    private Credential encryptionCredential;

    private SendMessageBuilder builder;

    private KeyDepotService mockKeyDepotService;

    private static Set<EncryptionToken> etkSet;

    private Crypto mockCrypto;

    private X509Certificate cert;

    private SessionManager mockSessionManager;

    private SessionItem mockSession;

    /**
     * Reinitialise the KeyDepotService with a new one and EncryptionCredentials with null
     * 
     * @throws Exception
     */
    @AfterClass
    public static void tearDown() throws Exception {
        KeyDepotManagerFactory.getKeyDepotManager().setKeyDepotService(new KeyDepotServiceImpl());
        Session.setSessionManager(null);
    }


    /**
     * Creates required mocks
     * 
     * @throws java.lang.Exception
     */
    @Before
    public void setUp() throws Exception {
        encryptionCredential = EasyMock.createMock(Credential.class);
        mockCrypto = EasyMock.createMock(Crypto.class);
        mockKeyDepotService = EasyMock.createMock(KeyDepotService.class);
        mockSession = EasyMock.createMock(SessionItem.class);
        mockSessionManager = EasyMock.createMock(SessionManager.class);

        Session.setSessionManager(mockSessionManager);

        EasyMock.expect(mockSessionManager.getSession()).andReturn(mockSession).anyTimes();
        EasyMock.expect(mockSessionManager.hasValidSession()).andReturn(true).anyTimes();
        EasyMock.expect(mockSession.getEncryptionCrypto()).andReturn(mockCrypto).anyTimes();
        EasyMock.expect(mockSession.getHolderOfKeyCrypto()).andReturn(mockCrypto).anyTimes();
        EasyMock.expect(mockSession.getEncryptionCredential()).andReturn(encryptionCredential).anyTimes();
        EasyMock.expect(mockSession.getHolderOfKeyCredential()).andReturn(encryptionCredential).anyTimes();

        KeyDepotManagerFactory.getKeyDepotManager().setKeyDepotService(mockKeyDepotService);
        builder = new SendMessageBuilderImpl(KeyDepotManagerFactory.getKeyDepotManager());
    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehboxV2.builders.impl.SendMessageBuilderImpl#buildMessage(be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage)}
     * .
     * 
     * @throws Exception
     */
    @Test
    public void testBuildMessageDocumentMessage() throws Exception {

        final String methodIdentifier = ".testBuildMessageDocumentMessage";
        DocumentMessage<Message> message = EhboxTestUtil.createFilledDocumentMessage(methodIdentifier);


        Addressee destination = message.getDestinations().get(0);

        // Setup Easymock

        if (message.isEncrypted()) {
            EasyMock.expect(mockKeyDepotService.getETKSet(destination.getIdentifierTypeHelper(), destination.getId(), destination.getApplicationId())).andReturn(prepareEtkSet());
            EasyMock.expect(mockCrypto.seal(EasyMock.eq(prepareEtkSet()), (byte[]) EasyMock.anyObject())).andReturn("freeText".getBytes()).anyTimes();
            cert = EasyMock.createMock(X509Certificate.class);
            EasyMock.expect(encryptionCredential.getCertificate()).andReturn(cert).anyTimes();
            String subject = "CN=SSIN=82051234978, OU=eHealth-platform Belgium, OU=HANNES DE CLERCQ, OU=SSIN=82051234978,O=Federal Government, C=BE";
            EasyMock.expect(cert.getSubjectX500Principal()).andReturn(new X500Principal(subject));
            EasyMock.expect(mockKeyDepotService.getETKSet(EasyMock.eq(IdentifierType.SSIN), EasyMock.eq("82051234978"), EasyMock.eq(""))).andReturn(prepareEtkSet());
        }
        EasyMock.replay(mockKeyDepotService, mockCrypto, encryptionCredential, cert);
        EasyMock.replay(mockSession, mockSessionManager);

        // build message
        SendMessageRequest result = builder.buildMessage(message);
        Assert.assertNotNull(result);

        Assert.assertNotNull(EhboxTestUtil.FREE_TEXT, result.getContentContext().getContent().getFreeInformations().getEncryptableFreeText());
        Assert.assertNotNull(EhboxTestUtil.PATIENT_INSS, result.getContentContext().getContent().getEncryptableINSSPatient());

        // check mandatee
        Assert.assertEquals(EhboxTestUtil.MANDATEE_QUALITY.getIdentifierType().formatIdentifierValue(Long.parseLong(EhboxTestUtil.MANDATEE_ID)), result.getMandator().getId());
        Assert.assertEquals(EhboxTestUtil.MANDATEE_QUALITY.getQuality(), result.getMandator().getQuality());

        Assert.assertTrue(result.getContentContext().getContentSpecification().isIsEncrypted());
        Assert.assertTrue(result.getContentContext().getContentSpecification().isIsImportant());

        // check destination
        Assert.assertEquals(1, result.getDestinationContexts().size());
        Assert.assertEquals(EhboxTestUtil.QUALITY.getIdentifierType().formatIdentifierValue(Long.parseLong(EhboxTestUtil.ID)), result.getDestinationContexts().get(0).getId());
        Assert.assertEquals(EhboxTestUtil.QUALITY.getQuality(), result.getDestinationContexts().get(0).getQuality());

        // check document
        Assert.assertEquals(EhboxTestUtil.FILENAME, result.getContentContext().getContent().getDocument().getDownloadFileName());
        Assert.assertEquals(EhboxTestUtil.MIME_TYPE, result.getContentContext().getContent().getDocument().getMimeType());
        Assert.assertEquals(EhboxTestUtil.TITLE + " " + methodIdentifier, result.getContentContext().getContent().getDocument().getTitle());

        // check annexes
        Assert.assertEquals(EhboxTestUtil.ANNEX_FILENAME, result.getContentContext().getContent().getAnnices().get(0).getDownloadFileName());
        Assert.assertEquals(EhboxTestUtil.ANNEX_MIME_TYPE, result.getContentContext().getContent().getAnnices().get(0).getMimeType());

        EasyMock.verify(mockKeyDepotService, mockCrypto);

    }


    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehboxV2.builders.impl.SendMessageBuilderImpl#buildMessage(be.ehealth.businessconnector.ehbox.api.domain.NewsMessage)}
     * .
     * 
     * @throws Exception
     */
    @Test
    public void testBuildMessageNewsMessage() throws Exception {


        NewsMessage<Message> message = new NewsMessage<Message>();
        Document news = new Document();

        // destination
        Addressee destination = new Addressee(EhboxTestUtil.QUALITY.getIdentifierType());

        destination.setApplicationId(EhboxTestUtil.APPLICATION_ID);
        destination.setFirstName(EhboxTestUtil.FIRST_NAME);
        destination.setId(EhboxTestUtil.ID);
        destination.setLastName(EhboxTestUtil.LAST_NAME);
        destination.setOrganizationName(EhboxTestUtil.ORGANIZATION_NAME);
        destination.setQuality(EhboxTestUtil.QUALITY);


        message.getDestinations().add(destination);

        news.setTitle(EhboxTestUtil.FILENAME);
        message.setDocument(news);
        news.setContent("blabla".getBytes());


        if (message.isEncrypted()) {
            EasyMock.expect(mockKeyDepotService.getETKSet(destination.getIdentifierTypeHelper(), destination.getId(), destination.getApplicationId())).andReturn(prepareEtkSet());
            EasyMock.expect(mockCrypto.seal(EasyMock.eq(prepareEtkSet()), EasyMock.aryEq(toByteArray(message.getDocument().getContent())))).andReturn("freeText".getBytes());
        }
        EasyMock.replay(mockKeyDepotService, mockCrypto);

        SendMessageRequest result = builder.buildMessage(message);
        Assert.assertNotNull(result);
        Assert.assertEquals(EhboxTestUtil.FILENAME, result.getContentContext().getContent().getNews().getTitle());

        EasyMock.verify(mockKeyDepotService, mockCrypto);
    }

    /**
     * Returns EtkSet and initialise it with kgss.etk file if not already done
     * 
     * @return
     * @throws Exception
     */
    private Set<EncryptionToken> prepareEtkSet() throws Exception {
        if (etkSet == null) {
            InputStream is = SendMessageBuilderImplTest.class.getResourceAsStream("/etee/kgss.etk");

            byte[] etkByteArray = ConnectorIOUtils.getBytes(is);

            etkSet = new HashSet<EncryptionToken>();
            etkSet.add(new EncryptionToken(etkByteArray));
        }
        return etkSet;
    }

    /**
     * Tranforms a content object into a byte[] using OutputStreams
     * 
     * @param content
     * @return
     * @throws Exception
     */
    private byte[] toByteArray(Object content) throws Exception {
        ByteArrayOutputStream bStream = new ByteArrayOutputStream();
        ObjectOutputStream oStream = new ObjectOutputStream(bStream);
        oStream.writeObject(content);
        byte[] byteVal = bStream.toByteArray();
        return byteVal;
    }
}
