/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.ehboxV2.builders.impl;

import java.io.IOException;

import javax.activation.DataHandler;
import javax.activation.DataSource;

import org.bouncycastle.util.encoders.Base64;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import be.ehealth.businessconnector.ehbox.api.domain.Addressee;
import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.domain.exception.EhboxBusinessConnectorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ByteArrayDatasource;
import be.fgov.ehealth.ehbox.consultation.protocol.v2.GetFullMessageResponse;
import be.fgov.ehealth.ehbox.core.v2.ContentContext;
import be.fgov.ehealth.ehbox.core.v2.ContentSpecificationType;
import be.fgov.ehealth.ehbox.core.v2.ContentType;
import be.fgov.ehealth.ehbox.core.v2.DestinationContext;
import be.fgov.ehealth.ehbox.core.v2.DocumentType;
import be.fgov.ehealth.ehbox.core.v2.EhboxIdentifierType;
import be.fgov.ehealth.ehbox.core.v2.FreeInformationsType;
import be.fgov.ehealth.ehbox.core.v2.MessageType;
import be.fgov.ehealth.ehbox.core.v2.SenderType;
import be.fgov.ehealth.ehbox.core.v2.User;


/**
 * Test class for {@link ConsultationMessageBuilderImpl}
 * 
 * @author EHP
 * 
 */
public class ConsultationMessageBuilderImplTest {


    private static final String ID = "1";


    private static final String INSS_PATIENT = "INSSPatient";


    private static final String FREETEXT = "FREETEXT";


    private static final String MIME_TYPE = "mime/type";


    private static final String TITLE = "TITLE";


    private static final String FILENAME = "FILENAME";


    private static final String QUALITY = "quality";


    private static final String SUBTYPE = "RETIREMENT";


    private static final String TYPE = "NIHII";


    private static final String LASTNAME = "lastname";

    private static final String FIRSTNAME = "firstname";

    private ConsultationMessageBuilderImpl builder;

    /**
     * @throws java.lang.Exception
     */
    @Before
    public void setUp() throws Exception {
        builder = new ConsultationMessageBuilderImpl();

    }

    /**
     * Reinitialise the KeyDepotService
     * 
     * @throws Exception
     */
    @AfterClass
    public static void tearDown() throws Exception {
    }


    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehboxV2.builders.impl.ConsultationMessageBuilderImpl#buildFullMessage(be.fgov.ehealth.ehbox.consultation.protocol.v2.GetFullMessageResponse)}
     * .
     * 
     * @throws IOException
     * @throws TechnicalConnectorException
     * @throws EhboxBusinessConnectorException
     */
    @Test
    public void testBuildFullMessage() throws IOException, EhboxBusinessConnectorException, TechnicalConnectorException {

        GetFullMessageResponse response = new GetFullMessageResponse();

        MessageType message = new MessageType();
        DestinationContext destination = new DestinationContext();
        destination.setId(ID);
        destination.setType(TYPE);
        destination.setSubType(SUBTYPE);
        destination.setQuality(QUALITY);
        message.getDestinationContexts().add(destination);

        ContentContext contentContext = new ContentContext();


        // build content
        contentContext.setContent(new ContentType());

        contentContext.getContent().setDocument(new DocumentType());
        contentContext.getContent().getDocument().setDownloadFileName(FILENAME);
        contentContext.getContent().getDocument().setTitle(TITLE);
        contentContext.getContent().getDocument().setMimeType(MIME_TYPE);
        contentContext.getContent().setFreeInformations(new FreeInformationsType());
        contentContext.getContent().getFreeInformations().setEncryptableFreeText(Base64.encode(FREETEXT.getBytes()));


        byte[] b64 = Base64.encode(INSS_PATIENT.getBytes());


        DataSource dataSource = new ByteArrayDatasource(b64);
        contentContext.getContent().setEncryptableINSSPatient(new DataHandler(dataSource));


        // build contentSpecification
        contentContext.setContentSpecification(new ContentSpecificationType());
        contentContext.getContentSpecification().setIsEncrypted(false);
        contentContext.getContentSpecification().setIsImportant(true);

        // build mandatee

        response.setMandatee(new SenderType());
        response.getMandatee().setFirstName(FIRSTNAME);
        response.getMandatee().setId(ID);
        response.getMandatee().setName(LASTNAME);
        response.getMandatee().setQuality(QUALITY);
        response.getMandatee().setSubType(SUBTYPE);
        response.getMandatee().setType(TYPE);

        // build sender
        response.setSender(new SenderType());
        response.getSender().setFirstName(FIRSTNAME);
        response.getSender().setId(ID);
        response.getSender().setName(LASTNAME);
        response.getSender().setQuality(QUALITY);
        response.getSender().setSubType(SUBTYPE);
        response.getSender().setType(TYPE);

        message.setContentContext(contentContext);
        response.setMessage(message);

        DocumentMessage<?> result = (DocumentMessage<?>) builder.buildFullMessage(response);

        Assert.assertNotNull(result);
        Assert.assertFalse(result.isEncrypted());
        Assert.assertTrue(result.isImportant());
        Assert.assertEquals(TITLE, result.getDocumentTitle());
        Assert.assertEquals(FREETEXT, result.getFreeText());


        Assert.assertNotNull(result.getDestinations().get(0));
        Assert.assertNotNull(result.getSender());
        Assert.assertNotNull(result.getPatientInss());

    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehboxV2.builders.impl.ConsultationMessageBuilderImpl#buildAddressee(be.fgov.ehealth.ehbox.core.v2.SenderType)}
     * .
     */
    @Test
    public void testBuildAddresseeSenderType() {
        SenderType senderType = new SenderType();

        senderType.setFirstName(FIRSTNAME);
        senderType.setName(LASTNAME);
        senderType.setQuality(QUALITY);
        senderType.setSubType(SUBTYPE);
        senderType.setType(TYPE);

        Addressee result = builder.buildAddressee(senderType);

        Assert.assertNotNull(result);
        Assert.assertEquals(FIRSTNAME, result.getFirstName());
        Assert.assertEquals(LASTNAME, result.getLastName());
        Assert.assertEquals(QUALITY, result.getQuality());
        Assert.assertEquals(SUBTYPE, result.getSubType());
        Assert.assertEquals(TYPE, result.getType());


    }

    @Test(expected = IllegalArgumentException.class)
    public void testWithEmptySenderType() throws Exception {
        SenderType senderType = new SenderType();
        builder.buildAddressee(senderType);

    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehboxV2.builders.impl.ConsultationMessageBuilderImpl#buildAddressee(be.fgov.ehealth.ehbox.core.v2.EhboxIdentifierType)}
     * .
     */
    @Test
    public void testBuildAddresseeEhboxIdentifierType() {
        EhboxIdentifierType identifierType = new EhboxIdentifierType();
        User user = new User();
        user.setFirstName(FIRSTNAME);
        user.setLastName(LASTNAME);
        identifierType.setUser(user);

        identifierType.setQuality(QUALITY);
        identifierType.setSubType(SUBTYPE);
        identifierType.setType(TYPE);

        Addressee result = builder.buildAddressee(identifierType);

        Assert.assertNotNull(result);
        Assert.assertEquals(FIRSTNAME, result.getFirstName());
        Assert.assertEquals(LASTNAME, result.getLastName());
        Assert.assertEquals(QUALITY, result.getQuality());
        Assert.assertEquals(SUBTYPE, result.getSubType());
        Assert.assertEquals(TYPE, result.getType());


    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuildAddresseeWithEmptyIdentifierType() throws Exception {
        EhboxIdentifierType identifierType = new EhboxIdentifierType();
        // the needed arguments are not provided in the identifierType
        builder.buildAddressee(identifierType);

    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehboxV2.builders.impl.ConsultationMessageBuilderImpl#buildAddressee(be.fgov.ehealth.ehbox.core.v2.SenderType)}
     * .
     */
    @Test
    public void testBuildAddresseeSenderTypeWithOrganisation() {
        SenderType senderType = new SenderType();
        senderType.setName(LASTNAME);
        senderType.setQuality(QUALITY);
        senderType.setSubType(SUBTYPE);
        senderType.setType(TYPE);
        senderType.setPersonInOrganisation(ID);

        Addressee result = builder.buildAddressee(senderType);

        Assert.assertNotNull(result);
        Assert.assertNull(result.getFirstName());
        Assert.assertNull(result.getLastName());
        Assert.assertEquals(LASTNAME, result.getOrganizationName());

        Assert.assertEquals(ID, result.getPersonInOrganisation());
        Assert.assertEquals(QUALITY, result.getQuality());
        Assert.assertEquals(SUBTYPE, result.getSubType());
        Assert.assertEquals(TYPE, result.getType());


    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuildWithEmptySenderType() throws Exception {

        SenderType senderType = new SenderType();
        builder.buildAddressee(senderType);

    }

}
