/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.civics.session;

import java.math.BigInteger;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.SoaErrorException;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.ehealth.technicalconnector.utils.DateUtils;
import be.fgov.ehealth.samcivics.type.v1.AtmAndChildrenType;
import be.fgov.ehealth.samcivics.type.v1.ExcludedParagraphType;
import be.fgov.ehealth.samcivics.type.v1.FindCNKRequest;
import be.fgov.ehealth.samcivics.type.v1.FindCNKResponse;
import be.fgov.ehealth.samcivics.type.v1.FindParagraphTextResponse;
import be.fgov.ehealth.samcivics.type.v1.FindReimbursementConditionsRequest;
import be.fgov.ehealth.samcivics.type.v1.FindReimbursementConditionsResponse;
import be.fgov.ehealth.samcivics.type.v1.GetAddedDocumentsResponse;
import be.fgov.ehealth.samcivics.type.v1.GetParagraphExclusionsResponse;
import be.fgov.ehealth.samcivics.type.v1.GetParagraphIncludedSpecialitiesResponse;
import be.fgov.ehealth.samcivics.type.v1.GetProfessionalAuthorisationsRequest;
import be.fgov.ehealth.samcivics.type.v1.GetProfessionalAuthorisationsResponse;
import be.fgov.ehealth.samcivics.type.v1.LanguageType;
import be.fgov.ehealth.samcivics.type.v1.ParagraphAndVersesRequestType;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;

/**
 * CIVICS Service Integration Tests This test shows the use of the CIVICS Service through the use of the Business Connector.
 * 
 * All Tests require the following pre-requisites to be met: - An active and valid eHealth-platform session (initiated via the Session
 * Management component), in these tests this is done in the AbstractServiceTest class
 * 
 * @author EHP
 */
public class CivicsSessionServiceIntegrationTest {

    private static final Logger LOG = LoggerFactory.getLogger(CivicsSessionServiceIntegrationTest.class);

    @BeforeClass
    public static void init() throws Exception {
        SessionInitializer.init("/be.ehealth.businessconnector.civics.test.properties");
    }

    @AfterClass
    public static void tearDown() throws Exception {
        Session.getInstance().unloadSession();
    }

    /**
     * Version number of paragraphs to retrieve
     */
    private static final int VERSION = 1;

    /**
     * Set this parameter to true if you want additional parameters in the request (additional parameter is relative to the version of the
     * paragraph) : it can be a date (we will retrieve versions starting from this date) XOR the version of the paragraph to retrieve
     */
    private static boolean optional;

    /**
     * Set this boolean to true if you want to select paragraphs with a version starting from this date If false, and optional set to true,
     * the version VERSION will be selected
     */
    private static boolean date;


    /**
     * Test the FindParagraphText operation from the CIVICS Service through the use of the Business Connector. This test retrieves the
     * textual part and related data with regards to reimbursement stipulations.
     * 
     * @throws ConnectorException
     */
    @Test
    public final void testFindParagraphText() throws ConnectorException {
        /*
         * Set the values to use in this test
         */
        // the chapter to search
        String chapterName = "IV";
        // the paragraph to search
        String paragraphName = "330100";
        // Set the language of the user
        LanguageType lang = LanguageType.NL;


        /*
         * Create the request
         */
        ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
        // set the chapter name
        request.setChapterName(chapterName);
        // set the paragraph name
        request.setParagraphName(paragraphName);
        // set the language
        request.setLanguage(lang);
        if (optional) {
            if (date) { // optionally: the start date, retrieve version starting from this date
                request.setStartDate(DateUtils.parseDate("2011-04-11"));
            } else {
                // optionally: the version of the paragraph to retrieve
                BigInteger optionalVersion = BigInteger.valueOf(VERSION);
                request.setVersion(optionalVersion);
            }
        }


        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */
        CivicsService service = CivicsSessionServiceFactory.getCivicsService();
        FindParagraphTextResponse response = service.findParagraphText(request);

        /*
         * Verify the response
         */
        // Check if response has been received
        Assert.assertNotNull(response);
        // Check the chapter name
        Assert.assertNotNull(response.getParagraph().getChapterName());
    }

    /**
     * Test the ParagraphIncludedSpecialties operation from the CIVICS Service through the use of the Business Connector. This test
     * retrieves all specialities which are reimbursed within the requested paragraph.
     * 
     * @throws ConnectorException
     */
    @Test
    public final void getParagraphIncludedSpecialties() throws ConnectorException {
        /*
         * Set the values to use in this test
         */
        // the chapter to search
        String chapterName = "IV";
        // the paragraph to search
        String paragraphName = "330100";
        // Set the language of the user
        LanguageType lang = LanguageType.NL;

        /*
         * Create the request
         */
        ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
        // set the chapter name
        request.setChapterName(chapterName);
        // set the paragraph name
        request.setParagraphName(paragraphName);
        // set the language
        request.setLanguage(lang);
        if (optional) {
            if (date) { // optionally: the start date, retrieve version starting from this date
                request.setStartDate(DateUtils.parseDate("2011-04-11"));
            } else {
                // optionally: the version of the paragraph to retrieve
                BigInteger optionalVersion = BigInteger.valueOf(1);
                request.setVersion(optionalVersion);
            }
        }

        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */
        CivicsService service = CivicsSessionServiceFactory.getCivicsService();
        GetParagraphIncludedSpecialitiesResponse response = service.getParagraphIncludedSpecialties(request);

        /*
         * Verify the response
         */
        // Check if response has been received
        Assert.assertNotNull(response);
    }

    /**
     * Test the ParagraphExclusions operation from the CIVICS Service through the use of the Business Connector. This test retrieves the
     * Specialities that may not be reimbursed in combination with other administered specialities for which there already is a refund
     * agreement.
     * 
     * @throws ConnectorException
     */
    @Test
    public final void testGetParagraphExclusions() throws ConnectorException {
        /*
         * Set the values to use in this test
         */
        // the chapter to search
        String chapterName = "IV";
        // the paragraph to search
        String paragraphName = "5580000";
        // Set the language of the user
        LanguageType lang = LanguageType.FR;
        // optionally: the start date, retrieve version starting from this date

        /*
         * Create the request
         */
        ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
        // set the chapter name
        request.setChapterName(chapterName);
        // set the paragraph name
        request.setParagraphName(paragraphName);
        // set the language
        request.setLanguage(lang);
        if (optional) {
            if (date) { // optionally: the start date, retrieve version starting from this date
                request.setStartDate(DateUtils.parseDate("2011-04-11"));
            } else {
                // optionally: the version of the paragraph to retrieve
                BigInteger optionalVersion = BigInteger.valueOf(1);
                request.setVersion(optionalVersion);
            }
        }

        try {
            /*
             * Invoke the business connector framework's eHealthBox's getMessageList operation
             */
            CivicsService service = CivicsSessionServiceFactory.getCivicsService();
            GetParagraphExclusionsResponse response = service.getParagraphExclusions(request);

            /*
             * Verify the response
             */
            // Check if response has been received
            Assert.assertNotNull(response);
            for (ExcludedParagraphType paraf : response.getExclusions().getExcludedParagraphList().getExcludedParagraphs()) {
                for (AtmAndChildrenType child : paraf.getAtmList().getAtms()) {
                    LOG.debug("AtmAndChildrenType: " + ConnectorXmlUtils.toString(child));
                }
            }
        } catch (SoaErrorException e) {
            if (!e.getErrorType().getMessages().get(0).getValue().equalsIgnoreCase("No data found")) {
                throw e;
            };
        }
    }

    /**
     * Test the ProfessionalAuthorizations operation from the CIVICS Service through the use of the Business Connector. This test retrieves
     * the qualification list, stipulating requirements concerning the professional qualification.
     * 
     * @throws ConnectorException
     */
    @Test
    public final void getProfessionalAuthorizations() throws ConnectorException {
        /*
         * Set the values to use in this test
         */
        // the paragraph to search
        String qualificationlist = "4";
        // Set the language of the user
        LanguageType lang = LanguageType.NL;
        // the input date
        // Calendar inputDate = Calendar.getInstance();
        // inputDate.set(2010, 5, 11);

        /*
         * Create the request
         */
        GetProfessionalAuthorisationsRequest request = new GetProfessionalAuthorisationsRequest();
        // set the input date
        // request.setInputDate(inputDate);
        // set the language
        request.setLanguage(lang);
        // set the qualification list
        request.setQualificationList(qualificationlist);

        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */
        CivicsService service = CivicsSessionServiceFactory.getCivicsService();
        GetProfessionalAuthorisationsResponse response = service.getProfessionalAuthorizations(request);

        /*
         * Verify the response
         */
        // Check if response has been received
        Assert.assertNotNull(response);
    }

    /**
     * Test the GetAddedDocuments operation from the CIVICS Service through the use of the Business Connector. This test retrieves the
     * description and identification of the structured forms annexed to a paragraph.
     * 
     * @throws ConnectorException
     */
    @Test
    public final void testGetAddedDocuments() throws ConnectorException {
        /*
         * Set the values to use in this test
         */
        // the chapter to search
        String chapterName = "IV";
        // the paragraph to search
        String paragraphName = "1070000";
        // Set the language of the user
        LanguageType lang = LanguageType.NL;


        /*
         * Create the request
         */
        ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
        // set the chapter name
        request.setChapterName(chapterName);
        // set the paragraph name
        request.setParagraphName(paragraphName);
        // set the language
        request.setLanguage(lang);
        if (optional) {
            if (date) { // optionally: the start date, retrieve version starting from this date
                request.setStartDate(DateUtils.parseDate("2011-04-11"));
            } else {
                // optionally: the version of the paragraph to retrieve
                BigInteger optionalVersion = BigInteger.valueOf(1);
                request.setVersion(optionalVersion);
            }
        }

        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */
        CivicsService service = CivicsSessionServiceFactory.getCivicsService();
        try {
            GetAddedDocumentsResponse response = service.getAddedDocuments(request);

            /*
             * Verify the response
             */
            // Check if response has been received
            Assert.assertNotNull(response);


        } catch (SoaErrorException e) {
            if (!e.getErrorType().getMessages().get(0).getValue().equalsIgnoreCase("No data found")) {
                throw e;
            };
        }


    }

    /**
     * Test the FindReimbursementConditions operation from the CIVICS Service through the use of the Business Connector. This test retrieves
     * all information on prices and reimbursement amounts.
     * 
     * @throws ConnectorException
     */
    @Test
    public final void testFindReimbursementConditions() throws ConnectorException {
        /*
         * Set the values to use in this test
         */
        // the chapter to search
        String chapterName = "IV";
        // the paragraph to search
        String paragraphName = "330100";
        // Set the language of the user
        LanguageType lang = LanguageType.NL;

        /*
         * Create the request
         */
        FindReimbursementConditionsRequest request = new FindReimbursementConditionsRequest();
        // set the chapter name
        request.setChapterName(chapterName);
        // set the paragraph name
        request.setParagraphName(paragraphName);
        // set the language
        request.setLanguage(lang);
        if (optional) {
            if (date) { // optionally: the start date, retrieve version starting from this date
                request.setStartDate(DateUtils.parseDate("2011-04-11"));
            } else {
                // optionally: the version of the paragraph to retrieve
                BigInteger optionalVersion = BigInteger.valueOf(1);
                request.setVersion(optionalVersion);
            }
        }

        /*
         * Invoke the business connector framework's eHealthBox's getMessageList operation
         */
        CivicsService service = CivicsSessionServiceFactory.getCivicsService();
        FindReimbursementConditionsResponse response = service.findReimbursementConditions(request);

        /*
         * Verify the response
         */
        // Check if response has been received
        Assert.assertNotNull(response);
    }

    @Test
    public final void testFindCNK() throws ConnectorException {
        String productName = "PLAVIX";
        FindCNKRequest request = new FindCNKRequest();
        request.setProductName(productName);
        request.setLanguage(LanguageType.FR);

        CivicsService service = CivicsSessionServiceFactory.getCivicsService();
        FindCNKResponse response = service.findCNK(request);

        Assert.assertNotNull(response);


    }
}