/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.crl;

import be.fgov.ehealth.etee.crypto.cert.CertificateStatus;
import be.fgov.ehealth.etee.crypto.crl.CRLChecker;
import be.fgov.ehealth.etee.crypto.crl.CRLCheckerResult;
import be.fgov.ehealth.etee.crypto.crl.CRLData;
import be.fgov.ehealth.etee.crypto.status.CryptoResult;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.status.NotificationFatal;
import be.fgov.ehealth.etee.crypto.status.NotificationWarning;
import be.fgov.ehealth.etee.crypto.utils.KeyManager;
import be.fgov.ehealth.etee.crypto.utils.Preconditions;
import be.fgov.ehealth.etee.crypto.utils.Streams;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CRLException;
import java.security.cert.CertStore;
import java.security.cert.CertStoreException;
import java.security.cert.CertStoreParameters;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509CRLSelector;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.jce.MultiCertStoreParameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class CRLCheckerBasic
implements CRLChecker {
    private static final Logger LOGGER = LoggerFactory.getLogger(CRLCheckerBasic.class);
    private final CertStore certStore;

    public CRLCheckerBasic(CertStore inCertStore) {
        this.certStore = inCertStore;
    }

    @Override
    public CryptoResult<CRLData> validate(X509Certificate cert) {
        return this.validate(cert, null, null);
    }

    @Override
    public CryptoResult<CRLData> validate(X509Certificate cert, Date validationDate) {
        return this.validate(cert, null, validationDate);
    }

    @Override
    public CryptoResult<CRLData> validate(X509Certificate cert, CertStore inCertStore) {
        return this.validate(cert, inCertStore, null);
    }

    @Override
    public CryptoResult<CRLData> validate(X509Certificate cert, CertStore inCertStore, Date validationDate) {
        Preconditions.checkNotNull(cert, "Certificate cannot be null.");
        CRLCheckerResult result = new CRLCheckerResult();
        try {
            Collection<X509CRL> storeCrls;
            LOGGER.info("Check CRL of certificate " + cert.getSubjectX500Principal());
            CertStore mergedStore = this.prepareStore(inCertStore);
            X509Certificate issuer = this.extractIssuer(cert, mergedStore);
            if (issuer == null) {
                LOGGER.warn("CertStore does not contain issuer " + cert.getIssuerX500Principal() + ". Signature validation of its CRL will not be possible.");
                result.getWarnings().add(NotificationWarning.CRL_TRUST_NOT_VERIFIED);
            }
            if (!(storeCrls = this.extractCRLs(cert, mergedStore)).isEmpty()) {
                this.verifyFromCache(cert, result, issuer, storeCrls, validationDate);
            } else {
                this.verifyFromDistributionPoint(cert, result, issuer, validationDate);
            }
        }
        catch (NoSuchProviderException e) {
            LOGGER.error("Default SecurityConfiguration not set.", (Throwable)e);
            result = new CRLCheckerResult(NotificationFatal.CRL_CHECK_FAILED);
        }
        catch (IOException e) {
            LOGGER.error("CRL could not be loaded.", (Throwable)e);
            result = new CRLCheckerResult(NotificationFatal.CRL_CHECK_FAILED);
        }
        catch (Exception e) {
            LOGGER.error("CRL Check failed.", (Throwable)e);
            result = new CRLCheckerResult(NotificationFatal.CRL_CHECK_FAILED);
        }
        return result;
    }

    private void verifyFromCache(X509Certificate cert, CRLCheckerResult result, X509Certificate issuer, Collection<X509CRL> cachedCrls, Date validationDate) throws CertificateException, NoSuchAlgorithmException, NoSuchProviderException, CRLException {
        for (X509CRL crl : cachedCrls) {
            LOGGER.debug("Selected non expired CRL of issuer from certStore (expires at " + crl.getNextUpdate() + ")");
            result.addCrl(crl);
            if (this.isRevokedAtGivenDate(cert, issuer, crl, result, validationDate)) {
                result.setCertStatus(CertificateStatus.REVOKED);
                return;
            }
            result.setCertStatus(CertificateStatus.VALID);
        }
    }

    private void verifyFromDistributionPoint(X509Certificate cert, CRLCheckerResult result, X509Certificate issuer, Date validationDate) throws IOException, CertificateException, CRLException, NoSuchProviderException, NoSuchAlgorithmException {
        List<String> crlDistPoints = this.getCrlDistributionPoints(cert);
        if (crlDistPoints.isEmpty()) {
            LOGGER.debug("Certificate has no CRL Distribution Points: revocation status considered as VALID.");
            result.setCertStatus(CertificateStatus.VALID);
            return;
        }
        result.setCertStatus(CertificateStatus.UNSPECIFIED);
        for (String crlDP : crlDistPoints) {
            LOGGER.debug("Downloading CRL from distribution point " + crlDP);
            X509CRL crl = this.downloadCRL(crlDP);
            if (crl == null) continue;
            result.addCrl(crl);
            if (this.isRevokedAtGivenDate(cert, issuer, crl, result, validationDate)) {
                result.setCertStatus(CertificateStatus.REVOKED);
                return;
            }
            result.setCertStatus(CertificateStatus.VALID);
        }
    }

    private boolean isRevokedAtGivenDate(X509Certificate cert, X509Certificate issuer, X509CRL crl, CRLCheckerResult result, Date validationDate) throws CertificateException, NoSuchAlgorithmException, NoSuchProviderException, CRLException {
        X509CRLEntry crlEntry;
        if (issuer != null) {
            if (!KeyManager.verifyIssuer(cert, issuer)) {
                LOGGER.error("Given issuer does not match issuer of given certificate.");
                result.getErrors().add(NotificationError.CRL_TRUST_FAILED);
            }
            if (!this.verifyIssuer(crl, issuer)) {
                LOGGER.error("Public key of given issuer does not match signature on CRL.");
                result.getErrors().add(NotificationError.CRL_TRUST_FAILED);
            }
        }
        if ((crlEntry = crl.getRevokedCertificate(cert)) != null) {
            LOGGER.debug("Certificate revoked: " + crlEntry + " on " + crlEntry.getRevocationDate());
            return validationDate == null || !crlEntry.getRevocationDate().after(validationDate);
        }
        return false;
    }

    private X509CRL downloadCRL(String crlURL) throws CertificateException, CRLException, NoSuchProviderException, IOException {
        if (crlURL.startsWith("http://") || crlURL.startsWith("https://") || crlURL.startsWith("ftp://")) {
            return this.downloadCRLFromWeb(crlURL);
        }
        LOGGER.warn("Scheme of Certificate Distribution Point not supported: " + crlURL + ". Will be ignored.");
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<String> getCrlDistributionPoints(X509Certificate cert) throws IOException {
        byte[] crldpExt;
        ASN1InputStream asn1Stream;
        ArrayList<String> crlUrls;
        block5: {
            crlUrls = new ArrayList<String>();
            if (cert == null) {
                return crlUrls;
            }
            asn1Stream = null;
            crldpExt = cert.getExtensionValue(Extension.cRLDistributionPoints.getId());
            if (crldpExt != null) break block5;
            ArrayList<String> arrayList = new ArrayList<String>();
            Streams.closeQuietly(asn1Stream);
            return arrayList;
        }
        try {
            asn1Stream = new ASN1InputStream((InputStream)new ByteArrayInputStream(crldpExt));
            ASN1Primitive derObjCrlDP = asn1Stream.readObject();
            DEROctetString dosCrlDP = (DEROctetString)derObjCrlDP;
            byte[] crldpExtOctets = dosCrlDP.getOctets();
            asn1Stream.close();
            asn1Stream = new ASN1InputStream((InputStream)new ByteArrayInputStream(crldpExtOctets));
            ASN1Primitive derObj2 = asn1Stream.readObject();
            CRLDistPoint distPoint = CRLDistPoint.getInstance((Object)derObj2);
            crlUrls = new ArrayList();
            for (DistributionPoint dp : distPoint.getDistributionPoints()) {
                crlUrls.addAll(this.extractCRLs(dp));
            }
        }
        catch (Throwable throwable) {
            Streams.closeQuietly(asn1Stream);
            throw throwable;
        }
        Streams.closeQuietly((Closeable)asn1Stream);
        return crlUrls;
    }

    private List<String> extractCRLs(DistributionPoint dp) {
        ArrayList<String> crlUrls = new ArrayList<String>();
        DistributionPointName dpn = dp.getDistributionPoint();
        if (dpn != null && dpn.getType() == 0) {
            for (GeneralName genName : GeneralNames.getInstance((Object)dpn.getName()).getNames()) {
                if (genName.getTagNo() != 6) continue;
                String url = DERIA5String.getInstance((Object)genName.getName()).getString();
                crlUrls.add(url);
            }
        }
        return crlUrls;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private X509CRL downloadCRLFromWeb(String crlURL) throws IOException, CertificateException, CRLException, NoSuchProviderException {
        URL url = new URL(crlURL);
        InputStream crlStream = url.openStream();
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509", "BC");
            X509CRL x509CRL = (X509CRL)cf.generateCRL(crlStream);
            return x509CRL;
        }
        finally {
            Streams.closeQuietly(crlStream);
        }
    }

    private boolean verifyIssuer(X509CRL crl, X509Certificate issuer) throws CRLException, NoSuchAlgorithmException, NoSuchProviderException {
        try {
            crl.verify(issuer.getPublicKey());
            return true;
        }
        catch (SignatureException e) {
            LOGGER.trace("Invalid signature --> different issuer", (Throwable)e);
            return false;
        }
        catch (InvalidKeyException e) {
            LOGGER.trace("Invalid key --> different issuer", (Throwable)e);
            return false;
        }
    }

    private X509Certificate extractIssuer(X509Certificate cert, CertStore inCertStore) throws CertStoreException, CertificateException {
        X509CertSelector certIssuerSelector = new X509CertSelector();
        certIssuerSelector.setSubject(cert.getIssuerX500Principal());
        Collection<? extends Certificate> issuers = inCertStore.getCertificates(certIssuerSelector);
        if (issuers.isEmpty()) {
            if (KeyManager.isSelfSigned(cert)) {
                return cert;
            }
            return null;
        }
        return (X509Certificate)issuers.iterator().next();
    }

    private Collection<X509CRL> extractCRLs(X509Certificate cert, CertStore inCertStore) throws CertStoreException {
        X509CRLSelector crlSelector = new X509CRLSelector();
        crlSelector.setIssuers(Collections.singletonList(cert.getIssuerX500Principal()));
        crlSelector.setDateAndTime(new Date());
        return inCertStore.getCRLs(crlSelector);
    }

    private CertStore prepareStore(CertStore inCertStore) throws InvalidAlgorithmParameterException, NoSuchAlgorithmException {
        ArrayList<CertStore> storeList = new ArrayList<CertStore>();
        if (this.certStore != null) {
            storeList.add(this.certStore);
        }
        if (inCertStore != null) {
            storeList.add(inCertStore);
        }
        return CertStore.getInstance("Multi", (CertStoreParameters)new MultiCertStoreParameters(storeList));
    }
}

