/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.dicsv2.session;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.technicalconnector.idgenerator.IdGeneratorFactory;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.fgov.ehealth.dics.protocol.v2.FindAmpRequest;
import be.fgov.ehealth.dics.protocol.v2.FindAmpResponse;
import be.fgov.ehealth.dics.protocol.v2.FindByActualProductType;
import be.fgov.ehealth.dics.protocol.v2.FindByPackageType;
import be.fgov.ehealth.dics.protocol.v2.FindByTherapeuticMoietyType;
import be.fgov.ehealth.dics.protocol.v2.FindCompanyRequest;
import be.fgov.ehealth.dics.protocol.v2.FindCompanyResponse;
import be.fgov.ehealth.dics.protocol.v2.FindLegalBasesType;
import be.fgov.ehealth.dics.protocol.v2.FindLegislationTextRequest;
import be.fgov.ehealth.dics.protocol.v2.FindLegislationTextResponse;
import be.fgov.ehealth.dics.protocol.v2.FindReferencesRequest;
import be.fgov.ehealth.dics.protocol.v2.FindReferencesResponse;
import be.fgov.ehealth.dics.protocol.v2.FindReimbursementRequest;
import be.fgov.ehealth.dics.protocol.v2.FindReimbursementResponse;
import be.fgov.ehealth.dics.protocol.v2.FindVmpRequest;
import be.fgov.ehealth.dics.protocol.v2.FindVmpResponse;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;

/**
 * Integration tests for tarification.
 * 
 * @author eHealth Platform
 * 
 */
public final class DicsIntegrationTest {

    /**
     * @param oa
     */
    public DicsIntegrationTest() {
        super();
    }


    private static final Logger LOG = LoggerFactory.getLogger(DicsIntegrationTest.class);

    @BeforeClass
    public static void setupsession() throws Exception {
        LOG.debug("DicsIntegrationTest.loadSession:");
        SessionInitializer.init("/be.ehealth.businessconnector.dicsv2.test.properties");
    }

    @AfterClass
    public static void teardownSession() throws Exception {
        Session.getInstance().unloadSession();
    }

    @Test
    public void findAmpTest() throws Exception {
        FindAmpRequest findAmpRequest = new FindAmpRequest();
        findAmpRequest.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        findAmpRequest.setIssueInstant(new DateTime());
        findAmpRequest.setLang("fr");

        FindByActualProductType product = new FindByActualProductType();
        product.setAmpCode("SAM276385-00");
        findAmpRequest.setFindByProduct(product);

        FindAmpResponse response = DicsSessionServiceFactory.getDicsSession().findAmp(findAmpRequest);

        logResponse(response, FindAmpResponse.class);
        Assert.assertNotNull(response.getAmps());
        Assert.assertNotEquals(0, response.getAmps().size());
    }

    @Test
    public void findCompanyTest() throws Exception {
        FindCompanyRequest findCompanyRequest = new FindCompanyRequest();
        findCompanyRequest.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        findCompanyRequest.setIssueInstant(new DateTime());
        findCompanyRequest.setLang("fr");

        findCompanyRequest.setCompanyActorNr("05006");

        FindCompanyResponse response = DicsSessionServiceFactory.getDicsSession().findCompany(findCompanyRequest);

        logResponse(response, FindCompanyResponse.class);
        Assert.assertNotNull(response.getCompanies());
        Assert.assertNotEquals(0, response.getCompanies().size());
    }

    @Test
    public void findLegislationTextTest() throws Exception {
        FindLegislationTextRequest findLegTextRequest = new FindLegislationTextRequest();
        findLegTextRequest.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        findLegTextRequest.setIssueInstant(new DateTime());
        findLegTextRequest.setLang("fr");

        findLegTextRequest.setFindLegalBases(new FindLegalBasesType());

        FindLegislationTextResponse response = DicsSessionServiceFactory.getDicsSession().findLegislationText(findLegTextRequest);

        logResponse(response, FindLegislationTextResponse.class);
        Assert.assertNotNull(response.getStatus());
        Assert.assertNotNull(response.getStatus().getStatusCode());
        Assert.assertNotNull(response.getStatus().getStatusCode().getValue());
        Assert.assertEquals("urn:be:fgov:ehealth:2.0:status:Success", response.getStatus().getStatusCode().getValue());
    }

    @Test
    public void findReferencesTest() throws Exception {
        FindReferencesRequest findRefRequest = new FindReferencesRequest();
        findRefRequest.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        findRefRequest.setIssueInstant(new DateTime());

        findRefRequest.setReferenceEntity("Wada");

        FindReferencesResponse response = DicsSessionServiceFactory.getDicsSession().findReferences(findRefRequest);

        logResponse(response, FindReferencesResponse.class);
        Assert.assertNotNull(response.getWadas());
        Assert.assertNotEquals(0, response.getWadas().size());
    }

    @Test
    public void findReimboursementTest() throws Exception {
        FindReimbursementRequest findReimboursementRequest = new FindReimbursementRequest();
        findReimboursementRequest.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        findReimboursementRequest.setIssueInstant(new DateTime());
        findReimboursementRequest.setLang("fr");

        FindByPackageType packageType = new FindByPackageType();
        packageType.setCtiExtendedCode("444053-03");
        findReimboursementRequest.setFindByPackage(packageType);

        FindReimbursementResponse response = DicsSessionServiceFactory.getDicsSession().findReimbursement(findReimboursementRequest);

        logResponse(response, FindReimbursementResponse.class);
        Assert.assertNotNull(response.getStatus());
        Assert.assertNotNull(response.getStatus().getStatusCode());
        Assert.assertNotNull(response.getStatus().getStatusCode().getValue());
        Assert.assertEquals("urn:be:fgov:ehealth:2.0:status:Success", response.getStatus().getStatusCode().getValue());
    }

    @Test
    public void findVmpTest() throws Exception {
        FindVmpRequest findVmpRequest = new FindVmpRequest();
        findVmpRequest.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        findVmpRequest.setIssueInstant(new DateTime());
        findVmpRequest.setLang("fr");


        FindByTherapeuticMoietyType therapeuticMoiety = new FindByTherapeuticMoietyType();
        therapeuticMoiety.setTherapeuticMoietyCode(new BigInteger("10"));
        findVmpRequest.setFindByTherapeuticMoiety(therapeuticMoiety);

        FindVmpResponse response = DicsSessionServiceFactory.getDicsSession().findVmp(findVmpRequest);

        logResponse(response, FindVmpResponse.class);
        Assert.assertNotNull(response.getStatus());
        Assert.assertNotNull(response.getStatus().getStatusCode());
        Assert.assertEquals("urn:be:fgov:ehealth:2.0:status:Success", response.getStatus().getStatusCode().getValue());
    }

    /**
     * @param response
     * @throws Exception
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    private void logResponse(Object response, Class classToUnmarshall) throws Exception {
        MarshallerHelper<Object, Object> responseMarshaller;
        responseMarshaller = new MarshallerHelper(classToUnmarshall, classToUnmarshall);
        String unmarshalledString = responseMarshaller.toString(response);
        LOG.debug("response : " + unmarshalledString);
    }


    /**
     * @param result
     * @param fileName
     * @throws FileNotFoundException
     * @throws IOException
     */
    @SuppressWarnings("unused")
    private void saveFileValue(byte[] fileValue, String fileName) throws FileNotFoundException, IOException {
        FileOutputStream fos = new FileOutputStream(fileName);
        fos.write(fileValue);
        fos.close();
    }

}
