package be.ehealth.businessconnector.medadmin.session;

import be.cin.encrypted.EncryptedKnownContent;
import be.ehealth.business.mycarenetdomaincommons.domain.InputReference;
import be.ehealth.businessconnector.genericasync.domain.GetRequest;
import be.ehealth.businessconnector.genericasync.domain.ProcessedGetResponse;
import be.ehealth.businessconnector.genericasync.domain.ProcessedMsgResponse;
import be.ehealth.businessconnector.medadmin.domain.M4ACnfXmlProcessedGetResponse;
import be.ehealth.businessconnector.medadmin.domain.M4AXmlProcessedGetResponse;
import be.ehealth.businessconnector.medadmin.domain.RejectXmlProcessedGetResponse;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.junit.Rule;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.concurrent.TimeUnit;

import static org.junit.Assert.*;

public class MedAdminGetAsyncIntegrationTest {

    private static final Logger LOG = LoggerFactory.getLogger(MedAdminGetAsyncIntegrationTest.class);

    private static final int SECONDS_WAITING = 120;

    @Rule
    public SessionRule rule = SessionRule.withActiveSession().baseOn("/be.ehealth.businessconnector.medadmin.test.properties").build();

    @Test
    public void getM4AXmlMessage() throws Exception {
        M4AXmlProcessedGetResponse response = MedadminSessionServiceFactory.getMedadminService()
                .getM4AXml(GetRequest.newBuilder().withDefaults().build());

        MedadminSessionServiceFactory.getMedadminService().confirmAllTAcks(response);

        assertTrue(response.getMsgResponses().get(0).getSingleNurseContractualCareResponses().size() > 0
                || response.getMsgResponses().get(0).getSinglePalliativeCareResponses().size() > 0
                || response.getMsgResponses().get(0).getSingleSpecificTechnicalCareResponses().size() > 0
                || response.getMsgResponses().get(0).getSingleNurseContractualCareUpdates().size() > 0);

        MedadminSessionServiceFactory.getMedadminService().confirmAllMessages(response);
    }

    @Test
    public void getM4AFlatMessage() throws Exception {
        ProcessedGetResponse<byte[]> response = MedadminSessionServiceFactory.getMedadminService()
                .getM4AFlat(GetRequest.newBuilder().withMaxMessages(1).withMaxTAcks(0).build());

        byte[] businessContent = response.getMsgResponses().get(0).getBusinessResponse();
        LOG.info("Content of M4A_FLAT response: {}", new String(businessContent));
        assertNotNull(businessContent);

        assertSignature(response.getMsgResponses().get(0));
    }

    @Test
    public void getM4ACnfXml() throws Exception {
        M4ACnfXmlProcessedGetResponse response = MedadminSessionServiceFactory.getMedadminService()
                .getM4ACnfXml(GetRequest.newBuilder().withDefaults().build());

        assertTrue(response.getMsgResponses().get(0).getSingleNurseContractualCareRequests().size() > 0
                || response.getMsgResponses().get(0).getSinglePalliativeCareRequests().size() > 0
                || response.getMsgResponses().get(0).getSingleSpecificTechnicalCareRequests().size() > 0);

        MedadminSessionServiceFactory.getMedadminService().confirmAllMessages(response);
    }

    @Test
    public void getRejected() throws Exception {
        MedAdminService service = MedadminSessionServiceFactory.getMedadminService();
        GetRequest request = GetRequest.newBuilder().withMaxMessages(1).withMaxTAcks(0).build();
        RejectXmlProcessedGetResponse response = service.getRejected(request);

        if (response.getMsgResponses().isEmpty()) {
            service.postM4AFlat(
                ConnectorIOUtils.getResourceAsByteArray("/examples/medadmin/async/request/EH-420001.txt"),
                "600",
                new InputReference()
            );
            TimeUnit.SECONDS.sleep(SECONDS_WAITING);
        }

        request = GetRequest.newBuilder().withMaxMessages(1).withMaxTAcks(0).build();
        response = service.getRejected(request);

        assertFalse(response.getMsgResponses().isEmpty());
        assertTrue(response.getMsgResponses().get(0).getRejectInbResponses().size() > 0);

        service.confirmAllMessages(response);
    }

    @Test
    public void getM4ACnfFlat() throws Exception {
        ProcessedGetResponse<byte[]> response = MedadminSessionServiceFactory.getMedadminService()
                .getM4ACnfFlat(GetRequest.newBuilder().withDefaults().build());

        byte[] businessContent = response.getMsgResponses().get(0).getBusinessResponse();
        LOG.info("Content of M4A_CNF_FLAT response: {}", new String(businessContent));
        assertNotNull(businessContent);

        MedadminSessionServiceFactory.getMedadminService().confirmAllMessages(response);
    }

    private <T> void assertSignature(ProcessedMsgResponse<T> processedMsgResponse) {
        assertEquals("SignatureVerificationResult should contain no error", 0, processedMsgResponse.getSignatureVerificationResult().getErrors().size());
        assertNotNull("Missing xades", processedMsgResponse.getMsgResponse().getXadesT());
        XmlAsserter.assertSimilar(ConnectorXmlUtils.toObject(processedMsgResponse.getSignedData(), EncryptedKnownContent.class), processedMsgResponse.getRawDecryptedBlob());
    }
}
