/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.decrypt;

import be.fgov.ehealth.etee.crypto.cert.CertPathChecker;
import be.fgov.ehealth.etee.crypto.cert.CertificateStatus;
import be.fgov.ehealth.etee.crypto.crl.CRLChecker;
import be.fgov.ehealth.etee.crypto.crl.CRLData;
import be.fgov.ehealth.etee.crypto.decrypt.TimeStampTrustResult;
import be.fgov.ehealth.etee.crypto.status.CryptoResult;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.utils.KeyManager;
import java.io.OutputStream;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.cert.CertStore;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.security.cert.X509Extension;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cms.CMSSignatureAlgorithmNameGenerator;
import org.bouncycastle.cms.DefaultCMSSignatureAlgorithmNameGenerator;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.cms.SignerInformationVerifier;
import org.bouncycastle.cms.bc.BcRSASignerInfoVerifierBuilder;
import org.bouncycastle.operator.DefaultDigestAlgorithmIdentifierFinder;
import org.bouncycastle.operator.DefaultSignatureAlgorithmIdentifierFinder;
import org.bouncycastle.operator.DigestAlgorithmIdentifierFinder;
import org.bouncycastle.operator.DigestCalculator;
import org.bouncycastle.operator.DigestCalculatorProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.SignatureAlgorithmIdentifierFinder;
import org.bouncycastle.operator.bc.BcDigestCalculatorProvider;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.tsp.TSPValidationException;
import org.bouncycastle.tsp.TimeStampToken;
import org.bouncycastle.tsp.TimeStampTokenInfo;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.Selector;
import org.bouncycastle.util.Store;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

final class TimeStampTrustService {
    private static final Logger LOGGER = LoggerFactory.getLogger(TimeStampTrustService.class);
    private static final Marker fatalMarker = MarkerFactory.getMarker((String)"FATAL");
    private KeyStore trustStore;
    private CRLChecker crlChecker;
    private CertPathChecker certPathChecker;

    public TimeStampTrustService(KeyStore trustStore, CRLChecker crlChecker, CertPathChecker certPathChecker) {
        this.trustStore = trustStore;
        this.crlChecker = crlChecker;
        this.certPathChecker = certPathChecker;
    }

    public TimeStampTrustResult verify(TimeStampToken timeStampToken, SignerInformation signerInformation) {
        LOGGER.debug("Verify if timeStampToken is related to this signature.");
        TimeStampTrustResult result = this.verify(timeStampToken);
        try {
            TimeStampTokenInfo tstInfo = timeStampToken.getTimeStampInfo();
            DigestCalculator digCalc = new BcDigestCalculatorProvider().get(tstInfo.getHashAlgorithm());
            OutputStream dOut = digCalc.getOutputStream();
            dOut.write(signerInformation.getSignature());
            dOut.close();
            byte[] expectedDigest = digCalc.getDigest();
            if (!Arrays.constantTimeAreEqual((byte[])expectedDigest, (byte[])tstInfo.getMessageImprintDigest())) {
                LOGGER.error(fatalMarker, "Timestamp not applicable for the signature on this message: Incorrect digest in message imprint");
                result.getErrors().add(NotificationError.TIMESTAMPTOKEN_DOES_NOT_MATCH_OUTER_SIGNATURE);
            }
            LOGGER.info("Timestamp applicable for the signature on this message.");
        }
        catch (Exception e) {
            LOGGER.error(fatalMarker, "Validation of timestamp failed", (Throwable)e);
            result.getErrors().add(NotificationError.TIMESTAMPTOKEN_INVALID);
        }
        return result;
    }

    public TimeStampTrustResult verify(TimeStampToken timeStampToken) {
        return this.verify(timeStampToken, new ArrayList<X509CRL>());
    }

    public TimeStampTrustResult verify(TimeStampToken timeStampToken, List<X509CRL> crls) {
        LOGGER.debug("Verify if timeStampToken signature may be trusted.");
        TimeStampTrustResult result = new TimeStampTrustResult();
        try {
            X509Certificate[] certificates = this.getCertificatesFromToken(timeStampToken);
            this.verifyChain(certificates);
            for (int index = 0; index < certificates.length; ++index) {
                X509Certificate chainCert = certificates[index];
                if (this.isTrustStoreCert(chainCert)) {
                    LOGGER.info("ChainCertificate " + chainCert.getSubjectX500Principal() + " found in trustStore. No revocation checks will be done.");
                    break;
                }
                this.verifyRevocation(crls, result, certificates, index, chainCert);
                if (!result.hasErrors()) continue;
                return result;
            }
        }
        catch (TSPValidationException e) {
            LOGGER.warn("TimeStampToken did not validate", (Throwable)e);
            result.getErrors().add(NotificationError.TIMESTAMPTOKEN_INVALID);
        }
        catch (CertificateExpiredException e) {
            LOGGER.warn("TimeStampToken certificate chain is expired.", (Throwable)e);
            result.getErrors().add(NotificationError.TIMESTAMPTOKEN_CHAIN_EXPIRED);
        }
        catch (Exception e) {
            LOGGER.warn("Validation of TimeStampToken Security Requirements failed.", (Throwable)e);
            result.getErrors().add(NotificationError.TIMESTAMPTOKEN_TRUST_FAILED);
        }
        if (!result.hasErrors()) {
            LOGGER.info("TimeStampToken is TRUSTED.");
        }
        return result;
    }

    private void verifyRevocation(List<X509CRL> crls, TimeStampTrustResult result, X509Certificate[] certificates, int index, X509Certificate chainCert) throws KeyStoreException, InvalidAlgorithmParameterException, NoSuchAlgorithmException {
        X509Certificate issuer = this.deductIssuer(certificates, index, chainCert);
        if (issuer != null) {
            ArrayList<X509Extension> list = new ArrayList<X509Extension>();
            list.addAll(crls);
            list.add(issuer);
            CertStore certStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(list));
            CryptoResult<CRLData> crlResult = this.crlChecker.validate(chainCert, certStore);
            if (crlResult.hasErrors()) {
                result.getErrors().addAll(crlResult.getErrors());
                result.getErrors().add(NotificationError.TIMESTAMPTOKEN_TRUST_FAILED);
                return;
            }
            if (crlResult.hasWarnings()) {
                LOGGER.warn("CRL Check completed with warnings: " + crlResult.getWarnings());
                result.getWarnings().addAll(crlResult.getWarnings());
            }
            result.addCrls(crlResult.getData().getCrls());
            switch (crlResult.getData().getCertStatus()) {
                case VALID: {
                    break;
                }
                case REVOKED: {
                    LOGGER.warn("Certificate chain of TimeStampToken has been revoked. TimeStampToken is NOT TRUSTED.");
                    result.getErrors().add(NotificationError.TIMESTAMPTOKEN_CHAIN_REVOKED);
                    break;
                }
                default: {
                    LOGGER.warn("Revocation status of certificate " + chainCert.getSubjectDN() + " unknown. If the Root CA of this chain is trusted, this certificate will be accepted without validated revocation status.");
                }
            }
        }
    }

    private X509Certificate deductIssuer(X509Certificate[] certificates, int i, X509Certificate chainCert) throws KeyStoreException {
        if (i == certificates.length - 1) {
            return KeyManager.getIssuerFromTrustStore(this.trustStore, chainCert);
        }
        return certificates[i + 1];
    }

    private X509Certificate[] getCertificatesFromToken(TimeStampToken timeStampToken) throws OperatorCreationException, CertificateException, TSPException {
        Store certs = timeStampToken.getCertificates();
        Collection tokenSigners = certs.getMatches((Selector)timeStampToken.getSID());
        if (tokenSigners == null || tokenSigners.isEmpty()) {
            throw new CertificateException("TimeStampToken does not contain a certificate.");
        }
        X509CertificateHolder tokenSigner = (X509CertificateHolder)tokenSigners.iterator().next();
        SignerInformationVerifier siv = new BcRSASignerInfoVerifierBuilder((CMSSignatureAlgorithmNameGenerator)new DefaultCMSSignatureAlgorithmNameGenerator(), (SignatureAlgorithmIdentifierFinder)new DefaultSignatureAlgorithmIdentifierFinder(), (DigestAlgorithmIdentifierFinder)new DefaultDigestAlgorithmIdentifierFinder(), (DigestCalculatorProvider)new BcDigestCalculatorProvider()).build(tokenSigner);
        timeStampToken.validate(siv);
        return this.convert(certs.getMatches(null));
    }

    private boolean isTrustStoreCert(X509Certificate chainCert) throws KeyStoreException, CertificateException {
        if (this.trustStore != null) {
            return this.trustStore.getCertificateAlias(chainCert) != null;
        }
        return KeyManager.isSelfSigned(chainCert);
    }

    private void verifyChain(X509Certificate[] timeStampTokenCertList) throws CertificateException, KeyStoreException, NoSuchAlgorithmException, NoSuchProviderException {
        if (this.trustStore != null) {
            X509Certificate tsaCert;
            List<X509Certificate> timeStampTokenCertChain = java.util.Arrays.asList(timeStampTokenCertList);
            if (!timeStampTokenCertChain.isEmpty() && this.isTrustStoreCert(tsaCert = timeStampTokenCertChain.get(0))) {
                tsaCert.checkValidity();
                LOGGER.info("Valid Signing Certificate of TimeStampToken found in trustStore. Certificate chain accepted.");
                return;
            }
            CryptoResult<CertificateStatus> certPathResult = this.certPathChecker.validate(timeStampTokenCertChain);
            if (certPathResult.hasErrors()) {
                LOGGER.warn("CertPath Validation completed with errors: " + certPathResult);
                throw new CertificateException("CertPath of TimeStampToken could not be verified.");
            }
            if (certPathResult.hasWarnings()) {
                LOGGER.warn("CertPath Validation completed with warnings: " + certPathResult.getWarnings());
            }
            this.verifyCertStatus(certPathResult.getData());
        }
    }

    private void verifyCertStatus(CertificateStatus certStatus) throws CertificateException {
        switch (certStatus) {
            case VALID: {
                LOGGER.info("CertPath of TimeStampToken is valid.");
                break;
            }
            case EXPIRED: {
                throw new CertificateExpiredException("CertPath of TimeStampToken is EXPIRED.");
            }
            default: {
                throw new CertificateException("CertPath of TimeStampToken is NOT TRUSTED.");
            }
        }
    }

    private X509Certificate[] convert(Collection<X509CertificateHolder> collectionCertChain) {
        X509Certificate[] certChain = new X509Certificate[collectionCertChain.size()];
        int i = 0;
        for (X509CertificateHolder x509CertificateHolder : collectionCertChain) {
            certChain[i++] = this.extractCert(x509CertificateHolder);
        }
        return certChain;
    }

    private X509Certificate extractCert(X509CertificateHolder certificateHolder) {
        try {
            return new JcaX509CertificateConverter().getCertificate(certificateHolder);
        }
        catch (CertificateException e) {
            throw new IllegalArgumentException("The certificate could not be extracted", e);
        }
    }
}

