/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.decrypt;

import be.fgov.ehealth.etee.crypto.cert.CertPathChecker;
import be.fgov.ehealth.etee.crypto.cert.CertificateStatus;
import be.fgov.ehealth.etee.crypto.decrypt.CMSMessageContext;
import be.fgov.ehealth.etee.crypto.decrypt.SignedDataNotification;
import be.fgov.ehealth.etee.crypto.decrypt.SignedDataVerifier;
import be.fgov.ehealth.etee.crypto.decrypt.SignedDataVerifierResult;
import be.fgov.ehealth.etee.crypto.decrypt.SignerInfoAttributes;
import be.fgov.ehealth.etee.crypto.decrypt.SignerInfoAttributesReceiver;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPChecker;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPData;
import be.fgov.ehealth.etee.crypto.ocsp.RevocationValues;
import be.fgov.ehealth.etee.crypto.policies.SignatureLayer;
import be.fgov.ehealth.etee.crypto.policies.SigningCredential;
import be.fgov.ehealth.etee.crypto.policies.SigningPolicy;
import be.fgov.ehealth.etee.crypto.status.CryptoResult;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.utils.KeyManager;
import java.io.ByteArrayInputStream;
import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import javax.crypto.BadPaddingException;
import org.bouncycastle.asn1.x509.CertificateList;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.SignerId;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.cms.SignerInformationVerifier;
import org.bouncycastle.cms.jcajce.JcaSimpleSignerInfoVerifierBuilder;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class SignedDataVerifierAbstract
implements SignedDataVerifier {
    private static final Logger LOGGER = LoggerFactory.getLogger(SignedDataVerifierAbstract.class);
    private static final String CMS_SIGNATURE_NOT_VERIFIED = "The CMS signature could not be verified. The root cause can be that the Signed-Data content is empty. Check the exception trace.";
    private final Collection<SigningPolicy> signingPolicies;
    private SignerInfoAttributesReceiver signerInfoAttributesReceiver;
    private OCSPChecker ocspChecker;
    private CertPathChecker certPathChecker;
    private SignatureLayer layer;
    private int[] acceptableKeyUsages;

    public SignedDataVerifierAbstract(SignatureLayer layer, Collection<SigningPolicy> signingPolicies, int[] acceptableKeyUsages, OCSPChecker ocspChecker, CertPathChecker certPathChecker, SignerInfoAttributesReceiver signerInfoAttributesReceiver) {
        this.layer = layer;
        this.signingPolicies = Collections.unmodifiableCollection(signingPolicies);
        this.acceptableKeyUsages = (int[])acceptableKeyUsages.clone();
        this.ocspChecker = ocspChecker;
        this.certPathChecker = certPathChecker;
        this.signerInfoAttributesReceiver = signerInfoAttributesReceiver;
    }

    protected SigningPolicy definePolicy(SignerInformation signerInformation) {
        SignerId signerId = signerInformation.getSID();
        if (signerId != null && signerId.getSubjectKeyIdentifier() != null) {
            LOGGER.debug("Signing Policy detected. Message will be validated according to policy " + (Object)((Object)SigningPolicy.WEB_AUTHN));
            return SigningPolicy.WEB_AUTHN;
        }
        String encryptionAlgOID = signerInformation.getEncryptionAlgOID();
        for (SigningPolicy signingPolicy : this.signingPolicies) {
            if (signingPolicy == SigningPolicy.WEB_AUTHN) continue;
            String supportedEncryptionAlgorithm = signingPolicy.getEncryptionAlgorithmOID();
            if (supportedEncryptionAlgorithm.equals(encryptionAlgOID)) {
                LOGGER.debug("Signing Policy detected. Message will be validated according to policy " + (Object)((Object)signingPolicy));
                return signingPolicy;
            }
            for (String altEncryptionAlgOID : signingPolicy.getAltEncryptionAlgorithmOIDs()) {
                if (!altEncryptionAlgOID.equals(encryptionAlgOID)) continue;
                LOGGER.debug("Signing Policy detected. Message will be validated according to policy " + (Object)((Object)signingPolicy));
                return signingPolicy;
            }
        }
        LOGGER.debug("Signing Policy could not be deducted from message. Message will be validated according to default policy " + (Object)((Object)this.signingPolicies.iterator().next()));
        return null;
    }

    protected void verifyAlgorithmsUsed(SignedDataVerifierResult result, SignerInformation signerInfos, SigningPolicy signingPolicy) {
        boolean digestAlgorithmAccepted = this.digestAlgorithmOIDIsAccepted(signerInfos, signingPolicy);
        boolean signatureAlgorithmAccepted = this.signatureAlgorithmOIDIsAuthorized(signerInfos, signingPolicy);
        if (!digestAlgorithmAccepted) {
            result.getErrors().add(SignedDataNotification.SIGNATURE_DIGEST_ALGORITHM_UNAUTHORIZED.get(this.layer));
        }
        if (!signatureAlgorithmAccepted) {
            result.getErrors().add(SignedDataNotification.SIGNATURE_ALGORITHM_UNAUTHORIZED.get(this.layer));
        }
    }

    protected SignerInfoAttributes getSignerInfoAttributes(SignedDataVerifierResult result, SignerInformation signerInfos) {
        try {
            return this.signerInfoAttributesReceiver.parse(signerInfos);
        }
        catch (TSPException e) {
            LOGGER.warn(this.buildMsg("Contains invalid timestamptoken. Unsigned SignerInfo Attributes will be ignored."), (Throwable)e);
            result.getErrors().add(NotificationError.TIMESTAMPTOKEN_INVALID);
            return new SignerInfoAttributes();
        }
    }

    protected void verifyCertificate(SignedDataVerifierResult result, CMSMessageContext<?> msgContext, SignerInfoAttributes signerInfoAttributes, Date inSigningTime, SigningPolicy signingPolicy) {
        SignerInformation signerInformation = msgContext.getSignerInformation();
        X509Certificate signerCertificate = msgContext.getAuthenticationCertificate();
        List<X509Certificate> certChain = msgContext.getCertificateChain();
        Date signingTime = inSigningTime;
        if (signingTime == null) {
            signingTime = new Date();
        }
        this.verifySignature(result, signerInformation, signerCertificate);
        this.verifyCertificateKeyUsage(result, signerCertificate);
        this.verifyKeyLength(result, signerCertificate, signingPolicy);
        this.verifyCertificateValidity(result, signerCertificate, signingTime);
        this.verifyCertificateChainTrust(result, certChain, signingTime);
        this.verifyCertificateRevocationStatus(result, signerCertificate, signingTime, this.buildRevocationValues(signerInfoAttributes));
    }

    protected void verifyPublicKey(SignedDataVerifierResult result, CMSMessageContext<?> msgContext, String subjectKeyIdentifier, SignerInfoAttributes signerInfoAttributes, SigningPolicy signingPolicy, SigningCredential ... signers) {
        SignerInformation signerInformation = msgContext.getSignerInformation();
        PublicKey signerKey = this.getPublicKey(result, subjectKeyIdentifier, signers);
        if (signers == null || signers.length == 0) {
            LOGGER.warn(this.buildMsg("No Signer credentials available."));
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.getLayer()));
        } else {
            if (signerKey == null) {
                LOGGER.warn(this.buildMsg("Signer's Key missing in list of provided signing credentials."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.getLayer()));
                return;
            }
            this.verifySignature(result, signerInformation, signerKey);
            this.verifyKeyLength(result, signerKey, signingPolicy);
        }
    }

    protected String getSubjectKeyIdentifier(CMSMessageContext<?> msgContext) {
        SignerInformation signerInformation = msgContext.getSignerInformation();
        SignerId signerId = signerInformation.getSID();
        if (signerId == null || signerId.getSubjectKeyIdentifier() == null) {
            return null;
        }
        try {
            return new String(Base64.encode((byte[])msgContext.getSignerInformation().getSID().getSubjectKeyIdentifier()), "UTF8");
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException("Failed to encode Signer's SubjectKeyIdentifier.", e);
        }
    }

    private PublicKey getPublicKey(SignedDataVerifierResult result, String subjectKeyId, SigningCredential ... signers) {
        for (SigningCredential signer : signers) {
            if (!subjectKeyId.equals(signer.getSubjectKeyIdentifier())) continue;
            return signer.getPublicKey();
        }
        LOGGER.warn(this.buildMsg("No PublicKey found for subjectKeyIdentifier defined in signature: " + subjectKeyId));
        return null;
    }

    protected void verifySignature(SignedDataVerifierResult result, SignerInformation signerInfos, X509Certificate signerCertificate) {
        try {
            if (this.isSignatureDoneWithSignerCertificate(signerInfos, signerCertificate)) {
                LOGGER.debug(this.buildMsg("SignerInfo signature is OK."));
            } else {
                LOGGER.warn(this.buildMsg("SignerInfo signature is not created with private key corresponding to given cert."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
        }
        catch (OperatorCreationException e) {
            LOGGER.warn(this.buildMsg("Signer certificate could not be extracted from the message"), (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
        catch (CMSException e) {
            if (e.getCause() instanceof BadPaddingException || e.getCause() instanceof InvalidKeyException) {
                LOGGER.warn(this.buildMsg("Signature is not created with the private key that corresponds to the given certificate."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
            LOGGER.warn(CMS_SIGNATURE_NOT_VERIFIED, (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
    }

    protected void verifySignature(SignedDataVerifierResult result, SignerInformation signerInfos, PublicKey signerKey) {
        try {
            if (this.isSignatureDoneWithSignerKey(signerInfos, signerKey)) {
                LOGGER.debug(this.buildMsg("SignerInfo signature is OK."));
            } else {
                LOGGER.warn(this.buildMsg("SignerInfo signature is not created with private key corresponding to given cert."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
        }
        catch (OperatorCreationException e) {
            LOGGER.warn(this.buildMsg("Signer certificate could not be extracted from the message"), (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
        catch (CMSException e) {
            if (e.getCause() instanceof BadPaddingException || e.getCause() instanceof InvalidKeyException) {
                LOGGER.warn(this.buildMsg("Signature is not created with the private key that corresponds to the given certificate."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
            LOGGER.warn(CMS_SIGNATURE_NOT_VERIFIED, (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
    }

    private void verifyKeyLength(SignedDataVerifierResult result, X509Certificate signerCertificate, SigningPolicy signingPolicy) {
        this.verifyKeyLength(result, signerCertificate.getPublicKey(), signingPolicy);
    }

    private void verifyKeyLength(SignedDataVerifierResult result, PublicKey signerKey, SigningPolicy signingPolicy) {
        String binaryRepresentation = this.getBinaryRepresentation(signerKey);
        if (this.keyLengthNotAccepted(binaryRepresentation, signingPolicy)) {
            String errMsg = this.buildMsg("Signer's key length is not authorized.");
            if (binaryRepresentation != null && signingPolicy != null) {
                errMsg = errMsg + String.format("Is %d and should be %d", binaryRepresentation.length(), signingPolicy.getSignKeySize());
            }
            LOGGER.warn(errMsg);
            result.getErrors().add(SignedDataNotification.SIGNATURE_KEY_LENGTH_UNAUTHORIZED.get(this.layer));
        } else {
            LOGGER.debug(this.buildMsg("Signer's key length is authorized."));
        }
    }

    private void verifyCertificateValidity(SignedDataVerifierResult result, X509Certificate signerCertificate, Date signingTime) {
        try {
            signerCertificate.checkValidity(signingTime);
            LOGGER.debug(this.buildMsg("Signer cert valid at signing time " + signingTime));
        }
        catch (CertificateExpiredException e) {
            LOGGER.warn(this.buildMsg("Signer cert has expired."), (Throwable)e);
            result.getErrors().add(SignedDataNotification.CERTIFICATE_EXPIRED.get(this.layer));
        }
        catch (CertificateNotYetValidException e) {
            LOGGER.warn(this.buildMsg("Signer cert is not yet valid."), (Throwable)e);
            result.getErrors().add(SignedDataNotification.CERTIFICATE_NOT_YET_VALID.get(this.layer));
        }
    }

    private void verifyCertificateChainTrust(SignedDataVerifierResult result, List<X509Certificate> certificateChain, Date signingTime) {
        LOGGER.debug(this.buildMsg("Verifying the trust in the certificate chain in the Signed-Data"));
        boolean isTrustedChain = false;
        if (!certificateChain.isEmpty()) {
            CryptoResult<CertificateStatus> certPathResult = this.certPathChecker.validate(certificateChain, signingTime);
            if (certPathResult.hasErrors()) {
                LOGGER.warn("CertPath Validation completed with errors: " + certPathResult);
                result.getErrors().addAll(certPathResult.getErrors());
            } else {
                CertificateStatus certStatus = certPathResult.getData();
                if (certStatus == CertificateStatus.VALID) {
                    LOGGER.debug(this.buildMsg("Certificate chain trusted at signing time " + signingTime));
                    isTrustedChain = true;
                } else {
                    LOGGER.warn(this.buildMsg("Certificate chain is not trusted: " + (Object)((Object)certStatus)));
                }
            }
            if (certPathResult.hasWarnings()) {
                LOGGER.warn("CertPath Validation completed with warnings: " + certPathResult.getWarnings());
                result.getWarnings().addAll(certPathResult.getWarnings());
            }
        } else {
            LOGGER.warn(this.buildMsg("No cert chain available."));
        }
        if (!isTrustedChain) {
            result.getErrors().add(SignedDataNotification.CERTIFICATE_CHAIN_NOT_TRUSTED.get(this.layer));
        }
    }

    private void verifyCertificateRevocationStatus(SignedDataVerifierResult result, X509Certificate signerCertificate, Date signingTime, RevocationValues revocationValues) {
        CryptoResult<OCSPData> checkerResult = this.ocspChecker.validate(signerCertificate, signingTime, revocationValues);
        if (checkerResult.hasWarnings()) {
            LOGGER.error(this.buildMsg("OCSP Validation completed with warnings: " + checkerResult));
            result.getWarnings().addAll(checkerResult.getWarnings());
        }
        if (checkerResult.hasErrors()) {
            LOGGER.error(this.buildMsg("OCSP Validation completed with errors: " + checkerResult));
            result.getErrors().addAll(checkerResult.getErrors());
            result.getErrors().add(SignedDataNotification.CERTIFICATE_STATUS_UNKNOWN.get(this.layer));
        }
        if (checkerResult.getFatal() == null) {
            if (CertificateStatus.REVOKED.equals((Object)checkerResult.getData().getCertStatus())) {
                result.getErrors().add(SignedDataNotification.CERTIFICATE_REVOKED.get(this.layer));
            }
            result.setOcspData(checkerResult.getData());
        }
    }

    private RevocationValues buildRevocationValues(SignerInfoAttributes signerInfos) {
        RevocationValues revocationValues = new RevocationValues();
        for (BasicOCSPResp ocsp : signerInfos.getOcspResponses()) {
            try {
                revocationValues.getOcspVals().add(ocsp.getEncoded());
            }
            catch (Exception e) {
                LOGGER.warn(this.buildMsg("Embedded OCSP could not be encoded. It will be skipped."), (Throwable)e);
            }
        }
        for (CertificateList crl : signerInfos.getCrls()) {
            try {
                ByteArrayInputStream inStream = new ByteArrayInputStream(crl.getEncoded());
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                revocationValues.getCrlVals().add((X509CRL)cf.generateCRL(inStream));
                inStream.close();
            }
            catch (Exception e) {
                LOGGER.warn(this.buildMsg("Embedded CRL could not be encoded. It will be skipped."), (Throwable)e);
            }
        }
        return revocationValues;
    }

    private String getBinaryRepresentation(PublicKey pubKey) {
        String binaryRepresentation = null;
        if (pubKey instanceof RSAPublicKey) {
            binaryRepresentation = ((RSAPublicKey)pubKey).getModulus().toString(2);
        } else if (pubKey instanceof DSAPublicKey) {
            binaryRepresentation = ((DSAPublicKey)pubKey).getY().toString(2);
        }
        return binaryRepresentation;
    }

    boolean digestAlgorithmOIDIsAccepted(SignerInformation signerInfo, SigningPolicy signingPolicy) {
        if (signingPolicy == null) {
            return false;
        }
        String digestAlgorithm = signerInfo.getDigestAlgOID();
        String supportedDigestAlgorithm = signingPolicy.getDigestAlgorithmOID();
        if (supportedDigestAlgorithm.equals(digestAlgorithm)) {
            return true;
        }
        LOGGER.warn("Digest Algorithm OID [" + digestAlgorithm + "] not supported by policy [" + (Object)((Object)signingPolicy) + "]");
        return false;
    }

    boolean signatureAlgorithmOIDIsAuthorized(SignerInformation signerInfo, SigningPolicy signingPolicy) {
        if (signingPolicy == null) {
            return false;
        }
        String encryptionAlgOID = signerInfo.getEncryptionAlgOID();
        String supportedEncryptionAlgorithm = signingPolicy.getEncryptionAlgorithmOID();
        if (supportedEncryptionAlgorithm.equals(encryptionAlgOID)) {
            return true;
        }
        for (String altEncryptionAlgOID : signingPolicy.getAltEncryptionAlgorithmOIDs()) {
            if (!altEncryptionAlgOID.equals(encryptionAlgOID)) continue;
            return true;
        }
        LOGGER.warn("Encryption Algorithm OID [" + encryptionAlgOID + "] not supported by policy [" + (Object)((Object)signingPolicy) + "]");
        return false;
    }

    void verifyCertificateKeyUsage(SignedDataVerifierResult result, X509Certificate signerCertificate) {
        int certKeyUsage = KeyManager.getKeyUsage(signerCertificate);
        for (int i = 0; i < this.acceptableKeyUsages.length; ++i) {
            if ((certKeyUsage & this.acceptableKeyUsages[i]) == 0) continue;
            LOGGER.debug(this.buildMsg("Signer certificate's key usage is valid."));
            return;
        }
        LOGGER.warn(this.buildMsg("Signer certificate's key usage is invalid."));
        result.getErrors().add(SignedDataNotification.CERTIFICATE_HAS_INVALID_KEYUSAGE.get(this.getLayer()));
    }

    boolean isSignatureDoneWithSignerCertificate(SignerInformation signerInfos, X509Certificate signerCertificate) throws OperatorCreationException, CMSException {
        SignerInformationVerifier verifier = new JcaSimpleSignerInfoVerifierBuilder().build(signerCertificate);
        return signerInfos.verify(verifier);
    }

    boolean isSignatureDoneWithSignerKey(SignerInformation signerInfos, PublicKey signerKey) throws OperatorCreationException, CMSException {
        SignerInformationVerifier verifier = new JcaSimpleSignerInfoVerifierBuilder().build(signerKey);
        return signerInfos.verify(verifier);
    }

    boolean keyLengthNotAccepted(String binaryRepresentation, SigningPolicy signingPolicy) {
        return binaryRepresentation == null || signingPolicy == null || binaryRepresentation.length() < signingPolicy.getSignKeySize();
    }

    protected SignatureLayer getLayer() {
        return this.layer;
    }

    protected String buildMsg(String msg) {
        return "[Signature Layer " + (Object)((Object)this.getLayer()) + "] " + msg;
    }

    boolean isSignerCertificateRequired() {
        return this.layer == SignatureLayer.OUTER && !this.signingPolicies.contains((Object)SigningPolicy.WEB_AUTHN);
    }

    protected void verifySigner(SignedDataVerifierResult result, X509Certificate signerCertificate, SigningCredential ... signers) {
        boolean signerAcceptable = false;
        for (SigningCredential acceptedSigner : signers) {
            if (acceptedSigner.getCertificateChain() != null && !acceptedSigner.getCertificateChain().isEmpty() && signerCertificate.equals(acceptedSigner.getCertificateChain().get(0))) {
                signerAcceptable = true;
                break;
            }
            if (acceptedSigner.getPublicKey() == null || !signerCertificate.getPublicKey().equals(acceptedSigner.getPublicKey())) continue;
            signerAcceptable = true;
            break;
        }
        if (!signerAcceptable) {
            LOGGER.warn(this.buildMsg("Message is not signed by expected signer credential(s): Crt Serialnumber = " + signerCertificate.getSerialNumber()));
            result.getErrors().add(SignedDataNotification.SIGNER_NOT_ALLOWED.get(this.getLayer()));
        }
    }
}

