﻿/*
 * Copyright (c) eHealth
 */
using System;
using be.ehealth.business.kmehrcommons;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.fgov.ehealth.technicalconnector.tests.utils;
using java.lang;
using NUnit.Framework;
using java.util;
using org.joda.time;
using be.ehealth.business.common.domain;
using be.ehealth.businessconnector.therlink.builders;
using be.ehealth.businessconnector.therlink.domain;
using be.ehealth.businessconnector.therlink.domain.requests;
using be.ehealth.businessconnector.therlink.domain.responses;
using be.ehealth.businessconnector.therlink.exception;
using be.ehealth.businessconnector.therlink.mappers;
using be.ehealth.businessconnector.therlink.service;
using be.ehealth.businessconnector.therlink.util;
using be.ehealth.technicalconnector.config;
using be.ehealth.technicalconnector.exception;
using be.ehealth.technicalconnector.service.sts.security;
using be.ehealth.technicalconnector.session;
using be.ehealth.technicalconnector.session.impl;

namespace be.ehealth.businessconnector.therlink.service.impl
{
	/**
	 * Unit test for {@link TherLinkService}
	 * 
	 * @author EHP
	 * 
	 */
	[TestFixture]
	public class TherlinkServiceBasicUserIntegrationTest : AbstractTherlinkIsolationIntegrationTest
	{

		private static string MAIN_KMEHR_QUALITY = "main.kmehr.quality";

		private static string linkType="";

		private static HcParty concernedHealthCareProfessional;

		[SetUpAttribute]
		public static void init() {
			SessionInitializer.init("be.ehealth.businessconnector.therlink.test.properties", true, SupportedLanguages.NET.getAbbreviation());
			
			ConfigValidator configValidator = ConfigFactory.getConfigValidator();
			linkType = TherapeuticLinkTypeValues.valueOf(configValidator.getProperty(MAIN_KMEHR_QUALITY).ToUpper()).getTherlinkType();
			concernedHealthCareProfessional = ConfigReader.getCareProvider();
		}

		[Test]
		public void putTherapeuticLink() {
			TherLinkService therLinkService = ServiceFactory.getTherLinkService();
			SAMLToken samlToken = Session.getInstance().getSession().getSAMLToken();

			Proof proof = RequestObjectBuilderFactory.getProofBuilder().createProofForEidReading();
			be.ehealth.business.common.domain.Patient patient = TherlinkTestUtils.fillTherlinkPatientWithOfflinePatientDataForProof(proof);

			PutTherapeuticLinkRequest request = RequestObjectBuilderFactory.getRequestObjectBuilder().createPutTherapeuticLinkRequest(null, patient, concernedHealthCareProfessional, linkType, null, proof);

			be.fgov.ehealth.hubservices.core.v2.PutTherapeuticLinkRequest mapPutTherapeuticLinkRequest = MapperFactory.getRequestObjectMapper().mapPutTherapeuticLinkRequest(request);

			be.fgov.ehealth.hubservices.core.v2.PutTherapeuticLinkResponse putTherapeuticLink = therLinkService.putTherapeuticLink(samlToken, mapPutTherapeuticLinkRequest);
			PutTherapeuticLinkResponse response = MapperFactory.getResponseObjectMapper().mapJaxbToPutTherapeuticLinkResponse(putTherapeuticLink);

			logResponseAndCheckIfNoErrors(response);
		}

		[Test]
		public void putTherapeuticLinkWithIsi() {
			be.ehealth.business.common.domain.Patient patient = new be.ehealth.business.common.domain.Patient.Builder().withInss("08013126873").withFirstName("dummyname").withLastName("dummyName").withIsiPlus("9950014769").build();
			Proof isiProof = proofBuilder.createProofForIsiReading();
			TherlinkTestUtils.revokeNonReferrals(patient, isiProof);
			TherLinkService therLinkService = ServiceFactory.getTherLinkService();
			SAMLToken samlToken = Session.getInstance().getSession().getSAMLToken();

			PutTherapeuticLinkRequest request = requestObjectBuilder.createPutTherapeuticLinkRequest(null, patient, concernedHealthCareProfessional, linkType, null, isiProof);

			be.fgov.ehealth.hubservices.core.v2.PutTherapeuticLinkRequest mapPutTherapeuticLinkRequest = requestObjectMapper.mapPutTherapeuticLinkRequest(request);

			be.fgov.ehealth.hubservices.core.v2.PutTherapeuticLinkResponse putTherapeuticLink = therLinkService.putTherapeuticLink(samlToken, mapPutTherapeuticLinkRequest);
			PutTherapeuticLinkResponse response = responseObjectMapper.mapJaxbToPutTherapeuticLinkResponse(putTherapeuticLink);

			logResponseAndCheckIfNoErrors(response);
		}

		/**
		 * @param response
		 */
		private void logResponseAndCheckIfNoErrors(PutTherapeuticLinkResponse response) {
			Assert.IsNotNull(response);
			Assert.IsNotNull(response.getAcknowledge());
			Assert.IsNotNull(response.getAcknowledge().getListOfErrors());
			checkErrors(response);
		}

		[Test]
		public void getTherapeuticLink() {
			// create builders
			CommonObjectBuilder commonBuilder = RequestObjectBuilderFactory.getCommonBuilder();
			RequestObjectBuilder requestObjectBuilder = RequestObjectBuilderFactory.getRequestObjectBuilder();
			ProofBuilder proofBuilder = RequestObjectBuilderFactory.getProofBuilder();
			RequestObjectMapper requestObjectMapper = MapperFactory.getRequestObjectMapper();
			SAMLToken samlToken = Session.getInstance().getSession().getSAMLToken();
			ResponseObjectMapper responseObjectMapper = MapperFactory.getResponseObjectMapper();
			TherLinkService service = ServiceFactory.getTherLinkService();

			// retrieve all info to create therapeutic link
			string kmehrId = commonBuilder.createKmehrID();
			Author author = commonBuilder.createAuthor(requestObjectBuilder.getAuthorHcParties());

			Proof proof = proofBuilder.createProofForEidReading();
			be.ehealth.business.common.domain.Patient patientInfo = TherlinkTestUtils.fillTherlinkPatientWithOfflinePatientDataForProof(proof);
			HcParty careProviderHcParty = ConfigReader.getCareProvider();
			TherapeuticLink therapeuticLink = commonBuilder.createTherapeuticLink(new org.joda.time.DateTime(), new org.joda.time.DateTime().plusMonths(6), patientInfo, HcPartyUtil.getAuthorKmehrQuality(), null, null, careProviderHcParty);

			int maxRows = requestObjectBuilder.getMaxRows();


			GetTherapeuticLinkRequest request = new GetTherapeuticLinkRequest(new org.joda.time.DateTime(), kmehrId, author, therapeuticLink, maxRows, proof);

			be.fgov.ehealth.hubservices.core.v2.GetTherapeuticLinkRequest mappedRequest = requestObjectMapper.mapGetTherapeuticLinkRequest(request);

			be.fgov.ehealth.hubservices.core.v2.GetTherapeuticLinkResponse getTherapeuticLinkResponse = service.getTherapeuticLink(samlToken, mappedRequest);
			GetTherapeuticLinkResponse response = responseObjectMapper.mapJaxbToGetTherapeuticLinkResponse(getTherapeuticLinkResponse);

			Assert.IsNotNull(response);
			Assert.IsNotNull(response.getAcknowledge());
			Assert.IsNotNull(response.getAcknowledge().getListOfErrors());
			checkErrors(response);
		}

		/**
		 * @param response
		 */
		private void checkErrors(GeneralResponse response) {
			if (response.getAcknowledge().getListOfErrors().size() > 0) {
				StringBuilder sb = new StringBuilder();
				for (int indError=0; indError < response.getAcknowledge().getListOfErrors().size(); indError++) {
					TherapeuticLinkResponseError error = (TherapeuticLinkResponseError) response.getAcknowledge().getListOfErrors().get(indError);
					string errorDescription = error.getErrorDescription();
					string errorCode = error.getErrorCode();
					sb.append(errorCode).append(" : ").append(errorDescription).append(" ");
				}
				Assert.Fail(sb.toString());
			}
		}

		[Test]
		public void revokeTherapeuticLink() {
			Proof proof = proofBuilder.createProofForEidReading();
			be.ehealth.business.common.domain.Patient patient = TherlinkTestUtils.fillTherlinkPatientWithOfflinePatientDataForProof(proof);
			HcParty concernedHealthCareProfessionel = ConfigReader.getCareProvider();
			RequestObjectBuilder requestObjectBuilder = RequestObjectBuilderFactory.getRequestObjectBuilder();
			ResponseObjectMapper responseObjectMapper = MapperFactory.getResponseObjectMapper();
			RequestObjectMapper requestObjectMapper = MapperFactory.getRequestObjectMapper();

			TherLinkService service = ServiceFactory.getTherLinkService();
			PutTherapeuticLinkRequest putRequest = requestObjectBuilder.createPutTherapeuticLinkRequest(patient, concernedHealthCareProfessionel, linkType, proof);
			service.putTherapeuticLink(SessionManagerImpl.getInstance().getSession().getSAMLToken(), MapperFactory.getRequestObjectMapper().mapPutTherapeuticLinkRequest(putRequest));

			RevokeTherapeuticLinkRequest request = requestObjectBuilder.createRevokeTherapeuticLinkRequest(new org.joda.time.DateTime(), null, patient, concernedHealthCareProfessionel, linkType, null, proof);

			SAMLToken samlToken = Session.getInstance().getSession().getSAMLToken();
			be.fgov.ehealth.hubservices.core.v2.RevokeTherapeuticLinkRequest mapRevokeTherapeuticLinkRequest = requestObjectMapper.mapRevokeTherapeuticLinkRequest(request);
			be.fgov.ehealth.hubservices.core.v2.RevokeTherapeuticLinkResponse revokeTherapeuticLink = service.revokeTherapeuticLink(samlToken, mapRevokeTherapeuticLinkRequest);
			RevokeTherapeuticLinkResponse response = responseObjectMapper.mapJaxbToRevokeTherapeuticLinkResponse(revokeTherapeuticLink);


			Assert.IsNotNull(response);
			Assert.IsNotNull(response.getAcknowledge());
			Assert.IsNotNull(response.getAcknowledge().getListOfErrors());
			checkErrors(response);
		}
	}
}
