/*
 * Copyright (c) eHealth
 */
package be.ehealth.technicalconnector.utils;

import org.joda.time.DateTime;
import org.junit.Assert;
import org.junit.Test;


/**
 * Unit test for {@link XSDDateTimeConverter}
 * 
 * @author EHP
 * 
 */
public class XSDDateTimeConverterTest {

    public void test() throws Exception {
        Assert.assertTrue(DateUtils.parseDate("2006-06-09").equals(new DateTime(2006, 6, 9, 0, 0, 0, 0)));
    }

    @Test
    public void testDateCreatedInOtherTimezone() throws Exception {
        DateTime parsedDateTimeZonePlusOneJanuary = XSDDateTimeConverter.parseXSDDate("2013-01-01+01:00");
        DateTime parsedDateTimeZonePlusOneMay = XSDDateTimeConverter.parseXSDDate("2013-05-01+01:00");
        Assert.assertEquals("2013-01-01", XSDDateTimeConverter.printXSDDate(parsedDateTimeZonePlusOneJanuary));
        Assert.assertEquals("2013-05-01", XSDDateTimeConverter.printXSDDate(parsedDateTimeZonePlusOneMay)); // loses the timezone
    }

    @Test
    public void testDateCreatedInOtherTimezonePrintDateTime() throws Exception {
        DateTime parsedDateTimeZonePlusOneJanuary = XSDDateTimeConverter.parseXSDDateTime("2013-01-01+01:00");
        DateTime parsedDateTimeZonePlusOneMay = XSDDateTimeConverter.parseXSDDateTime("2013-05-01+01:00");
        Assert.assertEquals("2013-01-01T00:00:00+01:00", XSDDateTimeConverter.printXSDDateTime(parsedDateTimeZonePlusOneJanuary));
        Assert.assertEquals("2013-05-01T01:00:00+02:00", XSDDateTimeConverter.printXSDDateTime(parsedDateTimeZonePlusOneMay)); // loses the
                                                                                                                               // timezone
    }

    @Test
    public void testDateTimeCreatedInOtherTimezone() throws Exception {
        DateTime parsedDateTimeZonePlusOneJanuary = XSDDateTimeConverter.parseXSDDateTime("2013-01-01T00:00:00+01:00");
        DateTime parsedDateTimeZonePlusOneMay = XSDDateTimeConverter.parseXSDDateTime("2013-05-01T01:00:00+02:00");
        Assert.assertEquals("2013-01-01T00:00:00+01:00", XSDDateTimeConverter.printXSDDateTime(parsedDateTimeZonePlusOneJanuary));
        Assert.assertEquals("2013-05-01T01:00:00+02:00", XSDDateTimeConverter.printXSDDateTime(parsedDateTimeZonePlusOneMay)); // loses the
                                                                                                                               // timezone
    }

    @Test
    public void testDatesCreatedInDifferentTimesOfYear() throws Exception {
        DateTime parsedDateTimeZonePlusOneJanuary = XSDDateTimeConverter.parseXSDDateTime("2013-01-01T00:00:00+01:00");
        DateTime parsedDateTimeZonePlusOneMay = XSDDateTimeConverter.parseXSDDateTime("2013-05-01T01:00:00+01:00");
        DateTime parsedDateTimeZonePlusOneSeptember = XSDDateTimeConverter.parseXSDDateTime("2000-09-09+01:00");
        DateTime parsedDateTimeZonePlusTwoSeptember = XSDDateTimeConverter.parseXSDDateTime("2000-09-09+02:00");
        DateTime parsedDateTimeZonePlusOneDecember = XSDDateTimeConverter.parseXSDDateTime("2000-12-09+01:00");
        Assert.assertEquals("2013-01-01T00:00:00+01:00", XSDDateTimeConverter.printXSDDateTime(parsedDateTimeZonePlusOneJanuary));
        Assert.assertEquals("2013-05-01T02:00:00+02:00", XSDDateTimeConverter.printXSDDateTime(parsedDateTimeZonePlusOneMay));
        Assert.assertEquals("2000-12-09", XSDDateTimeConverter.printXSDDate(parsedDateTimeZonePlusOneDecember)); // loses the timezone
        Assert.assertEquals("2000-09-09", XSDDateTimeConverter.printXSDDate(parsedDateTimeZonePlusOneSeptember)); // loses the timezone
        Assert.assertEquals("2000-09-09", XSDDateTimeConverter.printXSDDate(parsedDateTimeZonePlusTwoSeptember)); // can swith dates if date
                                                                                                                  // is different in current
                                                                                                                  // timezone

    }

    @Test
    public void testParseTimeWithMilliseconds() throws Exception {
        DateTime parsedTime = XSDDateTimeConverter.parseXSDTime("15:15:57.270+02:00");
        Assert.assertEquals(15, parsedTime.getHourOfDay());
        Assert.assertEquals(47757270L, parsedTime.getMillis());
        Assert.assertEquals(15, parsedTime.getMinuteOfHour());
        Assert.assertEquals(57, parsedTime.getSecondOfMinute());
        Assert.assertEquals(270, parsedTime.getMillisOfSecond());

    }

    @Test
    public void testParseTimeWithMilliseconds2() throws Exception {
        DateTime parsedTime = XSDDateTimeConverter.parseXSDTime("15:15:57.2757953248+02:00");
        Assert.assertEquals(15, parsedTime.getHourOfDay());
        Assert.assertEquals(47757275L, parsedTime.getMillis());
        Assert.assertEquals(15, parsedTime.getMinuteOfHour());
        Assert.assertEquals(57, parsedTime.getSecondOfMinute());
        Assert.assertEquals(275, parsedTime.getMillisOfSecond());

    }

    @Test
    public void testPrintDateNull() throws Exception {
        Assert.assertNull(XSDDateTimeConverter.printXSDDate(null));
    }

    @Test
    public void testPrintDateTimeNull() throws Exception {
        Assert.assertNull(XSDDateTimeConverter.printXSDDateTime(null));
    }

    @Test
    public void testPrintTimeNull() throws Exception {
        Assert.assertNull(XSDDateTimeConverter.printXSDTime(null));
    }
    
}
