/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.hub.session;

import java.util.List;

import org.joda.time.DateTime;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import be.ehealth.business.kmehrcommons.helper.ErrorTypeManagement;
import be.ehealth.businessconnector.hub.exception.IntraHubBusinessConnectorException;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.fgov.ehealth.hubservices.core.v1.ConsentType;
import be.fgov.ehealth.hubservices.core.v1.GetTransactionListResponse;
import be.fgov.ehealth.hubservices.core.v1.GetTransactionResponse;
import be.fgov.ehealth.hubservices.core.v1.LocalSearchType;
import be.fgov.ehealth.hubservices.core.v1.PatientIdType;
import be.fgov.ehealth.hubservices.core.v1.PutTransactionResponse;
import be.fgov.ehealth.hubservices.core.v1.RevokeTransactionResponse;
import be.fgov.ehealth.hubservices.core.v1.TherapeuticLinkType;
import be.fgov.ehealth.hubservices.core.v1.TransactionBaseType;
import be.fgov.ehealth.hubservices.core.v1.TransactionIdType;
import be.fgov.ehealth.hubservices.core.v1.TransactionWithPeriodType;
import be.fgov.ehealth.standards.kmehr.schema.v1.ErrorType;
import be.fgov.ehealth.standards.kmehr.schema.v1.FolderType;
import be.fgov.ehealth.standards.kmehr.schema.v1.Kmehrmessage;


/**
 * Transactions operations Test
 * 
 * @author EH054
 */
public class TransactionIntegrationTest extends AbstractIntrahubIntegrationTest {

    private static ConsentType consent;

    private static TherapeuticLinkType therapeuticLink;

    @BeforeClass
    public static void initConsent() throws ConnectorException {
        consent = createConsentType();
        therapeuticLink = createTherapeuticLinkType();

        hubServiceComplete.putPatientConsent(consent);
        hubServiceComplete.putTherapeuticLink(therapeuticLink);
    }

    @AfterClass
    public static void teardownConsent() throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        therapeuticLink.setEnddate(new DateTime());
        hubServiceComplete.revokeTherapeuticLink(therapeuticLink);

        consent.setRevokedate(new DateTime());
        hubServiceComplete.revokePatientConsent(consent);
    }

    /**
     * PutTransaction Test
     */
    @Test
    public void putTransaction() throws Exception {
        PutTransactionResponse putTransaction = hubServiceComplete.putTransaction(createTransactionForPut());

        hubServiceComplete.revokeTransaction(createPatientIdType(), createTransactionId());

        List<ErrorType> errors = putTransaction.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    /**
     * GetTransaction Test
     */
    @Test
    public void getTransaction() throws Exception {
        hubServiceComplete.putTransaction(createTransactionForPut());

        PatientIdType patientId = createPatientIdType();
        GetTransactionResponse transactionResponse = hubServiceComplete.getTransaction(patientId, createTransactionBaseType());
        hubServiceComplete.revokeTransaction(patientId, createTransactionId());

        List<ErrorType> errors = transactionResponse.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    /**
     * GetTransactionList Test
     */
    @Test
    public void getTransactionList() throws Exception {
        hubServiceComplete.putTransaction(createTransactionForPut());

        PatientIdType patientId = createPatientIdType();
        GetTransactionListResponse transactionList = hubServiceComplete.getTransactionList(patientId, LocalSearchType.LOCAL, createTransactionWithPeriodType());
        hubServiceComplete.revokeTransaction(patientId, createTransactionId());

        List<ErrorType> errors = transactionList.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void revokeTransaction() throws Exception {
        hubServiceComplete.putTransaction(createTransactionForPut());

        RevokeTransactionResponse revokeTransaction = hubServiceComplete.revokeTransaction(createPatientIdType(), createTransactionId());

        List<ErrorType> errors = revokeTransaction.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    /*
     * Methods used by the tests
     */

    /**
     * Create a transactionWithPeriod type for transaction operations
     */
    private TransactionWithPeriodType createTransactionWithPeriodType() {
        TransactionWithPeriodType transaction = new TransactionWithPeriodType();
        transaction.setAuthor(createAuthor());
        transaction.setBegindate(new DateTime());
        transaction.setEnddate(new DateTime());
        return transaction;
    }

    /**
     * Create a transaction base type for transaction operations
     */
    private TransactionBaseType createTransactionBaseType() throws Exception {
        TransactionBaseType transaction = new TransactionBaseType();
        transaction.setAuthor(createAuthor());
        transaction.setId(createTransactionId(transactionId));
        return transaction;
    }

    private Kmehrmessage createTransactionForPut() throws Exception {
        Kmehrmessage transaction = new Kmehrmessage();
        transaction.setHeader(createHeader());

        FolderType folder = new FolderType();
        folder.getIds().add(createMessageId(MESSAGE_ID));
        folder.setPatient(createPatientForTransaction());
        folder.getTransactions().add(createTransactionType());
        transaction.getFolders().add(folder);
        return transaction;
    }

    private TransactionIdType createTransactionId() {
        TransactionIdType transid = new TransactionIdType();
        transid.getIds().add(createTransactionId(transactionId));
        return transid;
    }

}
