/*
 * Copyright (c) eHealth
 */
package be.fgov.ehealth.technicalconnector.signature.impl;

import static be.ehealth.technicalconnector.utils.ConnectorIOUtils.getResourceAsByteArray;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.service.sts.security.SAMLToken;
import be.ehealth.technicalconnector.service.sts.security.impl.SAMLHolderOfKeyToken;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.technicalconnector.signature.AdvancedElectronicSignatureEnumeration;
import be.fgov.ehealth.technicalconnector.signature.SignatureBuilderFactory;
import be.fgov.ehealth.technicalconnector.signature.domain.SignatureVerificationError;
import be.fgov.ehealth.technicalconnector.signature.domain.SignatureVerificationResult;
import be.fgov.ehealth.technicalconnector.signature.domain.XadesOption;


/**
 * tests the SignatureBuilder.
 *
 * @author EHP
 */
public class XadesBesSignatureTest extends AbstractSignatureTest {

    private static final Logger LOG = LoggerFactory.getLogger(XadesBesSignatureTest.class);

    @BeforeClass
    public static void initBuilder() throws Exception {
        builder = SignatureBuilderFactory.getSignatureBuilder(AdvancedElectronicSignatureEnumeration.XAdES);
    }

    @Test
    public void testBuildXadesBes() throws Exception {
        byte[] generatedXadesBes = generateXadesBes();
        LOG.debug(new String(generatedXadesBes));
        Assert.assertNotNull(generatedXadesBes);
        Assert.assertTrue(generatedXadesBes.length > 0);
    }

    private byte[] generateXadesBes() throws Exception {
        Map<String, Object> options = new HashMap<String, Object>();
        List<String> tranforms = new ArrayList<String>();
        tranforms.add("http://www.w3.org/2000/09/xmldsig#base64");
        options.put(XadesOption.TRANSFORMERLIST, tranforms);
        options.put(XadesOption.BASEURI, "_82403f5e-8f31-4b6f-a0ef-c76d60cc2f13");
        return builder.sign(credential, textToSign, options);
    }

    @Test
    public void testValidateGeneratedXadesBes() throws Exception {
        verifyXades(generateXadesBes(), textToSign);
    }

    @Test
    public void alteredXadesBes() throws Exception {
        verifyXades(generateXadesBes(), textToSign_bis, SignatureVerificationError.SIGNATURE_COULD_NOT_BE_VERIFIED);
    }

    @Test
    public void testBuildSAMLXades() throws Exception {
        SAMLToken token = new SAMLHolderOfKeyToken(ConnectorXmlUtils.toElement(getResourceAsByteArray(rule.getSessionProperty("test.session.sas"))), credential);
        byte[] prescription = getResourceAsByteArray("/examples/sample-prescription.xml");
        Map<String, Object> options = new HashMap<String, Object>();
        List<String> tranforms = new ArrayList<String>();
        tranforms.add("http://www.w3.org/2001/10/xml-exc-c14n#");
        options.put(XadesOption.TRANSFORMERLIST, tranforms);
        options.put(XadesOption.BASEURI, "_82403f5e-8f31-4b6f-a0ef-c76d60cc2f13");
        options.put(XadesOption.ENCAPSULATE, true);
        byte[] generatedXadesBes = builder.sign(token, prescription, options);
        LOG.debug(new String(generatedXadesBes));

        Map<String, Object> verifyOptions = new HashMap<String, Object>();
        options.put(XadesOption.FOLLOWNESTEDMANIFEST, Boolean.TRUE);
        SignatureVerificationResult result = builder.verify(generatedXadesBes, verifyOptions);
        for (SignatureVerificationError error : result.getErrors()) {
            LOG.debug(error.getErrorName());
        }
        Assert.assertTrue("Xades(T) is not valid.", result.isValid());
    }

    @Override
    String getVerificationTemplate() {
        return "/result/expected/xades.bes.xml";
    }
}
