/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.hubv3.security;

import be.ehealth.businessconnector.hubv3.util.RequestTypeBuilder;
import be.ehealth.businessconnector.test.testcommons.BusinessSessionRule;
import be.ehealth.technicalconnector.config.ConfigFactory;
import be.ehealth.technicalconnector.service.etee.Crypto;
import be.ehealth.technicalconnector.service.etee.domain.EncryptionToken;
import be.ehealth.technicalconnector.utils.SessionUtil;
import be.fgov.ehealth.hubservices.core.v3.RequestType;
import be.fgov.ehealth.standards.kmehr.id.v1.IDHCPARTY;
import be.fgov.ehealth.standards.kmehr.id.v1.IDHCPARTYschemes;
import be.fgov.ehealth.standards.kmehr.schema.v1.HcpartyType;
import org.apache.commons.codec.binary.Base64;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.TestRule;

import javax.xml.bind.JAXB;
import java.util.List;

import static org.junit.Assert.fail;


/**
 * Integration tests for encrypted requests.
 *
 * @author EHP
 */
public class HubRequestWithEncryptionIntegrationTest {

    @ClassRule
    public static TestRule sessionPrerequisite = BusinessSessionRule.withActiveSession("/be.ehealth.businessconnector.intrahubv3.test.properties");

    @Test
    public void testRequestBuilderWithoutEncryptionKey() throws Exception {

        TestResult testResult = testRequestBuilder();
        Assert.assertTrue(testResult.isContainsEncryption());
        Assert.assertFalse(testResult.isContainsEncryptionKey());
    }

    @Test
    public void testRequestBuilderWithEncryptionKey() throws Exception {
        ConfigFactory.getConfigValidator().getConfig().setProperty("kmehr.hubservicev3.identifier.id.idencryptionkey", "true");

        TestResult testResult = testRequestBuilder();
        Assert.assertTrue(testResult.isContainsEncryption());
        Assert.assertTrue(testResult.isContainsEncryptionKey());
    }

    private TestResult testRequestBuilder() throws Exception {
        RequestType req = RequestTypeBuilder.init().addAuthorWithEncryptionInformation().build();
        List<HcpartyType> partiesList = req.getAuthor().getHcparties();
        boolean containsEncryption = false;
        boolean containsEncryptionKey = false;
        for (HcpartyType hcpartyType : partiesList) {
            if (hcpartyType.getFirstname() != null) {
                for (IDHCPARTY hcparty : hcpartyType.getIds()) {
                    if (hcparty.getS().compareTo(IDHCPARTYschemes.ID_ENCRYPTION_ACTOR) == 0) {
                        if (containsEncryption) {
                            fail("There should be only one ID_ENCRYPTION_ACTOR");
                        }
                        containsEncryption = true;
                    } else if (hcparty.getS().compareTo(IDHCPARTYschemes.ID_ENCRYPTION_KEY) == 0) {
                        if (containsEncryptionKey) {
                            fail("There should be only one ID_ENCRYPTION_KEY");
                        }
                        containsEncryptionKey = true;
                        Assert.assertTrue("Encryption key mustbe base64 encoded", Base64.isBase64(hcparty.getValue()));
                        byte[] encryptedMessage = SessionUtil.getEncryptionCrypto().seal(Crypto.SigningPolicySelector.WITH_NON_REPUDIATION, new EncryptionToken(hcparty.getValue()), "string to seal".getBytes());
                        byte[] decryptedMessage = SessionUtil.getEncryptionCrypto().unseal(Crypto.SigningPolicySelector.WITH_NON_REPUDIATION, encryptedMessage).getContentAsByte();
                        Assert.assertEquals("string to seal", new String(decryptedMessage));
                    }
                }
            }
        }

        JAXB.marshal(req, System.out);

        return  new TestResult(containsEncryption, containsEncryptionKey);
    }


    private static class TestResult {
        boolean containsEncryption;
        boolean containsEncryptionKey;

        private TestResult(boolean containsEncryption, boolean containsEncryptionKey) {
            this.containsEncryption = containsEncryption;
            this.containsEncryptionKey = containsEncryptionKey;
        }

        public boolean isContainsEncryption() {
            return containsEncryption;
        }

        public boolean isContainsEncryptionKey() {
            return containsEncryptionKey;
        }
    }
}
