/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.decrypt;

import be.fgov.ehealth.etee.crypto.decrypt.SignerInfoAttributes;
import be.fgov.ehealth.etee.crypto.decrypt.SigningTimeVerifierResult;
import be.fgov.ehealth.etee.crypto.decrypt.TimeStampTrustResult;
import be.fgov.ehealth.etee.crypto.decrypt.TimeStampTrustService;
import be.fgov.ehealth.etee.crypto.policies.SigningOption;
import be.fgov.ehealth.etee.crypto.policies.SigningOptions;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.status.NotificationWarning;
import be.fgov.ehealth.etee.crypto.utils.TimeFrameValidator;
import java.io.ByteArrayInputStream;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import org.bouncycastle.asn1.x509.CertificateList;
import org.bouncycastle.tsp.TimeStampToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SigningTimeVerifier {
    private static final Logger LOGGER = LoggerFactory.getLogger(SigningTimeVerifier.class);
    private TimeFrameValidator timeFrameValidator;
    private final int signingTimeExpiration;
    private final TimeStampTrustService timeStampTrustService;

    public SigningTimeVerifier() {
        this(SigningOptions.defaultOptions(), null);
    }

    public SigningTimeVerifier(SigningOptions signingOptions, TimeStampTrustService timeStampTrustService) {
        Long clockSkew = signingOptions.getLong(SigningOption.CLOCK_SKEW);
        this.signingTimeExpiration = signingOptions.getInteger(SigningOption.SIGNING_TIME_EXPIRATION);
        boolean signingTimeTrustImplicit = signingOptions.getBoolean(SigningOption.SIGNING_TIME_TRUST_IMPLICIT);
        if (!signingTimeTrustImplicit) {
            this.timeFrameValidator = TimeFrameValidator.create(clockSkew);
        }
        this.timeStampTrustService = timeStampTrustService;
    }

    SigningTimeVerifierResult verify(SignerInfoAttributes signerInfoAttributes) {
        Date signingTime = signerInfoAttributes.getSigningTime();
        if (signingTime == null) {
            LOGGER.debug("No SignerInfo Attribute found with Signing-Time");
            return this.handleSigningTimeMissing(signerInfoAttributes);
        }
        Date now = new Date();
        Date expirationTime = this.getExpiration(this.signingTimeExpiration, now);
        if (!this.validateExpirationTime(expirationTime, signingTime, now)) {
            LOGGER.debug("SignerInfo Attribute found with expired Signing-Time: " + signingTime);
            return this.handleSigningTimeExpired(signerInfoAttributes, signingTime, now);
        }
        LOGGER.debug("Message Signing-Time accepted as is: " + signingTime);
        return new SigningTimeVerifierResult(signingTime);
    }

    private SigningTimeVerifierResult handleSigningTimeMissing(SignerInfoAttributes signerInfoAttributes) {
        if (signerInfoAttributes.getSignatureTimestamps().isEmpty()) {
            return this.buildResult(null, NotificationWarning.SIGNING_TIME_MISSING, "No SignerInfo Attribute found with Signature-TimeStampToken");
        }
        if (!this.supportsTimeStamps()) {
            return this.buildResult(null, NotificationWarning.SIGNING_TIME_MISSING, "Trust for Signature-TimeStampTokens is disabled.");
        }
        TimeStampToken timeStampToken = signerInfoAttributes.getSignatureTimestamps().get(0);
        TimeStampTrustResult tsResult = this.timeStampTrustService.verify(timeStampToken, this.getX509CRL(signerInfoAttributes.getCrls()));
        if (tsResult.hasErrors()) {
            return this.buildResult(null, tsResult.getErrors(), "TimeStampToken validation completed with errors");
        }
        Date signingTime = timeStampToken.getTimeStampInfo().getGenTime();
        LOGGER.debug("Found a valid Signature-TimeStampToken. Using generated time of timestamp as Signing-Time: " + signingTime);
        SigningTimeVerifierResult result = new SigningTimeVerifierResult(signingTime);
        if (tsResult.hasWarnings()) {
            LOGGER.error("TimeStampToken validation completed with warnings: " + tsResult.getWarnings());
            result.getWarnings().addAll(tsResult.getWarnings());
        }
        return result;
    }

    private SigningTimeVerifierResult handleSigningTimeExpired(SignerInfoAttributes signerInfoAttributes, Date signingTime, Date now) {
        if (signerInfoAttributes.getSignatureTimestamps().isEmpty()) {
            return this.buildResult(signingTime, NotificationWarning.SIGNING_TIME_EXPIRED, "Signing-Time is expired " + signingTime + " with respect to the current time and no Signature-TimeStampToken found - " + now);
        }
        if (!this.supportsTimeStamps()) {
            return this.buildResult(signingTime, NotificationWarning.SIGNING_TIME_EXPIRED, "Trust for Signature-TimeStampTokens is disabled.");
        }
        TimeStampToken timeStampToken = signerInfoAttributes.getSignatureTimestamps().get(0);
        TimeStampTrustResult tsResult = this.timeStampTrustService.verify(timeStampToken, this.getX509CRL(signerInfoAttributes.getCrls()));
        if (tsResult.hasErrors()) {
            return this.buildResult(signingTime, tsResult.getErrors(), "TimeStampToken validation completed with errors");
        }
        Date timeStampGenerationTime = timeStampToken.getTimeStampInfo().getGenTime();
        Date expirationTime = this.getExpiration(this.signingTimeExpiration, timeStampGenerationTime);
        if (!this.validateExpirationTime(expirationTime, signingTime, timeStampGenerationTime)) {
            return this.buildResult(signingTime, NotificationWarning.SIGNING_TIME_EXPIRED, "Signing-Time is expired " + signingTime + " with respect to the Signature-TimeStamp - " + timeStampGenerationTime);
        }
        LOGGER.debug("Found a valid Signature-TimeStampToken close to the Signing-Time. Signing-Time accepted: " + signingTime);
        SigningTimeVerifierResult result = new SigningTimeVerifierResult(signingTime);
        if (tsResult.hasWarnings()) {
            LOGGER.error("TimeStampToken validation completed with warnings: " + tsResult.getWarnings());
            result.getWarnings().addAll(tsResult.getWarnings());
        }
        return result;
    }

    private SigningTimeVerifierResult buildResult(Date signingTime, NotificationWarning warning, String msg) {
        LOGGER.warn(msg);
        SigningTimeVerifierResult result = new SigningTimeVerifierResult(signingTime);
        result.getWarnings().add(warning);
        return result;
    }

    private SigningTimeVerifierResult buildResult(Date signingTime, List<NotificationError> errors, String msg) {
        LOGGER.error(msg + ": " + errors);
        SigningTimeVerifierResult result = new SigningTimeVerifierResult(signingTime);
        result.getErrors().addAll(errors);
        return result;
    }

    private boolean supportsTimeStamps() {
        return this.timeStampTrustService != null;
    }

    private boolean validateExpirationTime(Date expiration, Date signingTime, Date now) {
        if (this.timeFrameValidator == null) {
            return true;
        }
        return this.timeFrameValidator.validate(expiration, signingTime, now);
    }

    private Date getExpiration(int max, Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.add(12, -max);
        return c.getTime();
    }

    private List<X509CRL> getX509CRL(List<CertificateList> crls) {
        ArrayList<X509CRL> x509Crls = new ArrayList<X509CRL>();
        for (CertificateList crl : crls) {
            try {
                ByteArrayInputStream inStream = new ByteArrayInputStream(crl.getEncoded());
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                x509Crls.add((X509CRL)cf.generateCRL(inStream));
                inStream.close();
            }
            catch (Exception e) {
                LOGGER.warn("Embedded CRL could not be encoded. It will be skipped.", (Throwable)e);
            }
        }
        return x509Crls;
    }
}

