/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.ehbox.v3.session;

import be.ehealth.businessconnector.ehbox.api.domain.Document;
import be.ehealth.businessconnector.ehbox.v3.session.ServiceFactory;
import be.ehealth.businessconnector.test.ehbox.v3.util.EhboxTestDataBuilder;
import be.ehealth.businessconnector.test.ehbox.v3.util.MessageBuilder;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetMessageAcknowledgmentsStatusResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.MoveMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.Row;
import be.fgov.ehealth.ehbox.core.v3.BoxIdType;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.MethodSorters;

import be.ehealth.businessconnector.ehbox.api.domain.Addressee;
import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.utils.QualityType;
import be.ehealth.businessconnector.test.ehbox.v3.util.EhboxTestUtilities;
import be.ehealth.businessconnector.ehbox.v3.builders.BuilderFactory;
import be.ehealth.technicalconnector.utils.IdentifierType;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetFullMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.Message;
import be.fgov.ehealth.ehbox.publication.protocol.v3.SendMessageRequest;
import be.fgov.ehealth.ehbox.publication.protocol.v3.SendMessageResponse;
import org.junit.runners.Parameterized;

import java.util.Arrays;
import java.util.Collection;

import static be.ehealth.technicalconnector.utils.ConnectorIOUtils.getBytes;


/**
 * @author EHP
 */
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
@RunWith(Parameterized.class)
public class EhboxV3WalkthroughIntegrationTest {

    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persphysician").forUser("olivier").build();

    private static final String MESSAGE_TITLE = "Connector integration test";
    private static final String MESSAGE_CONTENT = "message test connector integration test with special characters {}()6§7^2²³ü*µ~`ù`´´][|çà&";
    private static final String MESSAGE_ANNEX_TITLE = "Annex.pdf";
    private static final String MESSAGE_ANNEX_MIME_TYPE = "application/pdf";
    private static final String MESSAGE_ANNEX_CONTENT = "/examples/test.pdf";
    private static final String MESSAGE_FREE_TEXT = "Free information test integration test";
    private static final String MESSAGE_CUSTOM_META_KEY = "CustomMeta";
    private static final String MESSAGE_CUSTOM_META_VALUE = "[Connector]";
    private static final String EXPECTED_RESPONSE_CODE = "100";
    private static final String INBOX = "INBOX";
    private static final String BININBOX = "BININBOX";
    private static final String SENTBOX = "SENTBOX";

    private static String currentMessageId;

    private String destinationId;
    private IdentifierType destinationType;
    private QualityType destinationQuality;
    private String mimeType;
    private MessageEncryption messageEncryption;

    public EhboxV3WalkthroughIntegrationTest(String destinationId, IdentifierType destinationType,
                                             QualityType destinationQuality, String mimeType,
                                             MessageEncryption messageEncryption) {
        this.destinationId = destinationId;
        this.destinationType = destinationType;
        this.destinationQuality = destinationQuality;
        this.messageEncryption = messageEncryption;
        this.mimeType = mimeType;
    }

    @Parameterized.Parameters(name = "{index}: {0},{1},{2},{3},{4}")
    public static Collection testCasesParams() {
        return Arrays.asList(new Object[][]{
                {"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/plain", MessageEncryption.NOT_ENCRYPTED},
                {"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/plain", MessageEncryption.ENCRYPTED},
                {"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/plain", MessageEncryption.NOT_ENCRYPTED},
                {"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/plain", MessageEncryption.ENCRYPTED},
                {"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/html", MessageEncryption.NOT_ENCRYPTED},
                {"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/html", MessageEncryption.ENCRYPTED},
                {"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/html", MessageEncryption.NOT_ENCRYPTED},
                {"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/html", MessageEncryption.ENCRYPTED}
        });
    }

    @Test
    public void _01_sendMessage() throws Exception {
        final Addressee destination = EhboxTestDataBuilder.generateDestination(destinationId, destinationType, destinationQuality);

        DocumentMessage<Message> message = new DocumentMessage<Message>();
        Document document = new Document();
        document.setFilename("DownloadFileName.txt ");
        document.setMimeType(mimeType);
        document.setTitle(MESSAGE_TITLE);
        document.setContent(MESSAGE_CONTENT.getBytes());
        message.setBody(document);
        message.getDestinations().add(destination);
        message.setEncrypted(messageEncryption.isEncrypted());
        message.setImportant(true);
        message.setUseReceivedReceipt(true);
        message.setUsePublicationReceipt(true);
        message.setUseReadReceipt(true);

        message.getAnnexList().add(EhboxTestDataBuilder
                .generateDocument(MESSAGE_ANNEX_TITLE, MESSAGE_ANNEX_MIME_TYPE, getBytes(ConnectorIOUtils.getResourceAsStream(MESSAGE_ANNEX_CONTENT))));

        message.setFreeText(MESSAGE_FREE_TEXT);

        message.getCustomMetas().put(MESSAGE_CUSTOM_META_KEY, MESSAGE_CUSTOM_META_VALUE);

        SendMessageRequest request = BuilderFactory.getSendMessageBuilder().buildMessage(message);
        SendMessageResponse response = ServiceFactory.getEhealthBoxServiceV3().sendMessage(request);
        currentMessageId = response.getId();

        Assert.assertEquals(EXPECTED_RESPONSE_CODE, response.getStatus().getCode());
        Assert.assertNotNull("Message ID should not be null", currentMessageId);

        EhboxTestUtilities.waitForProcessing(1500);
    }

    @Test
    public void _02_getMessageAcknowledgements() throws Exception {
        GetMessageAcknowledgmentsStatusResponse acks = EhboxTestUtilities.getMessageAcknowledgements(currentMessageId, EhboxTestDataBuilder.getEhboxSender());

        for (Row currentAcknowledgementsStatus : acks.getAcknowledgmentsStatus().getRows()) {
            Assert.assertNull("The message should not have been read yet", currentAcknowledgementsStatus.getRead());
            Assert.assertNull("The message should not have been received", currentAcknowledgementsStatus.getReceived());
        }
    }

    @Test
    public void _03_getMessageList() throws Exception {
        Assert.assertTrue("The message with id " + currentMessageId + " should be in the list", EhboxTestUtilities.isMessageInList(currentMessageId, INBOX,
                boxId()));
    }

    @Test
    public void _04_getAllEhboxesMessagesList() throws Exception {
        Assert.assertTrue("The message with id " + currentMessageId + " should be in the list", EhboxTestUtilities.isMessageInAllEhBoxesList(currentMessageId, INBOX));
    }

    @Test
    public void _05_getFullMessage() throws Exception {
        GetFullMessageResponse fullMessage = EhboxTestUtilities.getFullMessage(currentMessageId, INBOX,
                boxId());

        be.ehealth.businessconnector.ehbox.api.domain.Message<GetFullMessageResponse> retrievedFullMessage = BuilderFactory.getConsultationMessageBuilder().buildFullMessage(fullMessage);

        DocumentMessage<GetFullMessageResponse> inboxMessage = (DocumentMessage<GetFullMessageResponse>) retrievedFullMessage;

        String messageTitle = inboxMessage.getDocumentTitle();
        Assert.assertEquals("Message title should be \"" + MESSAGE_TITLE + "\" but was \"" + messageTitle + "\"", MESSAGE_TITLE, messageTitle);
        Assert.assertEquals(MESSAGE_CONTENT, new String(inboxMessage.getBody().getContent(), "UTF-8"));
        Assert.assertEquals(mimeType, inboxMessage.getOriginal().getMessage().getContentContext().getContent().getDocument().getMimeType());
        Assert.assertEquals("The message should " + (messageEncryption.isEncrypted() ? "be encrypted" : "not be encryted"), messageEncryption.isEncrypted(), inboxMessage.isEncrypted());
        Assert.assertTrue("The message should be important", inboxMessage.isImportant());
        Assert.assertTrue("The message should have an annex", inboxMessage.isHasAnnex());
        Assert.assertEquals("The message should have 1 annex", 1, inboxMessage.getAnnexList().size());
        Assert.assertTrue("The message should have some free information", inboxMessage.isHasFreeInformations());
        Assert.assertEquals("The message should have a custom metadata", 1, inboxMessage.getCustomMetas().size());

        String annexTitle = inboxMessage.getAnnexList().get(0).getTitle();
        Assert.assertEquals("The message annex title should be \"" + MESSAGE_ANNEX_TITLE + "\" but was \"" + annexTitle + "\"", MESSAGE_ANNEX_TITLE, annexTitle);
        Assert.assertEquals(MESSAGE_ANNEX_MIME_TYPE, inboxMessage.getAnnexList().get(0).getMimeType());

        String customMeta = inboxMessage.getCustomMetas().get(MESSAGE_CUSTOM_META_KEY);
        Assert.assertEquals("The message custom metadata should be \"" + MESSAGE_CUSTOM_META_VALUE + "\" but was \"" + customMeta + "\"", MESSAGE_CUSTOM_META_VALUE, customMeta);
    }

    @Test
    public void _06_moveMessage() throws Exception {
        MoveMessageResponse response = ServiceFactory.getEhealthBoxServiceV3().moveMessage(BuilderFactory.getRequestBuilder().createMoveMessageRequest(INBOX, BININBOX,
                boxId(), currentMessageId));
        Assert.assertEquals("The message should have been moved", EXPECTED_RESPONSE_CODE, response.getStatus().getCode());
    }

    @Test
    public void _07_deleteMessage() throws Exception {
        EhboxTestUtilities.deleteMessages(BININBOX, boxId(), currentMessageId);
        EhboxTestUtilities.deleteMessages(SENTBOX, EhboxTestDataBuilder.getEhboxSender(), currentMessageId);
    }

    private BoxIdType boxId() {
        return EhboxTestDataBuilder.generateBoxId(destinationId, destinationType.getType(IdentifierType.EHBOX), destinationQuality.getQuality());
    }

    private enum MessageEncryption {
        ENCRYPTED, NOT_ENCRYPTED;

        boolean isEncrypted() {
            return this.equals(ENCRYPTED);
        }
    }

}
