package be.ehealth.businessconnector.hub.session;

import java.util.List;
import be.ehealth.businessconnector.testcommons.BusinessSessionRule;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.rules.TestRule;
import org.junit.runners.MethodSorters;

import be.ehealth.business.kmehrcommons.helper.ErrorTypeManagement;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.fgov.ehealth.hubservices.core.v1.Consent;
import be.fgov.ehealth.hubservices.core.v1.ConsentHCPartyType;
import be.fgov.ehealth.hubservices.core.v1.ConsentType;
import be.fgov.ehealth.hubservices.core.v1.GetPatientConsentResponse;
import be.fgov.ehealth.hubservices.core.v1.GetTherapeuticLinkResponse;
import be.fgov.ehealth.hubservices.core.v1.GetTransactionListResponse;
import be.fgov.ehealth.hubservices.core.v1.GetTransactionResponse;
import be.fgov.ehealth.hubservices.core.v1.LocalSearchType;
import be.fgov.ehealth.hubservices.core.v1.PatientIdType;
import be.fgov.ehealth.hubservices.core.v1.PutHCPartyConsentResponse;
import be.fgov.ehealth.hubservices.core.v1.PutPatientConsentResponse;
import be.fgov.ehealth.hubservices.core.v1.PutPatientResponse;
import be.fgov.ehealth.hubservices.core.v1.PutTherapeuticLinkResponse;
import be.fgov.ehealth.hubservices.core.v1.PutTransactionResponse;
import be.fgov.ehealth.hubservices.core.v1.RevokePatientConsentResponse;
import be.fgov.ehealth.hubservices.core.v1.RevokeTherapeuticLinkResponse;
import be.fgov.ehealth.hubservices.core.v1.RevokeTransactionResponse;
import be.fgov.ehealth.hubservices.core.v1.SelectGetHCPartyPatientConsentType;
import be.fgov.ehealth.hubservices.core.v1.SelectGetPatientConsentType;
import be.fgov.ehealth.hubservices.core.v1.TherapeuticLinkType;
import be.fgov.ehealth.hubservices.core.v1.TransactionBaseType;
import be.fgov.ehealth.hubservices.core.v1.TransactionIdType;
import be.fgov.ehealth.hubservices.core.v1.TransactionWithPeriodType;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENT;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENTschemes;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENTvalues;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDTHERAPEUTICLINK;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDTHERAPEUTICLINKschemes;
import be.fgov.ehealth.standards.kmehr.schema.v1.ErrorType;
import be.fgov.ehealth.standards.kmehr.schema.v1.FolderType;
import be.fgov.ehealth.standards.kmehr.schema.v1.Kmehrmessage;
import be.fgov.ehealth.standards.kmehr.schema.v1.PersonType;

@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class WalkThroughIntegrationTest {

    @ClassRule
    public static TestRule sessionPrerequisite = BusinessSessionRule.withActiveSession("/be.ehealth.businessconnector.intrahub.test.properties");

    @ClassRule
    public static InitHubRule hubPrerequesite = new InitHubRule();

    private static String transactionId;

    private HubServiceComplete hubServiceComplete = hubPrerequesite.getHubServiceComplete();
    

    @Test
    public void _01_putHcPartyConsent() throws Exception {
        ConsentHCPartyType consent = new ConsentHCPartyType();
        consent.setAuthor(HubTestHelper.createAuthor());
        consent.setHcparty(HubTestHelper.createHcPartyIdProfessional());
        consent.setRevokedate(new DateTime());
        consent.setSigndate(new DateTime());
        PutHCPartyConsentResponse putHCPartyConsent = hubServiceComplete.putHCPartyConsent(consent);

        List<ErrorType> errors = putHCPartyConsent.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty() || StringUtils.contains(errors.get(0).getCds().get(0).getValue(), "PERHCP.CONSENT.EXISTS"));
    }

    @Test
    public void _02_putPatient() throws Exception {
        PersonType person = HubTestHelper.createPatient();
        PutPatientResponse putPatient = hubServiceComplete.putPatient(person);

        List<ErrorType> errors = putPatient.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _03_putPatientConsent() throws Exception {
        ConsentType consent = HubTestHelper.createConsentType();
        PutPatientConsentResponse putPatientConsent = hubServiceComplete.putPatientConsent(consent);

        List<ErrorType> errors = putPatientConsent.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _04_putTherapeuticLink() throws Exception {
        TherapeuticLinkType therapeuticLink = HubTestHelper.createTherapeuticLinkType();
        PutTherapeuticLinkResponse putTherapeuticLink = hubServiceComplete.putTherapeuticLink(therapeuticLink);

        List<ErrorType> errors = putTherapeuticLink.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _05_putTransaction() throws Exception {
        Kmehrmessage kmehrmessage = createTransactionForPut();
        Assert.assertEquals("number of request folders", 1, kmehrmessage.getFolders().size());
        Assert.assertEquals("number of request transactions",1, kmehrmessage.getFolders().get(0).getTransactions().size());
        Assert.assertEquals("number of request transaction ids", 3, kmehrmessage.getFolders().get(0).getTransactions().get(0).getIds().size());
        ConnectorXmlUtils.dump(kmehrmessage);
        PutTransactionResponse putTransaction = hubServiceComplete.putTransaction(kmehrmessage);
        transactionId = putTransaction.getTransaction().getIds().get(0).getValue();

        List<ErrorType> errors = putTransaction.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _06_getTherapeuticLink() throws Exception {
        GetTherapeuticLinkResponse getResponse = hubServiceComplete.getTherapeuticLink(createSelectGetHCPartyPatientConsentType());

        List<ErrorType> errors = getResponse.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _07_getTransactionList() throws Exception {
        PatientIdType patientId = HubTestHelper.createPatientIdType();
        GetTransactionListResponse transactionList = hubServiceComplete.getTransactionList(patientId, LocalSearchType.LOCAL, createTransactionWithPeriodType());

        List<ErrorType> errors = transactionList.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _08_getPatientConsent() throws ConnectorException {
        GetPatientConsentResponse patientConsent = hubServiceComplete.getPatientConsent(createSelectGetPatientConsentType());

        List<ErrorType> errors = patientConsent.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _09_getTransaction() throws Exception {
        PatientIdType patientId = HubTestHelper.createPatientIdType();
        GetTransactionResponse transactionResponse = hubServiceComplete.getTransaction(patientId, createTransactionBaseType());

        List<ErrorType> errors = transactionResponse.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _10_revokeTransaction() throws Exception {
        RevokeTransactionResponse revokeTransaction = hubServiceComplete.revokeTransaction(HubTestHelper.createPatientIdType(), createTransactionId());

        List<ErrorType> errors = revokeTransaction.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _11_revokeTherapeuticLink() throws Exception {
        TherapeuticLinkType therapeuticLink = HubTestHelper.createTherapeuticLinkType();
        therapeuticLink.setEnddate(new DateTime());
        RevokeTherapeuticLinkResponse revokeTherapeuticLink = hubServiceComplete.revokeTherapeuticLink(therapeuticLink);

        List<ErrorType> errors = revokeTherapeuticLink.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void _12_revokePatientConsent() throws Exception {
        ConsentType consent = HubTestHelper.createConsentType();
        consent.setRevokedate(new DateTime());
        RevokePatientConsentResponse revokePatientConsent = hubServiceComplete.revokePatientConsent(consent);

        List<ErrorType> errors = revokePatientConsent.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }


    // HELPER METHODS ------------------------------------------

    /**
     * Create a selectGetPatientConsentType used for consent operations
     */
    private SelectGetPatientConsentType createSelectGetPatientConsentType() {
        SelectGetPatientConsentType patientConsent = new SelectGetPatientConsentType();

        Consent selectConsent = new Consent();
        CDCONSENT cdConsent = new CDCONSENT();
        cdConsent.setS(CDCONSENTschemes.CD_CONSENTTYPE);
        cdConsent.setSV("1.0");
        cdConsent.setValue(CDCONSENTvalues.RETROSPECTIVE);
        selectConsent.getCds().add(cdConsent);
        patientConsent.setConsent(selectConsent);

        patientConsent.setPatient(HubTestHelper.createPatientIdType());
        return patientConsent;
    }

    /**
     * Create a SelectGetHCPartyPatientConsentType used for ther link operations
     */
    private SelectGetHCPartyPatientConsentType createSelectGetHCPartyPatientConsentType() throws TechnicalConnectorException {
        SelectGetHCPartyPatientConsentType selectConsent = new SelectGetHCPartyPatientConsentType();

        CDTHERAPEUTICLINK cdTherLink = new CDTHERAPEUTICLINK();
        cdTherLink.setS(CDTHERAPEUTICLINKschemes.CD_THERAPEUTICLINKTYPE);
        cdTherLink.setSV(HubConfig.SV_CD);
        cdTherLink.setValue("gpconsultation");
        selectConsent.getCds().add(cdTherLink);

        selectConsent.getPatientsAndHcparties().add(HubTestHelper.createPatientIdType());
        selectConsent.getPatientsAndHcparties().add(HubTestHelper.createHcPartyIdProfessional());
        selectConsent.setBegindate(new DateTime());
        selectConsent.setEnddate(new DateTime());

        return selectConsent;
    }

    private Kmehrmessage createTransactionForPut() throws Exception {
        Kmehrmessage transaction = new Kmehrmessage();
        transaction.setHeader(HubTestHelper.createHeader());

        FolderType folder = new FolderType();
        folder.getIds().add(HubTestHelper.createMessageId(HubConfig.MESSAGE_ID));
        folder.setPatient(HubTestHelper.createPatientForTransaction());
        folder.getTransactions().add(HubTestHelper.createTransactionType());
        transaction.getFolders().add(folder);
        return transaction;
    }

    private TransactionIdType createTransactionId() {
        TransactionIdType transid = new TransactionIdType();
        transid.getIds().add(HubTestHelper.createTransactionId(transactionId));
        return transid;
    }

    /**
     * Create a transactionWithPeriod type for transaction operations
     */
    private TransactionWithPeriodType createTransactionWithPeriodType() {
        TransactionWithPeriodType transaction = new TransactionWithPeriodType();
        transaction.setAuthor(HubTestHelper.createAuthor());
        transaction.setBegindate(new DateTime());
        transaction.setEnddate(new DateTime());
        return transaction;
    }

    /**
     * Create a transaction base type for transaction operations
     */
    private TransactionBaseType createTransactionBaseType() throws Exception {
        TransactionBaseType transaction = new TransactionBaseType();
        transaction.setAuthor(HubTestHelper.createAuthor());
        transaction.setId(HubTestHelper.createTransactionId(transactionId));
        return transaction;
    }
}
