/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.ehbox.v3.builders.impl;

import java.io.IOException;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.businessconnector.ehbox.api.domain.Addressee;
import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.domain.exception.EhboxBusinessConnectorException;
import be.ehealth.businessconnector.ehbox.v3.builders.BuilderFactory;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.ConsultationContentType;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.ConsultationDocumentType;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.ConsultationMessageType;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.ContentContextType;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.DestinationContextType;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetFullMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetMessagesListResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.Message;
import be.fgov.ehealth.ehbox.core.v3.ContentSpecificationType;
import be.fgov.ehealth.ehbox.core.v3.EhboxIdentifierType;
import be.fgov.ehealth.ehbox.core.v3.FreeInformationsType;
import be.fgov.ehealth.ehbox.core.v3.SenderType;
import be.fgov.ehealth.ehbox.core.v3.User;


/**
 * Test class for {@link ConsultationMessageBuilderImpl}.
 * 
 * @author EHP
 * 
 */
public class ConsultationFullMessageBuilderTest {

    private static final Logger LOG = LoggerFactory.getLogger(ConsultationFullMessageBuilderTest.class);


    private static final String ID = "1";


    private static final String INSS_PATIENT = "INSSPatient";


    private static final String FREETEXT = "@zerty/1234 90%£*=¨~~234";


    private static final String MIME_TYPE = "mime/type";


    private static final String TITLE = "TITLE";


    private static final String FILENAME = "FILENAME";


    private static final String QUALITY = "quality";


    private static final String SUBTYPE = "RETIREMENT";


    private static final String TYPE = "NIHII";


    private static final String LASTNAME = "lastname";

    private static final String FIRSTNAME = "firstname";

    private ConsultationFullMessageBuilder builder;

    /**
     * @throws java.lang.Exception
     */
    @Before
    public void setUp() throws Exception {
        builder = new ConsultationFullMessageBuilder();

    }

    @Test
    public void testIssueEhconext169() throws Exception {
        byte[] bytes = ConnectorIOUtils.getBytes(ConnectorIOUtils.getResourceAsStream("/examples/getMessageListResponse.xml"));
        MarshallerHelper<GetMessagesListResponse, GetMessagesListResponse> helper = new MarshallerHelper<GetMessagesListResponse, GetMessagesListResponse>(GetMessagesListResponse.class, GetMessagesListResponse.class);
        GetMessagesListResponse response = helper.toObject(bytes);
        Assert.assertNotNull(response);
        Assert.assertNotNull(response.getMessages());
        Assert.assertEquals(18, response.getMessages().size());
        for (Message message : response.getMessages()) {
            boolean isEncrypted = message.getContentSpecification().isIsEncrypted();
            LOG.debug("message (unprocessed)" + message.getMessageId() + " encrypted : " + isEncrypted);
            be.ehealth.businessconnector.ehbox.api.domain.Message<Message> buildMessage = BuilderFactory.getConsultationMessageBuilder().buildMessage(message);
            boolean encrypted = buildMessage.isEncrypted();
            LOG.debug("after convertiong to  ehbox.api.domain.Message: id = " + buildMessage.getId() + ", isEncrypted = " + encrypted);
            Assert.assertEquals("both booleans should be the same", isEncrypted, encrypted);
        }


    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehbox.v3.builders.impl.ConsultationMessageBuilderImpl#buildFullMessage(be.fgov.ehealth.ehbox.consultation.protocol.v3.GetFullMessageResponse)}
     * .
     * 
     * @throws IOException
     * @throws TechnicalConnectorException
     * @throws EhboxBusinessConnectorException
     */
    @Test
    public void testBuildFullMessage() throws IOException, EhboxBusinessConnectorException, TechnicalConnectorException {


        GetFullMessageResponse response = new GetFullMessageResponse();

        ConsultationMessageType message = new ConsultationMessageType();
        DestinationContextType destination = new DestinationContextType();
        destination.setId(ID);
        destination.setType(TYPE);
        destination.setSubType(SUBTYPE);
        destination.setQuality(QUALITY);
        message.getDestinationContexts().add(destination);

        ContentContextType contentContext = new ContentContextType();

        // build content
        contentContext.setContent(new ConsultationContentType());

        contentContext.getContent().setDocument(new ConsultationDocumentType());
        contentContext.getContent().getDocument().setDownloadFileName(FILENAME);
        contentContext.getContent().getDocument().setTitle(TITLE);
        contentContext.getContent().getDocument().setMimeType(MIME_TYPE);
        contentContext.getContent().setFreeInformations(new FreeInformationsType());
        contentContext.getContent().getFreeInformations().setEncryptableFreeText(FREETEXT.getBytes(be.ehealth.technicalconnector.enumeration.Charset.UTF_8.getName()));


        contentContext.getContent().setEncryptableINSSPatient(INSS_PATIENT.getBytes());

        // build contentSpecification
        contentContext.setContentSpecification(new ContentSpecificationType());
        contentContext.getContentSpecification().setIsEncrypted(false);
        contentContext.getContentSpecification().setIsImportant(true);
        contentContext.getContentSpecification().setContentType("DOCUMENT");

        // build sender
        response.setSender(new SenderType());
        response.getSender().setFirstName(FIRSTNAME);
        response.getSender().setId(ID);
        response.getSender().setName(LASTNAME);
        response.getSender().setQuality(QUALITY);
        response.getSender().setSubType(SUBTYPE);
        response.getSender().setType(TYPE);

        message.setContentContext(contentContext);
        response.setMessage(message);

        DocumentMessage<?> result = (DocumentMessage<?>) builder.buildFullMessage(response);

        Assert.assertNotNull(result);
        Assert.assertFalse(result.isEncrypted());
        Assert.assertTrue(result.isImportant());
        Assert.assertEquals(TITLE, result.getDocumentTitle());
        Assert.assertEquals(FREETEXT, result.getFreeText());

        Assert.assertNotNull(result.getDestinations().get(0));
        Assert.assertNotNull(result.getSender());
        Assert.assertNotNull(result.getPatientInss());

    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuildAddresseEmptySenderType() throws Exception {
        SenderType senderType = new SenderType();
        builder.buildAddressee(senderType);
    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehbox.v3.builders.impl.ConsultationMessageBuilderImpl#buildAddressee(be.fgov.ehealth.ehbox.core.v3.SenderType)}
     * .
     */
    @Test
    public void testBuildAddresseeSenderType() {
        SenderType senderType = new SenderType();

        senderType.setFirstName(FIRSTNAME);
        senderType.setName(LASTNAME);
        senderType.setQuality(QUALITY);
        senderType.setSubType(SUBTYPE);
        senderType.setType(TYPE);

        Addressee result = builder.buildAddressee(senderType);

        Assert.assertNotNull(result);
        Assert.assertEquals(FIRSTNAME, result.getFirstName());
        Assert.assertEquals(LASTNAME, result.getLastName());
        Assert.assertEquals(QUALITY, result.getQuality());
        Assert.assertEquals(SUBTYPE, result.getSubType());
        Assert.assertEquals(TYPE, result.getType());


    }

    @Test(expected = IllegalArgumentException.class)
    public void testName() throws Exception {
        EhboxIdentifierType identifierType = new EhboxIdentifierType();
        builder.buildAddressee(identifierType);
    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehbox.v3.builders.impl.ConsultationMessageBuilderImpl#buildAddressee(be.fgov.ehealth.ehbox.core.v3.EhboxIdentifierType)}
     * .
     */
    @Test
    public void testBuildAddresseeEhboxIdentifierType() {
        EhboxIdentifierType identifierType = new EhboxIdentifierType();
        User user = new User();
        user.setFirstName(FIRSTNAME);
        user.setLastName(LASTNAME);
        identifierType.setUser(user);

        identifierType.setQuality(QUALITY);
        identifierType.setSubType(SUBTYPE);
        identifierType.setType(TYPE);

        Addressee result = builder.buildAddressee(identifierType);

        Assert.assertNotNull(result);
        Assert.assertEquals(FIRSTNAME, result.getFirstName());
        Assert.assertEquals(LASTNAME, result.getLastName());
        Assert.assertEquals(QUALITY, result.getQuality());
        Assert.assertEquals(SUBTYPE, result.getSubType());
        Assert.assertEquals(TYPE, result.getType());


    }

    @Test(expected = IllegalArgumentException.class)
    public void testBuildAddresseeSenderWithEmptySenderType() throws Exception {
        SenderType senderType = new SenderType();
        builder.buildAddressee(senderType);
    }

    /**
     * Test method for
     * {@link be.ehealth.businessconnector.ehbox.v3.builders.impl.ConsultationMessageBuilderImpl#buildAddressee(be.fgov.ehealth.ehbox.core.v3.SenderType)}
     * .
     */
    @Test
    public void testBuildAddresseeSenderTypeWithOrganisation() {
        SenderType senderType = new SenderType();

        senderType.setName(LASTNAME);
        senderType.setQuality(QUALITY);
        senderType.setSubType(SUBTYPE);
        senderType.setType(TYPE);
        senderType.setPersonInOrganisation(ID);

        Addressee result = builder.buildAddressee(senderType);

        Assert.assertNotNull(result);
        Assert.assertNull(result.getFirstName());
        Assert.assertNull(result.getLastName());
        Assert.assertEquals(LASTNAME, result.getOrganizationName());

        Assert.assertEquals(ID, result.getPersonInOrganisation());
        Assert.assertEquals(QUALITY, result.getQuality());
        Assert.assertEquals(SUBTYPE, result.getSubType());
        Assert.assertEquals(TYPE, result.getType());


    }

}
