/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.decrypt;

import be.fgov.ehealth.etee.crypto.cert.CertPathChecker;
import be.fgov.ehealth.etee.crypto.cert.CertificateStatus;
import be.fgov.ehealth.etee.crypto.decrypt.CMSMessageContext;
import be.fgov.ehealth.etee.crypto.decrypt.SignedDataNotification;
import be.fgov.ehealth.etee.crypto.decrypt.SignedDataVerifier;
import be.fgov.ehealth.etee.crypto.decrypt.SignedDataVerifierResult;
import be.fgov.ehealth.etee.crypto.decrypt.SignerInfoAttributes;
import be.fgov.ehealth.etee.crypto.decrypt.SignerInfoAttributesReceiver;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPChecker;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPData;
import be.fgov.ehealth.etee.crypto.ocsp.RevocationValues;
import be.fgov.ehealth.etee.crypto.policies.KeyType;
import be.fgov.ehealth.etee.crypto.policies.SignatureLayer;
import be.fgov.ehealth.etee.crypto.policies.SigningCredential;
import be.fgov.ehealth.etee.crypto.policies.SigningPolicy;
import be.fgov.ehealth.etee.crypto.status.CryptoResult;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.utils.KeyManager;
import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import javax.crypto.BadPaddingException;
import org.bouncycastle.asn1.x509.CertificateList;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.SignerId;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.cms.SignerInformationVerifier;
import org.bouncycastle.cms.jcajce.JcaSimpleSignerInfoVerifierBuilder;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class SignedDataVerifierAbstract
implements SignedDataVerifier {
    private static final Logger LOGGER = LoggerFactory.getLogger(SignedDataVerifierAbstract.class);
    private static final String CMS_SIGNATURE_NOT_VERIFIED = "The CMS signature could not be verified. The root cause can be that the Signed-Data content is empty. Check the exception trace.";
    private final Collection<SigningPolicy> signingPolicies;
    private final SignerInfoAttributesReceiver signerInfoAttributesReceiver;
    private final OCSPChecker ocspChecker;
    private final CertPathChecker certPathChecker;
    private final SignatureLayer layer;
    private final int[] acceptableKeyUsages;

    public SignedDataVerifierAbstract(SignatureLayer layer, Collection<SigningPolicy> signingPolicies, int[] acceptableKeyUsages, OCSPChecker ocspChecker, CertPathChecker certPathChecker, SignerInfoAttributesReceiver signerInfoAttributesReceiver) {
        this.layer = layer;
        this.signingPolicies = Collections.unmodifiableCollection(signingPolicies);
        this.acceptableKeyUsages = (int[])acceptableKeyUsages.clone();
        this.ocspChecker = ocspChecker;
        this.certPathChecker = certPathChecker;
        this.signerInfoAttributesReceiver = signerInfoAttributesReceiver;
    }

    protected void verifySigningKeySpecs(SignedDataVerifierResult result, SignerInformation signerInformation, PublicKey signerKey) {
        KeyType signerKeyType = KeyType.from(signerKey);
        int signerKeyLength = this.getKeyLength(signerKey);
        Set<SigningPolicy> applicablePolicies = this.signingPolicies.stream().filter(policy -> policy.getDigestAlgorithmOID().equals(signerInformation.getDigestAlgOID())).filter(policy -> policy.getEncryptionAlgorithmOIDs().contains(signerInformation.getEncryptionAlgOID())).filter(policy -> policy.getSignKeySize() <= signerKeyLength).collect(Collectors.toSet());
        result.setSigningPolicies(applicablePolicies);
        if (applicablePolicies.isEmpty()) {
            LOGGER.warn("Signature KeySpecs  [KeyType=" + (Object)((Object)signerKeyType) + ", digestAlgOID=" + signerInformation.getDigestAlgOID() + ", encryptionAlgOID " + signerInformation.getEncryptionAlgOID() + ", keyLength=" + signerKeyLength + "] not supported by policies [" + this.signingPolicies + "]");
            result.getErrors().add(SignedDataNotification.SIGNATURE_KEY_SPECS_UNAUTHORIZED.get(this.layer));
        }
    }

    private int getKeyLength(PublicKey pubKey) {
        switch (KeyType.from(pubKey)) {
            case RSA: {
                return ((RSAPublicKey)pubKey).getModulus().bitLength();
            }
            case DSA: {
                return ((DSAPublicKey)pubKey).getParams().getP().bitLength();
            }
            case EC: {
                return ((ECPublicKey)pubKey).getParams().getOrder().bitLength();
            }
        }
        return 0;
    }

    protected SignerInfoAttributes getSignerInfoAttributes(SignedDataVerifierResult result, SignerInformation signerInfo) {
        try {
            return this.signerInfoAttributesReceiver.parse(signerInfo);
        }
        catch (TSPException e) {
            LOGGER.warn(this.buildMsg("Contains invalid timestamp token. Unsigned SignerInfo Attributes will be ignored."), (Throwable)e);
            result.getErrors().add(NotificationError.TIMESTAMPTOKEN_INVALID);
            return new SignerInfoAttributes();
        }
    }

    protected void verifyCertificate(SignedDataVerifierResult result, CMSMessageContext<?> msgContext, SignerInfoAttributes signerInfoAttributes, Date inSigningTime) {
        SignerInformation signerInformation = msgContext.getSignerInformation();
        X509Certificate signerCertificate = msgContext.getAuthenticationCertificate();
        List<X509Certificate> certChain = msgContext.getCertificateChain();
        Date signingTime = inSigningTime;
        if (signingTime == null) {
            signingTime = new Date();
        }
        this.verifySignature(result, signerInformation, signerCertificate);
        this.verifySigningKeySpecs(result, signerInformation, signerCertificate.getPublicKey());
        this.verifyCertificateKeyUsage(result, signerCertificate);
        this.verifyCertificateValidity(result, signerCertificate, signingTime);
        this.verifyCertificateChainTrust(result, certChain, signingTime);
        this.verifyCertificateRevocationStatus(result, signerCertificate, signingTime, this.buildRevocationValues(signerInfoAttributes));
    }

    protected void verifyPublicKey(SignedDataVerifierResult result, CMSMessageContext<?> msgContext, String subjectKeyIdentifier, SigningCredential ... signers) {
        SignerInformation signerInformation = msgContext.getSignerInformation();
        PublicKey signerKey = this.getPublicKey(subjectKeyIdentifier, signers);
        if (signers.length == 0) {
            LOGGER.warn(this.buildMsg("No Signer credentials available."));
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.getLayer()));
        } else if (signerKey == null) {
            LOGGER.warn(this.buildMsg("Signer's Key missing in list of provided signing credentials."));
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.getLayer()));
        } else {
            this.verifySignature(result, signerInformation, signerKey);
            this.verifySigningKeySpecs(result, signerInformation, signerKey);
        }
    }

    protected String getSubjectKeyIdentifier(CMSMessageContext<?> msgContext) {
        SignerInformation signerInformation = msgContext.getSignerInformation();
        SignerId signerId = signerInformation.getSID();
        if (signerId == null || signerId.getSubjectKeyIdentifier() == null) {
            return null;
        }
        return new String(Base64.encode((byte[])msgContext.getSignerInformation().getSID().getSubjectKeyIdentifier()), StandardCharsets.UTF_8);
    }

    private PublicKey getPublicKey(String subjectKeyId, SigningCredential ... signers) {
        for (SigningCredential signer : signers) {
            if (!subjectKeyId.equals(signer.getSubjectKeyIdentifier())) continue;
            return signer.getPublicKey();
        }
        LOGGER.warn(this.buildMsg("No PublicKey found for subjectKeyIdentifier defined in signature: " + subjectKeyId));
        return null;
    }

    protected void verifySignature(SignedDataVerifierResult result, SignerInformation signerInfo, X509Certificate signerCertificate) {
        try {
            if (this.isSignatureDoneWithSignerCertificate(signerInfo, signerCertificate)) {
                LOGGER.debug(this.buildMsg("SignerInfo signature is OK."));
            } else {
                LOGGER.warn(this.buildMsg("SignerInfo signature is not created with private key corresponding to given cert."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
        }
        catch (OperatorCreationException e) {
            LOGGER.warn(this.buildMsg("Signer certificate could not be extracted from the message"), (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
        catch (CMSException e) {
            if (e.getCause() instanceof BadPaddingException || e.getCause() instanceof InvalidKeyException) {
                LOGGER.warn(this.buildMsg("Signature is not created with the private key that corresponds to the given certificate."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
            LOGGER.warn(CMS_SIGNATURE_NOT_VERIFIED, (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
    }

    protected void verifySignature(SignedDataVerifierResult result, SignerInformation signerInfo, PublicKey signerKey) {
        try {
            if (this.isSignatureDoneWithSignerKey(signerInfo, signerKey)) {
                LOGGER.debug(this.buildMsg("SignerInfo signature is OK."));
            } else {
                LOGGER.warn(this.buildMsg("SignerInfo signature is not created with private key corresponding to given cert."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
        }
        catch (OperatorCreationException e) {
            LOGGER.warn(this.buildMsg("Signer certificate could not be extracted from the message"), (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
        catch (CMSException e) {
            if (e.getCause() instanceof BadPaddingException || e.getCause() instanceof InvalidKeyException) {
                LOGGER.warn(this.buildMsg("Signature is not created with the private key that corresponds to the given certificate."));
                result.getErrors().add(SignedDataNotification.SIGNATURE_DOES_NOT_MATCH_CERTIFICATE.get(this.layer));
            }
            LOGGER.warn(CMS_SIGNATURE_NOT_VERIFIED, (Throwable)e);
            result.getErrors().add(SignedDataNotification.SIGNATURE_COULD_NOT_BE_VERIFIED.get(this.layer));
        }
    }

    private void verifyCertificateValidity(SignedDataVerifierResult result, X509Certificate signerCertificate, Date signingTime) {
        try {
            signerCertificate.checkValidity(signingTime);
            LOGGER.debug(this.buildMsg("Signer cert valid at signing time " + signingTime));
        }
        catch (CertificateExpiredException e) {
            LOGGER.warn(this.buildMsg("Signer cert has expired."), (Throwable)e);
            result.getErrors().add(SignedDataNotification.CERTIFICATE_EXPIRED.get(this.layer));
        }
        catch (CertificateNotYetValidException e) {
            LOGGER.warn(this.buildMsg("Signer cert is not yet valid."), (Throwable)e);
            result.getErrors().add(SignedDataNotification.CERTIFICATE_NOT_YET_VALID.get(this.layer));
        }
    }

    private void verifyCertificateChainTrust(SignedDataVerifierResult result, List<X509Certificate> certificateChain, Date signingTime) {
        LOGGER.debug(this.buildMsg("Verifying the trust in the certificate chain in the Signed-Data"));
        boolean isTrustedChain = false;
        if (!certificateChain.isEmpty()) {
            CryptoResult<CertificateStatus> certPathResult = this.certPathChecker.validate(certificateChain, signingTime);
            if (certPathResult.hasErrors()) {
                LOGGER.warn("CertPath Validation completed with errors: " + certPathResult);
                result.getErrors().addAll(certPathResult.getErrors());
            } else {
                CertificateStatus certStatus = certPathResult.getData();
                if (certStatus == CertificateStatus.VALID) {
                    LOGGER.debug(this.buildMsg("Certificate chain trusted at signing time " + signingTime));
                    isTrustedChain = true;
                } else {
                    LOGGER.warn(this.buildMsg("Certificate chain is not trusted: " + (Object)((Object)certStatus)));
                }
            }
            if (certPathResult.hasWarnings()) {
                LOGGER.warn("CertPath Validation completed with warnings: " + certPathResult.getWarnings());
                result.getWarnings().addAll(certPathResult.getWarnings());
            }
        } else {
            LOGGER.warn(this.buildMsg("No cert chain available."));
        }
        if (!isTrustedChain) {
            result.getErrors().add(SignedDataNotification.CERTIFICATE_CHAIN_NOT_TRUSTED.get(this.layer));
        }
    }

    private void verifyCertificateRevocationStatus(SignedDataVerifierResult result, X509Certificate signerCertificate, Date signingTime, RevocationValues revocationValues) {
        CryptoResult<OCSPData> checkerResult = this.ocspChecker.validate(signerCertificate, signingTime, revocationValues);
        if (checkerResult.hasWarnings()) {
            LOGGER.error(this.buildMsg("OCSP Validation completed with warnings: " + checkerResult));
            result.getWarnings().addAll(checkerResult.getWarnings());
        }
        if (checkerResult.hasErrors()) {
            LOGGER.error(this.buildMsg("OCSP Validation completed with errors: " + checkerResult));
            result.getErrors().addAll(checkerResult.getErrors());
            result.getErrors().add(SignedDataNotification.CERTIFICATE_STATUS_UNKNOWN.get(this.layer));
        }
        if (checkerResult.getFatal() == null) {
            if (CertificateStatus.REVOKED.equals((Object)checkerResult.getData().getCertStatus())) {
                result.getErrors().add(SignedDataNotification.CERTIFICATE_REVOKED.get(this.layer));
            }
            result.setOcspData(checkerResult.getData());
        }
    }

    private RevocationValues buildRevocationValues(SignerInfoAttributes signerInfo) {
        RevocationValues revocationValues = new RevocationValues();
        for (BasicOCSPResp ocsp : signerInfo.getOcspResponses()) {
            try {
                revocationValues.getOcspVals().add(ocsp.getEncoded());
            }
            catch (Exception e) {
                LOGGER.warn(this.buildMsg("Embedded OCSP could not be encoded. It will be skipped."), (Throwable)e);
            }
        }
        for (CertificateList crl : signerInfo.getCrls()) {
            try {
                ByteArrayInputStream inStream = new ByteArrayInputStream(crl.getEncoded());
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                revocationValues.getCrlVals().add((X509CRL)cf.generateCRL(inStream));
                inStream.close();
            }
            catch (Exception e) {
                LOGGER.warn(this.buildMsg("Embedded CRL could not be encoded. It will be skipped."), (Throwable)e);
            }
        }
        return revocationValues;
    }

    void verifyCertificateKeyUsage(SignedDataVerifierResult result, X509Certificate signerCertificate) {
        int certKeyUsage = KeyManager.getKeyUsage(signerCertificate);
        for (int acceptableKeyUsage : this.acceptableKeyUsages) {
            if ((certKeyUsage & acceptableKeyUsage) == 0) continue;
            LOGGER.debug(this.buildMsg("Signer certificate's key usage is valid."));
            return;
        }
        LOGGER.warn(this.buildMsg("Signer certificate's key usage is invalid."));
        result.getErrors().add(SignedDataNotification.CERTIFICATE_HAS_INVALID_KEYUSAGE.get(this.getLayer()));
    }

    boolean isSignatureDoneWithSignerCertificate(SignerInformation signerInfo, X509Certificate signerCertificate) throws OperatorCreationException, CMSException {
        SignerInformationVerifier verifier = new JcaSimpleSignerInfoVerifierBuilder().build(signerCertificate);
        return signerInfo.verify(verifier);
    }

    boolean isSignatureDoneWithSignerKey(SignerInformation signerInfo, PublicKey signerKey) throws OperatorCreationException, CMSException {
        SignerInformationVerifier verifier = new JcaSimpleSignerInfoVerifierBuilder().build(signerKey);
        return signerInfo.verify(verifier);
    }

    protected SignatureLayer getLayer() {
        return this.layer;
    }

    protected String buildMsg(String msg) {
        return "[Signature Layer " + (Object)((Object)this.getLayer()) + "] " + msg;
    }

    protected void verifySigner(SignedDataVerifierResult result, X509Certificate signerCertificate, SigningCredential ... signers) {
        boolean signerAcceptable = false;
        for (SigningCredential acceptedSigner : signers) {
            if (acceptedSigner.getCertificateChain() != null && !acceptedSigner.getCertificateChain().isEmpty() && signerCertificate.equals(acceptedSigner.getCertificateChain().get(0))) {
                signerAcceptable = true;
                break;
            }
            if (acceptedSigner.getPublicKey() == null || !signerCertificate.getPublicKey().equals(acceptedSigner.getPublicKey())) continue;
            signerAcceptable = true;
            break;
        }
        if (!signerAcceptable) {
            LOGGER.warn(this.buildMsg("Message is not signed by expected signer credential(s): Crt Serialnumber = " + signerCertificate.getSerialNumber()));
            result.getErrors().add(SignedDataNotification.SIGNER_NOT_ALLOWED.get(this.getLayer()));
        }
    }
}

