/*
 * Copyright (c) eHealth
 */
package be.ehealth.technicalconnector.service.etee.impl;

import be.ehealth.technicalconnector.service.etee.CryptoFactory;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.fgov.ehealth.etee.crypto.policies.OCSPOption;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import org.junit.*;

import java.io.InputStream;
import java.security.KeyStore;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;


/**
 * Integration test for {@link ConnectorRevocationStatusChecker}
 *
 * @author EHP
 */
public class ConnectorRevocationStatusCheckerIntegrationTest {

    @ClassRule
    public static SessionRule rule = SessionRule.withInactiveSession().build();
    private List<String> aliasList = new ArrayList<String>();

    @Before
    public void before() throws Exception {
        KeyStore store = (KeyStore) CryptoFactory.getOCSPOptions().get(OCSPOption.TRUST_STORE);

        //clean store
        Enumeration<String> aliases = store.aliases();
        while (aliases.hasMoreElements()) {
            store.deleteEntry(aliases.nextElement());
        }

        //import specific one
        KeyStore loadedStore = KeyStore.getInstance("JKS");
        loadedStore.load(ConnectorIOUtils.getResourceAsStream("/etee/caCertificateKeystore.jks"), "system".toCharArray());
        aliases = loadedStore.aliases();

        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            aliasList.add(alias);
            store.setCertificateEntry(alias, loadedStore.getCertificate(alias));
        }
    }

    @After
    public void after() throws Exception {
        KeyStore store = (KeyStore) CryptoFactory.getOCSPOptions().get(OCSPOption.TRUST_STORE);
        for (String alias : aliasList) {
            store.deleteEntry(alias);
        }
    }


    /**
     * Tests RevocationStatusChecker on a *.ehealth.fgov.be
     */
    @Test
    public void testWithSSLCertificate() throws Exception {
        ConnectorRevocationStatusChecker checker = new ConnectorRevocationStatusChecker();
        boolean isRevoked = checker.isRevoked(generate("/etee/www.ehealth.fgov.be.cer"));
        Assert.assertFalse(isRevoked);
    }

    /**
     * Tests RevocationStatusChecker on a CA
     */
    @Test
    public void testWithSigningCertificate() throws Exception {
        ConnectorRevocationStatusChecker checker = new ConnectorRevocationStatusChecker();
        boolean isRevoked = checker.isRevoked(generate("/etee/servicelist.cer"));
        Assert.assertFalse(isRevoked);
    }

    @Test
    public void testWithSelfSignedCertificate() throws Exception {
        ConnectorRevocationStatusChecker checker = new ConnectorRevocationStatusChecker();
        boolean isRevoked = checker.isRevoked(generate("/etee/tsl.ehealth.fgov.be.cer"));
        Assert.assertFalse(isRevoked);
    }

    public static X509Certificate generate(String location) throws Exception {
        InputStream is = null;
        try {
            is = ConnectorIOUtils.getResourceAsStream(location);
            return (java.security.cert.X509Certificate) CertificateFactory.getInstance("X.509")
                    .generateCertificate(is);
        } finally {
            ConnectorIOUtils.closeQuietly(is);
        }
    }
}
