/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.ocsp;

import be.fgov.ehealth.etee.crypto.cert.CertPathChecker;
import be.fgov.ehealth.etee.crypto.crl.CRLChecker;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPChecker;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPCheckerResult;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPClient;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPClientFactory;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPData;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPException;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPResponderTrustResult;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPResponderTrustService;
import be.fgov.ehealth.etee.crypto.ocsp.RevocationValues;
import be.fgov.ehealth.etee.crypto.ocsp.RevocationValuesFactory;
import be.fgov.ehealth.etee.crypto.policies.OCSPOption;
import be.fgov.ehealth.etee.crypto.policies.OCSPOptions;
import be.fgov.ehealth.etee.crypto.policies.OCSPPolicy;
import be.fgov.ehealth.etee.crypto.status.CryptoResult;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.status.NotificationFatal;
import be.fgov.ehealth.etee.crypto.status.NotificationWarning;
import be.fgov.ehealth.etee.crypto.utils.Preconditions;
import be.fgov.ehealth.etee.crypto.utils.TimeFrameValidator;
import java.io.IOException;
import java.security.KeyStore;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.CertificateID;
import org.bouncycastle.cert.ocsp.CertificateStatus;
import org.bouncycastle.cert.ocsp.RevokedStatus;
import org.bouncycastle.cert.ocsp.SingleResp;
import org.bouncycastle.cert.ocsp.UnknownStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OCSPCheckerImpl
implements OCSPChecker {
    private static final Logger LOGGER = LoggerFactory.getLogger(OCSPCheckerImpl.class);
    private OCSPResponderTrustService trustEngine;
    private TimeFrameValidator timeFrameValidator;
    private OCSPPolicy policy;
    private OCSPClient ocspClient;
    private RevocationValuesFactory revocationValuesFactory;

    OCSPCheckerImpl(OCSPPolicy policy, OCSPOptions policyOptions, CRLChecker crlChecker, CertPathChecker certPathChecker) {
        LOGGER.debug("Initializing OCSPChecker with following options: " + policyOptions);
        this.policy = policy;
        this.revocationValuesFactory = new RevocationValuesFactory();
        this.ocspClient = OCSPClientFactory.getInstance().createOCSPClient(policyOptions);
        KeyStore trustedOscpResponderCaCerts = policyOptions.getKeyStore(OCSPOption.TRUST_STORE);
        this.trustEngine = new OCSPResponderTrustService(trustedOscpResponderCaCerts, crlChecker, certPathChecker);
        Long acceptedClockSkew = policyOptions.getLong(OCSPOption.CLOCK_SKEW);
        this.timeFrameValidator = TimeFrameValidator.create(acceptedClockSkew);
    }

    @Override
    public OCSPPolicy getPolicy() {
        return this.policy;
    }

    @Override
    public CryptoResult<OCSPData> validate(X509Certificate certificate) {
        return this.validate(certificate, new Date(), new RevocationValues());
    }

    @Override
    public CryptoResult<OCSPData> validate(X509Certificate certificate, Date signingTime, RevocationValues revocationValues) {
        return this.validate(new ArrayList<X509Certificate>(Arrays.asList(certificate)), signingTime, revocationValues);
    }

    @Override
    public CryptoResult<OCSPData> validate(List<X509Certificate> certificateChain) {
        return this.validate(certificateChain, new Date(), new RevocationValues());
    }

    @Override
    public CryptoResult<OCSPData> validate(List<X509Certificate> certificateChain, Date signingTime, RevocationValues revocationValues) {
        this.checkEndCertNotNull(certificateChain);
        OCSPCheckerResult result = null;
        try {
            switch (this.getPolicy()) {
                case NONE: {
                    LOGGER.debug("OCSPCheck disabled. Returning status VALID.");
                    return new OCSPCheckerResult(new OCSPData(be.fgov.ehealth.etee.crypto.cert.CertificateStatus.VALID, null, null));
                }
                case SENDER_OPTIONAL: {
                    return this.validateOptionalOCSP(certificateChain, signingTime, revocationValues);
                }
                case SENDER_MANDATORY: {
                    return this.validateMandatoryOCSP(certificateChain, signingTime, revocationValues);
                }
                case RECEIVER_OPTIONAL: {
                    return this.validateOCSP(certificateChain, signingTime, revocationValues);
                }
                case RECEIVER_MANDATORY: {
                    return this.validateOnlineOCSP(certificateChain, signingTime, revocationValues);
                }
            }
            LOGGER.error("Unsupported OCSPPolicy [" + (Object)((Object)this.getPolicy()) + "]. OCSPCheck failed.");
        }
        catch (Exception e) {
            LOGGER.error("OCSPCheck failed.", (Throwable)e);
            result = new OCSPCheckerResult(NotificationFatal.OCSP_CHECK_FAILED);
        }
        return result;
    }

    private CryptoResult<OCSPData> validateOptionalOCSP(List<X509Certificate> certificateChain, Date signingTime, RevocationValues revocationValues) throws OCSPException, IOException {
        if (!revocationValues.getOcspVals().isEmpty()) {
            return this.validateEmbeddedOCSP(certificateChain, signingTime, revocationValues);
        }
        OCSPCheckerResult result = new OCSPCheckerResult(new OCSPData(be.fgov.ehealth.etee.crypto.cert.CertificateStatus.VALID, null, null));
        result.getWarnings().add(NotificationWarning.OCSP_NOT_EMBEDDED);
        return result;
    }

    private CryptoResult<OCSPData> validateMandatoryOCSP(List<X509Certificate> certificateChain, Date signingTime, RevocationValues revocationValues) throws OCSPException, IOException {
        if (!revocationValues.getOcspVals().isEmpty()) {
            return this.validateEmbeddedOCSP(certificateChain, signingTime, revocationValues);
        }
        OCSPCheckerResult result = new OCSPCheckerResult(new OCSPData(be.fgov.ehealth.etee.crypto.cert.CertificateStatus.UNSPECIFIED, null, null));
        result.getErrors().add(NotificationError.OCSP_NOT_EMBEDDED);
        return result;
    }

    private CryptoResult<OCSPData> validateOCSP(List<X509Certificate> certificateChain, Date signingTime, RevocationValues revocationValues) throws OCSPException, IOException {
        if (!revocationValues.getOcspVals().isEmpty()) {
            return this.validateEmbeddedOCSP(certificateChain, signingTime, revocationValues);
        }
        return this.validateOnlineOCSP(certificateChain, signingTime, revocationValues);
    }

    private CryptoResult<OCSPData> validateEmbeddedOCSP(List<X509Certificate> certificateChain, Date signingTime, RevocationValues revocationValues) throws OCSPException, IOException {
        BasicOCSPResp embeddedOCSPResponse = this.getEmbeddedOCSPResponse(revocationValues, certificateChain.get(0));
        return this.checkEmbeddedOCSP(certificateChain, signingTime, embeddedOCSPResponse, revocationValues.getCrlVals());
    }

    private CryptoResult<OCSPData> validateOnlineOCSP(List<X509Certificate> certificateChain, Date signingTime, RevocationValues revocationValues) throws OCSPException, IOException {
        BasicOCSPResp basicResponse = this.ocspClient.sendRequestAndCheckResponse(certificateChain.get(0));
        return this.checkOCSP(certificateChain, signingTime, basicResponse, revocationValues.getCrlVals());
    }

    BasicOCSPResp getEmbeddedOCSPResponse(RevocationValues revocationValues, X509Certificate cert) throws OCSPException {
        try {
            Iterator<byte[]> i$ = revocationValues.getOcspVals().iterator();
            if (i$.hasNext()) {
                byte[] ocspEncoded = i$.next();
                return this.revocationValuesFactory.decodeOCSPResponse(ocspEncoded);
            }
        }
        catch (IOException e) {
            throw new OCSPException("Embedded OCSPResponse could not be read.", e);
        }
        return null;
    }

    CryptoResult<OCSPData> checkEmbeddedOCSP(List<X509Certificate> certificateChain, Date signingDate, BasicOCSPResp ocspResponse, List<X509CRL> crls) throws OCSPException, IOException {
        LOGGER.debug("Check embedded OCSP, produced at " + ocspResponse.getProducedAt());
        CryptoResult<OCSPData> result = this.checkOCSP(certificateChain, signingDate, ocspResponse, crls);
        if (result.getErrors().contains((Object)NotificationError.OCSP_RESPONDER_CHAIN_EXPIRED) || result.getErrors().contains((Object)NotificationError.OCSP_RESPONDER_CHAIN_REVOKED)) {
            LOGGER.warn("Embedded ocspResponse is no longer valid: " + result.getErrors() + ". An online OCSP check will be executed.");
            BasicOCSPResp onlineOcsp = this.ocspClient.sendRequestAndCheckResponse(certificateChain.get(0));
            result = this.checkOCSP(certificateChain, signingDate, onlineOcsp, crls);
        }
        return result;
    }

    CryptoResult<OCSPData> checkOCSP(List<X509Certificate> certificateChain, Date signingTime, BasicOCSPResp ocspResponse, List<X509CRL> crls) throws OCSPException, IOException {
        this.checkEndCertNotNull(certificateChain);
        Preconditions.checkNotNull(ocspResponse, "OCSP check can only be performed with a valid ocspResponse, this one is null");
        OCSPResponderTrustResult trust = this.trustEngine.verify(certificateChain, ocspResponse, crls);
        if (trust.hasErrors()) {
            OCSPCheckerResult result;
            if (trust.getFatal() != null) {
                result = new OCSPCheckerResult(trust.getFatal());
            } else {
                result = new OCSPCheckerResult(new OCSPData(be.fgov.ehealth.etee.crypto.cert.CertificateStatus.UNSPECIFIED, null, (List)trust.getData()));
                result.getErrors().addAll(trust.getErrors());
            }
            return result;
        }
        be.fgov.ehealth.etee.crypto.cert.CertificateStatus certStatus = this.getCertificateStatus(certificateChain.get(0), signingTime, ocspResponse);
        OCSPCheckerResult result = new OCSPCheckerResult(new OCSPData(certStatus, ocspResponse.getEncoded(), (List)trust.getData()));
        if (be.fgov.ehealth.etee.crypto.cert.CertificateStatus.UNSPECIFIED.equals((Object)certStatus)) {
            result.getErrors().add(NotificationError.OCSP_RESPONSE_INVALID);
        }
        if (trust.hasWarnings()) {
            result.getWarnings().addAll(trust.getWarnings());
        }
        return result;
    }

    private void checkEndCertNotNull(List<X509Certificate> certificateChain) {
        if (certificateChain == null || certificateChain.isEmpty() || certificateChain.get(0) == null) {
            throw new IllegalArgumentException("OCSP check can only be performed if the certificate is not null");
        }
    }

    private be.fgov.ehealth.etee.crypto.cert.CertificateStatus getCertificateStatus(X509Certificate certificate, Date inSigningTime, BasicOCSPResp basicOcspResp) throws OCSPException {
        SingleResp[] singleResps;
        Date signingTime = inSigningTime;
        if (signingTime != null) {
            if (!this.timeFrameValidator.validate(null, signingTime, basicOcspResp.getProducedAt())) {
                throw new OCSPException("The OCSP response was produced before the SigningTime.");
            }
        } else {
            signingTime = new Date();
        }
        for (SingleResp singleResp : singleResps = basicOcspResp.getResponses()) {
            if (!this.isMatch(singleResp.getCertID(), certificate)) continue;
            return this.validateStatus(certificate, signingTime, singleResp.getCertStatus());
        }
        String errMsg = " the OCSP response does not contain a status of the certificate in question [SerialNumber=" + certificate.getSerialNumber() + "]";
        LOGGER.warn(errMsg);
        return be.fgov.ehealth.etee.crypto.cert.CertificateStatus.UNSPECIFIED;
    }

    private be.fgov.ehealth.etee.crypto.cert.CertificateStatus validateStatus(X509Certificate certificate, Date signingTime, Object status) {
        if (status == CertificateStatus.GOOD) {
            LOGGER.debug("Certificate " + certificate.getSubjectX500Principal() + " is NOT REVOKED.");
            return be.fgov.ehealth.etee.crypto.cert.CertificateStatus.VALID;
        }
        if (status instanceof RevokedStatus) {
            Date revocationTime = ((RevokedStatus)status).getRevocationTime();
            if (signingTime.before(revocationTime)) {
                LOGGER.debug("OCSP revocation time [" + revocationTime + "] after the signing time, the certificate was still valid at " + signingTime);
                return be.fgov.ehealth.etee.crypto.cert.CertificateStatus.VALID;
            }
            LOGGER.warn("OCSP revocation time [" + revocationTime + "] before the signing time, the certificate was no longer valid at " + signingTime);
            return be.fgov.ehealth.etee.crypto.cert.CertificateStatus.REVOKED;
        }
        if (status instanceof UnknownStatus) {
            String msg = " the revocation status of certificate is UNKNOWN by the OCSP Service.";
            LOGGER.warn(" the revocation status of certificate is UNKNOWN by the OCSP Service.");
            return be.fgov.ehealth.etee.crypto.cert.CertificateStatus.UNSPECIFIED;
        }
        String errMsg = " the OCSP response contains an incorrect revocation status field: " + status;
        LOGGER.warn(errMsg);
        return be.fgov.ehealth.etee.crypto.cert.CertificateStatus.UNSPECIFIED;
    }

    private boolean isMatch(CertificateID certID, X509Certificate certificate) {
        if (certID == null) {
            LOGGER.warn("CertificateID is null.");
            return false;
        }
        if (certificate == null) {
            LOGGER.warn("CertificateID is null.");
            return false;
        }
        return certID.getSerialNumber().equals(certificate.getSerialNumber());
    }
}

