/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.consultrn.session;

import be.ehealth.businessconnector.consultrn.exception.identifyperson.ConsultrnIdentifyPersonException;
import be.ehealth.businessconnector.consultrn.exception.manageperson.ConsultrnRegisterExistingPersonException;
import be.ehealth.businessconnector.consultrn.exception.manageperson.ConsultrnRegisterPersonException;
import be.ehealth.businessconnector.consultrn.exception.phoneticsearch.ConsultrnPhoneticSearchException;
import be.ehealth.businessconnector.testcommons.utils.FileTestUtils;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.idgenerator.IdGeneratorFactory;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.commons._1_0.core.PeriodType;
import be.fgov.ehealth.consultrn._1_0.core.EncodedSSINType;
import be.fgov.ehealth.consultrn._1_0.core.InscriptionType;
import be.fgov.ehealth.consultrn._1_0.protocol.PhoneticCriteriaType;
import be.fgov.ehealth.consultrn._1_0.protocol.SearchBySSINReply;
import be.fgov.ehealth.consultrn._1_0.protocol.SearchBySSINRequest;
import be.fgov.ehealth.consultrn._1_0.protocol.SearchPhoneticReply;
import be.fgov.ehealth.consultrn._1_0.protocol.SearchPhoneticRequest;
import be.fgov.ehealth.consultrn.protocol.v2.RegisterPersonRequest;
import be.fgov.ehealth.consultrn.protocol.v2.RegisterPersonResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.apache.commons.lang.RandomStringUtils;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import javax.xml.ws.soap.SOAPFaultException;
import java.util.HashMap;
import java.util.Map;

import static org.junit.Assert.*;

/**
 * Integration tests for consultrn identify person and phonetic search.
 *
 * @author EHP
 *
 */
public final class ConsultrnIntegrationTest {

    private static final String SSIN = "85040309180";

    private static final DateTime BEGIN_DATE = new DateTime();

    private static final DateTime END_DATE = new DateTime().plusDays(1);

    @ClassRule
    public static SessionRule sessionRule = SessionRule.withActiveSession().baseOn("/be.ehealth.businessconnector.consultrn.test.properties").build();

    @Rule
    public ExpectedException expectedEx = ExpectedException.none();

    @Test
    public void searchBySSIN_Search() throws Exception {
        SearchBySSINRequest searchBySSINRequest = createSearchBySsinRequest("78080904422", "85040309180", "1", BEGIN_DATE, END_DATE);

        SearchBySSINReply searchBySSINReply = ConsultrnSessionServiceFactory.getConsultrnSession().search(searchBySSINRequest);
        Assert.assertNotNull(searchBySSINReply);
        String expectedResponse = ConnectorIOUtils.getResourceAsString("/examples/identifyPerson/response/searchBySSINReply-search.xml");
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(searchBySSINReply));
    }

    @Test
    public void searchBySSIN_SearchWrongIdentifier() throws ConnectorException {
        SearchBySSINRequest searchBySSINRequest = createSearchBySsinRequest("10410111363", "56082053893", "10", BEGIN_DATE, END_DATE);

        try {
            ConsultrnSessionServiceFactory.getConsultrnSession().search(searchBySSINRequest);
            fail("should throw ConsultrnBusinessConnectorIdentifyPersonException");
        } catch (ConsultrnIdentifyPersonException e) {
            assertNotNull(e.getSearchBySSINReply());
            assertEquals("60", e.getSearchBySSINReply().getStatus().getCode());
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><ns2:Status xmlns:ns2=\"urn:be:fgov:ehealth:commons:1_0:core\"><Code>60</Code><Message Lang=\"EN\">Business Problem : invalid request</Message><Message Lang=\"EN\">SSIN malformed</Message></ns2:Status>", e.getMessage());
        }
    }

    @Test
    public void searchBySSIN_SearchWrongQuality() throws Exception {
        expectedEx.expect(ConsultrnIdentifyPersonException.class);
        expectedEx.expectMessage("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><ns2:Status xmlns:ns2=\"urn:be:fgov:ehealth:commons:1_0:core\"><Code>86</Code><Message Lang=\"EN\">Business Error : wrong quality</Message><Message Lang=\"EN\">Wrong quality (IdentifyPerson)</Message></ns2:Status>");

        SearchBySSINRequest searchBySSINRequest = createSearchBySsinRequest("10410111363", "56082053894", "10", BEGIN_DATE, END_DATE);

        ConsultrnSessionServiceFactory.getConsultrnSession().search(searchBySSINRequest);
    }

    @Test
    public void searchBySSIN_SearchNoPeriodSoapFaultException() throws Exception {
        expectedEx.expect(SOAPFaultException.class);
        expectedEx.expectMessage("XML could not be validated against XSD.");
        SearchBySSINRequest searchBySSINRequest = createSearchBySsinRequest("10410111363", "56082053894", "10");

        ConsultrnSessionServiceFactory.getConsultrnSession().search(searchBySSINRequest);
    }

    @Test
    public void searchBySSIN_SearchSameBeginDate() throws Exception {
        SearchBySSINRequest searchBySSINRequest = createSearchBySsinRequest("78080904422", SSIN, "1", BEGIN_DATE, BEGIN_DATE);

        SearchBySSINReply searchBySSINReply = ConsultrnSessionServiceFactory.getConsultrnSession().search(searchBySSINRequest);
        assertNotNull(searchBySSINReply);
        String expectedResponse = ConnectorIOUtils.getResourceAsString("/examples/identifyPerson/response/searchBySSINReply-searchWrongBeginDate.xml");
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(searchBySSINReply));
    }

    @Test
    public void searchBySSIN_SearchWrongEndDateBusinessError() throws Exception {
        expectedEx.expect(ConsultrnIdentifyPersonException.class);
        expectedEx.expectMessage("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><ns2:Status xmlns:ns2=\"urn:be:fgov:ehealth:commons:1_0:core\"><Code>85</Code><Message Lang=\"EN\">Business Error : invalid dates</Message><Message Lang=\"EN\">Period is invalid</Message></ns2:Status>");
        SearchBySSINRequest searchBySSINRequest = createSearchBySsinRequest("78080904422", SSIN, "1", END_DATE, BEGIN_DATE);

        ConsultrnSessionServiceFactory.getConsultrnSession().search(searchBySSINRequest);
    }

    @Test
    public void searchPhonetic_Search() throws Exception {
        SearchPhoneticRequest searchPhoneticRequest = createSearchPhoneticRequest("10410111363", "Dupont", "Benoit", "1984-06-11");

        SearchPhoneticReply searchPhoneticReply = ConsultrnSessionServiceFactory.getConsultrnSession().search(searchPhoneticRequest);
        Assert.assertNotNull(searchPhoneticReply);
        String expectedResponse = ConnectorIOUtils.getResourceAsString("/examples/phoneticSearch/response/searchPhoneticReply-search.xml");
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(searchPhoneticReply));
    }

    @Test
    public void searchPhonetic_SearchWrongBirthdateSoapFaultException() throws Exception {
        expectedEx.expect(SOAPFaultException.class);
        expectedEx.expectMessage("XML could not be validated against XSD.");
        SearchPhoneticRequest searchPhoneticRequest = createSearchPhoneticRequest("10410111363", "Dupont", "Benoit", "1984-06-112");

        ConsultrnSessionServiceFactory.getConsultrnSession().search(searchPhoneticRequest);
    }

    @Test
    public void searchPhonetic_SearchBusinessError() throws Exception {
        SearchPhoneticRequest searchPhoneticRequest = createSearchPhoneticRequest("10410111363", "Dupont", "Benoit", "1984-00-00");

        try {
            ConsultrnSessionServiceFactory.getConsultrnSession().search(searchPhoneticRequest);
            fail("should throw ConsultrnBusinessConnectorIdentifyPersonException");
        } catch (ConsultrnPhoneticSearchException e) {
            assertNotNull(e.getSearchPhoneticReply());
            assertEquals("63", e.getSearchPhoneticReply().getStatus().getCode());
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><ns2:Status xmlns:ns2=\"urn:be:fgov:ehealth:commons:1_0:core\"><Code>63</Code><Message Lang=\"EN\">Business Problem : invalid request</Message><Message Lang=\"EN\">Invalid phonetic search criteria</Message></ns2:Status>", e.getMessage());
        }
    }
    
    @Test
    public void registerPerson_MidBirth() throws Exception {
        RegisterPersonResponse response = registerPerson("/examples/managePerson/request/registerPerson-MIDBirth.xml");
        XmlAsserter.assertSimilar(ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-MIDBirth.xml"), ConnectorXmlUtils.toString(response));
    }

    @Test
    public void registerPerson_MidBelgianAddress() throws Exception {
        RegisterPersonResponse response = registerPerson("/examples/managePerson/request/registerPerson-MIDBelgianAddress.xml");
        XmlAsserter.assertSimilar(ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-MIDBelgianAddress.xml"), ConnectorXmlUtils.toString(response));
    }

    @Test
    public void registerPerson_SuccessMidForeignAddress() throws Exception {
        RegisterPersonResponse response = registerPerson("/examples/managePerson/request/registerPerson-MIDForeignAddress.xml");
        XmlAsserter.assertSimilar(ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-MIDForeignAddress.xml"), ConnectorXmlUtils.toString(response));
    }

    @Test
    public void registerPerson_FullInformation() throws Exception {
        RegisterPersonResponse response = registerPerson("/examples/managePerson/request/registerPerson-FullInformation.xml");
        XmlAsserter.assertSimilar(ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-FullInformation.xml"), ConnectorXmlUtils.toString(response));
    }

    @Test
    public void registerPerson_ExistingPersonEHP() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-ExistingPerson-EHP.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterExistingPersonException e) {
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><Status xmlns=\"urn:be:fgov:ehealth:commons:core:v2\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:Requester\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:RequestDenied\"/></StatusCode><StatusMessage>Registration of the BIS person cannot proceed. One or more persons with a phonetic match to the given criteria already exist.</StatusMessage></Status>", e.getMessage());
            XmlAsserter.assertSimilar(
                ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-ExistingPerson-EHP.xml"),
                ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }

    @Test
    public void registerPerson_ExistingPersonFilteredData() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-ExistingPerson-FilteredData.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterExistingPersonException e) {
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><Status xmlns=\"urn:be:fgov:ehealth:commons:core:v2\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:Requester\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:RequestDenied\"/></StatusCode><StatusMessage>Registration of the BIS person cannot proceed. One or more persons with a phonetic match to the given criteria already exist.</StatusMessage></Status>", e.getMessage());
            XmlAsserter.assertSimilar(
                ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-ExistingPerson-FilteredData.xml"),
                ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }

    @Test
    public void registerPerson_ExistingPersonHospital() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-ExistingPerson-HOSPITAL.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterExistingPersonException e) {
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><Status xmlns=\"urn:be:fgov:ehealth:commons:core:v2\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:Requester\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:RequestDenied\"/></StatusCode><StatusMessage>Registration of the BIS person cannot proceed. One or more persons with a phonetic match to the given criteria already exist.</StatusMessage></Status>", e.getMessage());
            XmlAsserter.assertSimilar(
                ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-ExistingPerson-HOSPITAL.xml"),
                ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }

    @Test
    public void registerPerson_ExistingPersonLabo() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-ExistingPerson-LABO.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterExistingPersonException e) {
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><Status xmlns=\"urn:be:fgov:ehealth:commons:core:v2\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:Requester\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:RequestDenied\"/></StatusCode><StatusMessage>Registration of the BIS person cannot proceed. One or more persons with a phonetic match to the given criteria already exist.</StatusMessage></Status>", e.getMessage());
            XmlAsserter.assertSimilar(
                ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-ExistingPerson-LABO.xml"), 
                ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }

    @Test
    public void registerPerson_ExistingPerson() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-ExistingPerson.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterExistingPersonException e) {
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><Status xmlns=\"urn:be:fgov:ehealth:commons:core:v2\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:Requester\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:RequestDenied\"/></StatusCode><StatusMessage>Registration of the BIS person cannot proceed. One or more persons with a phonetic match to the given criteria already exist.</StatusMessage></Status>", e.getMessage());
            XmlAsserter.assertSimilar(
                ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-ExistingPerson.xml"), 
                ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }

    @Test
    public void registerPerson_SeveralExistingPersons() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-SeveralExistingPerson.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterExistingPersonException e) {
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><Status xmlns=\"urn:be:fgov:ehealth:commons:core:v2\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:Requester\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:RequestDenied\"/></StatusCode><StatusMessage>Registration of the BIS person cannot proceed. One or more persons with a phonetic match to the given criteria already exist.</StatusMessage></Status>", e.getMessage());
            XmlAsserter.assertSimilar(
                ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-SeveralExistingPerson.xml"),
                ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }
    
    @Test
    public void registerPerson_InvalidFirstname() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-InvalidFirstname.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterPersonException e) {
            assertEquals("Error while executing web service call: <?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><Status xmlns=\"urn:be:fgov:ehealth:commons:core:v2\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:Requester\"><StatusCode Value=\"urn:be:fgov:ehealth:2.0:status:InvalidInput\"/></StatusCode><StatusMessage>XSD compliance failure: string length (49) is greater than maxLength facet (48) for GivenNameType in namespace http://kszbcss.fgov.be/types/legaldata/v3</StatusMessage></Status>", e.getMessage());
            XmlAsserter.assertSimilar(
                ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-InvalidFirstname.xml"),
                ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }

    @Test
    public void registerPerson_ErrorNoValidMID() throws Exception {
        try {
            registerPerson("/examples/managePerson/request/registerPerson-InvalidInput-NoValidMID.xml");
            Assert.fail("This test should send an exception");
        } catch (ConsultrnRegisterPersonException e) {
            Assert.assertEquals("800036", e.getBusinessAnomalies().getBusinessAnomalies().get(0).getCode());
            XmlAsserter.assertSimilar(
                    ConnectorIOUtils.getResourceAsString("/examples/managePerson/response/registerPerson-InvalidInput-NoValidMID.xml"),
                    ConnectorXmlUtils.toString(e.getRegisterPersonResponse()));
        }
    }
    
    private SearchBySSINRequest createSearchBySsinRequest(String applicationId, String ssin, String qualityCode, DateTime beginDate, DateTime endDate) {
        SearchBySSINRequest searchBySSINRequest = createSearchBySsinRequest(applicationId, ssin, qualityCode);
        PeriodType periodType = new PeriodType();
        periodType.setBeginDate(beginDate);
        periodType.setEndDate(endDate);
        searchBySSINRequest.getInscription().setPeriod(periodType);
        return searchBySSINRequest;
    }

    private SearchBySSINRequest createSearchBySsinRequest(String applicationId, String ssin, String qualityCode) {
        SearchBySSINRequest searchBySSINRequest = new SearchBySSINRequest();
        searchBySSINRequest.setApplicationID(applicationId);
        InscriptionType inscription = new InscriptionType();
        EncodedSSINType encodedSSINType = new EncodedSSINType();
        encodedSSINType.setValue(ssin);
        inscription.setSSIN(encodedSSINType);
        inscription.setQualityCode(qualityCode);
        searchBySSINRequest.setInscription(inscription);
        return searchBySSINRequest;
    }

    private SearchPhoneticRequest createSearchPhoneticRequest(String applicationId, String lastName, String firstName, String birthDate) {
        SearchPhoneticRequest searchPhoneticRequest = new SearchPhoneticRequest();
        searchPhoneticRequest.setApplicationID(applicationId);
        PhoneticCriteriaType phoneticCriteriaType = new PhoneticCriteriaType();
        phoneticCriteriaType.setLastName(lastName);
        phoneticCriteriaType.setFirstName(firstName);
        phoneticCriteriaType.setBirthDate(birthDate);
        searchPhoneticRequest.setPhoneticCriteria(phoneticCriteriaType);
        return searchPhoneticRequest;
    }

    private RegisterPersonResponse registerPerson(String testRequestFileLocation) throws Exception {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("issueInstant", DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'+02:00'").print(new DateTime()));
        velocityContext.put("requestId", IdGeneratorFactory.getIdGenerator("uuid").generateId());
        velocityContext.put("currentDate", DateTimeFormat.forPattern("yyyy-MM-dd").print(new DateTime()));
        velocityContext.put("generatedChars1", RandomStringUtils.randomAlphabetic(10).toLowerCase());
        velocityContext.put("generatedChars2", RandomStringUtils.randomAlphabetic(10).toLowerCase());
        
        RegisterPersonRequest request = FileTestUtils.toObject(velocityContext, testRequestFileLocation, RegisterPersonRequest.class);
        return ConsultrnSessionServiceFactory.getConsultrnSession().registerPerson(request);
    }
    
}
