/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.ocsp;

import be.fgov.ehealth.etee.crypto.cert.CertPathChecker;
import be.fgov.ehealth.etee.crypto.cert.CertificateStatus;
import be.fgov.ehealth.etee.crypto.crl.CRLChecker;
import be.fgov.ehealth.etee.crypto.crl.CRLData;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPException;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPResponderTrustResult;
import be.fgov.ehealth.etee.crypto.status.CryptoResult;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.utils.CertStringBuilder;
import be.fgov.ehealth.etee.crypto.utils.KeyManager;
import java.io.ByteArrayInputStream;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.cert.CertStore;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.security.cert.X509Extension;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;

final class OCSPResponderTrustService {
    private static final Logger LOGGER = Logger.getLogger(OCSPResponderTrustService.class);
    private KeyStore trustStore;
    private CRLChecker crlChecker;
    private CertPathChecker certPathChecker;

    public OCSPResponderTrustService(KeyStore ocspTrustStore, CRLChecker crlChecker, CertPathChecker certPathChecker) {
        this.trustStore = ocspTrustStore;
        this.crlChecker = crlChecker;
        this.certPathChecker = certPathChecker;
    }

    public OCSPResponderTrustResult verify(X509Certificate endCert, BasicOCSPResp ocspResponse) {
        return this.verify(endCert, ocspResponse, null);
    }

    public OCSPResponderTrustResult verify(X509Certificate endCert, BasicOCSPResp ocspResponse, List<X509CRL> crls) {
        this.logDetails(endCert, ocspResponse, crls);
        OCSPResponderTrustResult result = new OCSPResponderTrustResult();
        try {
            X509Certificate[] certificates = this.getCertificatesFromResponse(ocspResponse);
            this.verifyChain(certificates);
            for (int index = 0; index < certificates.length; ++index) {
                X509Certificate chainCert = certificates[index];
                if (KeyManager.isTrustStoreCert(this.trustStore, chainCert)) {
                    LOGGER.info((Object)("ChainCertificate " + chainCert.getSubjectX500Principal() + " found in trustStore. No revocation checks will be done."));
                    break;
                }
                this.verifyRevocation(endCert, crls, result, certificates, index, chainCert);
                if (!result.hasErrors()) continue;
                return result;
            }
        }
        catch (CertificateExpiredException e) {
            LOGGER.debug((Object)"Certificate is expired.", (Throwable)e);
            result.getErrors().add(NotificationError.OCSP_RESPONDER_CHAIN_EXPIRED);
        }
        catch (Exception e) {
            LOGGER.warn((Object)"Validation of OCSP Security Requirements failed.", (Throwable)e);
            result.getErrors().add(NotificationError.OCSP_RESPONDER_TRUST_FAILED);
        }
        if (!result.hasErrors()) {
            LOGGER.info((Object)"OCSP Responder is TRUSTED.");
        }
        return result;
    }

    private void verifyRevocation(X509Certificate endCert, List<X509CRL> crls, OCSPResponderTrustResult result, X509Certificate[] certificates, int index, X509Certificate chainCert) throws KeyStoreException, OCSPException, NoSuchProviderException, CertificateException, NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        X509Certificate issuer = null;
        if (index == certificates.length - 1) {
            issuer = KeyManager.getIssuerFromTrustStore(this.trustStore, chainCert);
        } else {
            issuer = certificates[index + 1];
            if (index == 0) {
                LOGGER.debug((Object)("OCSP Responder cert : " + CertStringBuilder.build(chainCert)));
                if (!this.verifyOCSPSigner(endCert, chainCert, issuer)) {
                    return;
                }
            }
        }
        if (issuer != null) {
            ArrayList<X509Extension> list = new ArrayList<X509Extension>();
            list.addAll(crls);
            list.add(issuer);
            CertStore certStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(list));
            CryptoResult<CRLData> crlResult = this.crlChecker.validate(chainCert, certStore);
            if (crlResult.hasErrors()) {
                result.getErrors().addAll(crlResult.getErrors());
                result.getErrors().add(NotificationError.OCSP_RESPONDER_TRUST_FAILED);
                return;
            }
            if (crlResult.hasWarnings()) {
                LOGGER.warn((Object)("CRL Check completed with warnings: " + crlResult.getWarnings()));
                result.getWarnings().addAll(crlResult.getWarnings());
            }
            result.addCrls(crlResult.getData().getCrls());
            this.validateResponderCertStatus(result, chainCert, crlResult);
        }
    }

    private boolean verifyOCSPSigner(X509Certificate endCert, X509Certificate ocspSigner, X509Certificate ocspIssuer) throws OCSPException, CertificateException {
        List<String> extendedKeyUsage = ocspSigner.getExtendedKeyUsage();
        if (extendedKeyUsage != null && extendedKeyUsage.contains(KeyPurposeId.id_kp_OCSPSigning.getId())) {
            if (!KeyManager.verifyIssuer(endCert, ocspIssuer)) {
                throw new CertificateException("The OCSP Signature is not set by an authorized responder (RFC2560). If you want to trust '" + ocspSigner.getSubjectX500Principal() + "' as a valid responder, you'll need to add it to your trustStore (local configuration).");
            }
            if (ocspSigner.getExtensionValue(OCSPObjectIdentifiers.id_pkix_ocsp_nocheck.getId()) != null) {
                return false;
            }
        } else if (!KeyManager.verifyIssuer(endCert, ocspSigner)) {
            throw new CertificateException("The OCSP Signature is not set by an authorized responder (RFC2560). If you want to trust '" + ocspSigner.getSubjectX500Principal() + "' as a valid responder, you'll need to add it to your trustStore (local configuration).");
        }
        return true;
    }

    private void validateResponderCertStatus(OCSPResponderTrustResult result, X509Certificate chainCert, CryptoResult<CRLData> crlResult) {
        switch (crlResult.getData().getCertStatus()) {
            case VALID: {
                break;
            }
            case REVOKED: {
                LOGGER.warn((Object)"Certificate chain of OCSP Responder has been revoked. OCSP Response is NOT TRUSTED.");
                result.getErrors().add(NotificationError.OCSP_RESPONDER_CHAIN_REVOKED);
                break;
            }
            default: {
                LOGGER.warn((Object)("Revocation status of certificate " + chainCert.getSubjectDN() + " unknown. If the Root CA of this chain is trusted, this certificate will be accepted without validated revocation status."));
            }
        }
    }

    private void logDetails(X509Certificate endCert, BasicOCSPResp ocspResponse, List<X509CRL> crls) {
        LOGGER.debug((Object)"Verify OCSPResponder trust");
        if (LOGGER.isDebugEnabled()) {
            StringBuffer sb = new StringBuffer("OCSPResponder [");
            sb.append("Cert Subject=").append(endCert.getSubjectX500Principal());
            sb.append(", Cert Issuer=").append(endCert.getIssuerX500Principal());
            sb.append(", ProducedAt=").append(ocspResponse.getProducedAt());
            sb.append(", CRLs=[");
            for (X509CRL crl : crls) {
                sb.append("CRL[Issuer=" + crl.getIssuerX500Principal() + ", NextUpdate=" + crl.getNextUpdate() + "], ");
            }
            sb.append("]");
            LOGGER.debug((Object)sb.toString());
        }
    }

    private void verifyChain(X509Certificate[] ocspResponseCertList) throws CertificateException, KeyStoreException, NoSuchAlgorithmException, NoSuchProviderException {
        if (this.trustStore != null) {
            X509Certificate ocspCert;
            List<X509Certificate> ocspResponseCertChain = Arrays.asList(ocspResponseCertList);
            if (!ocspResponseCertChain.isEmpty() && KeyManager.isTrustStoreCert(this.trustStore, ocspCert = ocspResponseCertChain.get(0))) {
                ocspCert.checkValidity();
                LOGGER.info((Object)"Valid Signing Certificate of OCSPResponse found in trustStore. Certificate chain accepted.");
                return;
            }
            CryptoResult<CertificateStatus> certPathResult = this.certPathChecker.validate(ocspResponseCertChain);
            if (certPathResult.hasErrors()) {
                LOGGER.warn((Object)("CertPath Validation completed with errors: " + certPathResult));
                throw new CertificateException("CertPath of OCSPResponse could not be verified.");
            }
            if (certPathResult.hasWarnings()) {
                LOGGER.warn((Object)("CertPath Validation completed with warnings: " + certPathResult.getWarnings()));
            }
            this.validateResponderCertChain(certPathResult);
        }
    }

    private void validateResponderCertChain(CryptoResult<CertificateStatus> certPathResult) throws CertificateException {
        CertificateStatus certStatus = certPathResult.getData();
        switch (certStatus) {
            case VALID: {
                break;
            }
            case EXPIRED: {
                throw new CertificateExpiredException("OCSP Certificatechain is EXPIRED.");
            }
            default: {
                throw new CertificateException("The certificate chain of the OSCP response is NOT TRUSTED.");
            }
        }
    }

    private X509Certificate[] getCertificatesFromResponse(BasicOCSPResp basicOcspResp) throws CertificateException {
        X509CertificateHolder[] certHolderList = basicOcspResp.getCerts();
        if (certHolderList == null || certHolderList.length == 0) {
            throw new CertificateException("OCSPResponse does not contain a certificate.");
        }
        X509Certificate[] certs = new X509Certificate[certHolderList.length];
        for (int i = 0; i < certHolderList.length; ++i) {
            certs[i] = this.extractCertificate(certHolderList[i]);
        }
        this.validateSignature(basicOcspResp, certs[0]);
        return certs;
    }

    private X509Certificate extractCertificate(X509CertificateHolder certificateHolder) throws CertificateException {
        X509Certificate cert = new JcaX509CertificateConverter().getCertificate(certificateHolder);
        return (X509Certificate)CertificateFactory.getInstance("X.509").generateCertificate(new ByteArrayInputStream(cert.getEncoded()));
    }

    private void validateSignature(BasicOCSPResp basicOcspResp, X509Certificate cert) throws CertificateException {
        try {
            JcaContentVerifierProviderBuilder cvpBuilder = new JcaContentVerifierProviderBuilder();
            ContentVerifierProvider cvp = cvpBuilder.build(cert);
            if (!basicOcspResp.isSignatureValid(cvp)) {
                throw new CertificateException("The OSCP response is not signed with given certificate.");
            }
        }
        catch (OperatorCreationException e) {
            throw new CertificateException("Exception occurred when preparing to verify the response's signature.", e);
        }
        catch (org.bouncycastle.cert.ocsp.OCSPException e) {
            throw new CertificateException("Exception occurred when verifying the response's signature.", e);
        }
    }
}

