﻿/*
 * Copyright (c) eHealth
 */
using System;
using NUnit.Framework;

using java.lang;
using java.util;
using java.io;

using be.ehealth.technicalconnector.session;
using be.fgov.ehealth.businessconnector.tests.utils;
using be.fgov.ehealth.technicalconnector.tests.utils;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.ehealth.technicalconnector.exception;
using be.ehealth.technicalconnector.utils;

using be.fgov.ehealth.commons.core.v2;
using be.fgov.ehealth.mediprima.uma.core.v1;
using be.fgov.ehealth.mediprima.uma.protocol.v1;

using be.ehealth.businessconnector.mediprimauma.exception;

using javax.xml.soap;
using javax.xml.ws.soap;
using javax.xml.transform;
using javax.xml.transform.dom;
using javax.xml.transform.stream;

using org.apache.commons.compress.utils;

namespace be.ehealth.businessconnector.mediprimauma.session
{
	[TestFixture]
	public class MediprimaUmaIntegrationTest
	{
		private enum RequestType {SEND, SEARCH, DELETE};
	    
		private static string attestationNumber;
	    
		private static org.joda.time.DateTime ISSUE_INSTANT = new org.joda.time.DateTime();

	    private static org.joda.time.DateTime END_DATE = new org.joda.time.DateTime().plusDays(1);
		
	    /* Static methods */
	    
	    [SetUpAttribute]
		public static void setupsession() {
			SessionInitializer.init(".\\be.ehealth.businessconnector.mediprimauma.test.properties", true, SupportedLanguages.NET.getAbbreviation());
		}
	    
	    [TearDownAttribute]
		public static void teardownSession() {
			Session.getInstance().unloadSession();
		}
	    
	    /* Test methods */
	    
	    [Test]
	    public void _001_sendUrgentMedicalAidAttestation() {
	    	sendUrgentMedicalAidAttestation("19648339550", "80090116494", "hospitalization", ".\\examples\\mediprimauma\\expected\\response\\send.xml");
	    }
	    
	    [Test]
	    public void _002_searchUrgentMedicalAidAttestation() {
	        searchUrgentMedicalAidAttestation("80090116494", attestationNumber, ".\\examples\\mediprimauma\\expected\\response\\search.xml");
	    }
	    
	    [Test]
	    public void _003_deleteUrgentMedicalAidAttestation() {
	        deleteUrgentMedicalAidAttestation("19648339550", "80090116494", attestationNumber, ".\\examples\\mediprimauma\\expected\\response\\delete.xml");
	    }
	    
	    [Test]
	    public void _004_invalidNihii_sendUrgentMedicalAidAttestation() {
	        sendUrgentMedicalAidAttestation("84022611609", "80090116494", "hospitalization", ".\\examples\\mediprimauma\\expected\\error\\send_invalid_nihii.xml");
	    }
	    
	    [Test]
	    public void _005_invalidSsin_sendUrgentMedicalAidAttestation() {
	        sendUrgentMedicalAidAttestation("19648339550", "12345678901", "hospitalization", ".\\examples\\mediprimauma\\expected\\error\\send_invalid_niss.xml");
	    }
	    
	    [Test]
	    public void _006_invalidMedicalCover_sendUrgentMedicalAidAttestation() {
	        sendUrgentMedicalAidAttestation("19648339550", "80090116494", "something_wrong", ".\\examples\\mediprimauma\\expected\\error\\send_invalid_medical_cover.xml");
	    }
	    
	    [Test]
	    public void _007_noNihii_sendUrgentMedicalAidAttestation() {
	        sendUrgentMedicalAidAttestation(null, "80090116494", "hospitalization", ".\\examples\\mediprimauma\\expected\\error\\send_no_nihii.xml");
	    }
	    
	    [Test]
	    public void _008_noSsin_sendUrgentMedicalAidAttestation() {
	        sendUrgentMedicalAidAttestation("19648339550", null, "hospitalization", ".\\examples\\mediprimauma\\expected\\error\\send_no_ssin.xml");
	    }
	    
	    [Test]
	    public void _009_noMedicalCover_sendUrgentMedicalAidAttestation() {
	        sendUrgentMedicalAidAttestation("19648339550", "80090116494", null, ".\\examples\\mediprimauma\\expected\\error\\send_no_medical_cover.xml");
	    }
	    
	    [Test]
	    public void _010_invalidNihii_searchUrgentMedicalAidAttestation() {
	        searchUrgentMedicalAidAttestation("84022611609", "690-HOSP-180000004548", ".\\examples\\mediprimauma\\expected\\error\\search_invalid_nihii.xml");
	    }
	    
	    [Test]
	    public void _011_invalidAttestationNumber_searchUrgentMedicalAidAttestation() {
	        searchUrgentMedicalAidAttestation("80090116494", "SOME_WRONG_ATTEST_NBR", ".\\examples\\mediprimauma\\expected\\error\\search_invalid_attestation_number.xml");
	    }
	    
	    [Test]
	    public void _012_noNihii_searchUrgentMedicalAidAttestation() {
	        searchUrgentMedicalAidAttestation(null, "690-HOSP-180000004548", ".\\examples\\mediprimauma\\expected\\error\\search_no_nihii.xml");
	    }
	    
	    [Test]
	    public void _013_noAttestationNumber_searchUrgentMedicalAidAttestation() {
	        searchUrgentMedicalAidAttestation("80090116494", null, ".\\examples\\mediprimauma\\expected\\error\\search_no_attestation_number.xml");
	    }
	    
	    [Test]
	    public void _014_invalidNihii_deleteUrgentMedicalAidAttestation() {
	        deleteUrgentMedicalAidAttestation("84022611609", "80090116494", "690-HOSP-180000004548", ".\\examples\\mediprimauma\\expected\\error\\delete_invalid_nihii.xml");
	    }
	    
	    [Test]
	    public void _015_invalidSsin_deleteUrgentMedicalAidAttestation() {
	        deleteUrgentMedicalAidAttestation("19648339550", "12345678901", "690-HOSP-180000004548", ".\\examples\\mediprimauma\\expected\\error\\delete_invalid_ssin.xml");
	    }
	    
	    [Test]
	    public void _016_invalidAttestationNumber_deleteUrgentMedicalAidAttestation() {
	        deleteUrgentMedicalAidAttestation("19648339550", "80090116494", "SOME_WRONG_ATTEST_NBR", ".\\examples\\mediprimauma\\expected\\error\\delete_invalid_attestation_number.xml");
	    }
	    
	    [Test]
	    public void _017_noNihii_deleteUrgentMedicalAidAttestation() {
	        deleteUrgentMedicalAidAttestation(null, "80090116494", "690-HOSP-180000004548", ".\\examples\\mediprimauma\\expected\\error\\delete_no_nihii.xml");
	    }
	    
	    [Test]
	    public void _018_noSsin_deleteUrgentMedicalAidAttestation() {
	        deleteUrgentMedicalAidAttestation("19648339550", null, "690-HOSP-180000004548", ".\\examples\\mediprimauma\\expected\\error\\delete_no_ssin.xml");
	    }
	    
	    [Test]
	    public void _019_noAttestationNumber_deleteUrgentMedicalAidAttestation() {
	        deleteUrgentMedicalAidAttestation("19648339550", "80090116494", null, ".\\examples\\mediprimauma\\expected\\error\\delete_no_attestation_number.xml");
	    }
	    
	    /* Private methods */
	    
	    private void deleteUrgentMedicalAidAttestation(string authorNihii, string beneficiarySsin, string attestationNumber, string expectedResponseTemplate) {
	        sendRequest(createDeleteRequest(authorNihii, beneficiarySsin, attestationNumber), expectedResponseTemplate, RequestType.DELETE);
	    }
	    
	    private void searchUrgentMedicalAidAttestation(string beneficiarySsin, string attestationNumber, string expectedResponseTemplate) {
	        sendRequest(createSearchRequest(beneficiarySsin, attestationNumber), expectedResponseTemplate, RequestType.SEARCH);
	    }
	    
	    private void sendUrgentMedicalAidAttestation(string authorNihii, string beneficiarySsin, string medicalCover, string expectedResponseTemplate) {
	        sendRequest(createSendRequest(authorNihii, beneficiarySsin, medicalCover), expectedResponseTemplate, RequestType.SEND);
	    }
	    
	    private void sendRequest(object request, string expectedResponseTemplate, RequestType type) {
	    	object response = null;
        	
	        try {
	            MediprimaUmaService service = MediprimaUmaSessionServiceFactory.getMediprimaUmaSession();
	            
	            switch(type) {
	                case RequestType.SEND:
	                    SendUrgentMedicalAidAttestationResponse serviceResponse = service.sendUrgentMedicalAidAttestation((SendUrgentMedicalAidAttestationRequest) request);
	                    attestationNumber = serviceResponse.getAttestion().getAttestationNumber();
	                    response = serviceResponse;
	                break;
	                
	                case RequestType.SEARCH:
	                    response = service.searchUrgentMedicalAidAttestation((SearchUrgentMedicalAidAttestationRequest) request);
	                break;
	                
	                case RequestType.DELETE:
	                    response = service.deleteUrgentMedicalAidAttestation((DeleteUrgentMedicalAidAttestationRequest) request);
	                break;
	                
	                default: 
	                    throw new IllegalArgumentException("Unknown request type");
	            }
	            
	            Assert.IsNotNull(response);
	        } catch (MediprimaUmaSendException e1) {
	            response = e1.getResponse();
	            Assert.IsNotNull(e1.getRegistryStatus());
	        } catch (MediprimaUmaSearchException e2) {
	            response = e2.getResponse();
	            Assert.IsNotNull(e2.getRegistryStatus());
	        } catch (MediprimaUmaDeleteException e3) {
	            response = e3.getResponse();
	            Assert.IsNotNull(e3.getRegistryStatus());
	        } catch (SOAPFaultException e4) {
            	response = ConnectorXmlUtils.toString(e4.getMessage());
		    } catch (TechnicalConnectorException e5) {
		        response = ConnectorXmlUtils.toString(e5.getCause().Message);
		    }
	        
	        string expectedResponse = ConnectorIOUtils.getResourceAsString(expectedResponseTemplate);
	        string actualResponse = response is string? (string) response : ConnectorXmlUtils.toString(response);
	        be.fgov.ehealth.businessconnector.tests.utils.XmlAsserter.assertSimilar(expectedResponse, actualResponse);
	    }
	    
	    
	    private DeleteUrgentMedicalAidAttestationRequest createDeleteRequest(string authorNihii, string beneficiarySsin, string attestationNumber) {
	        DeleteUrgentMedicalAidAttestationRequest request = new DeleteUrgentMedicalAidAttestationRequest();
	        request.setAuthor(createAuthor(authorNihii));
	        request.setAttestationNumber(attestationNumber);
	        request.setBeneficiarySsin(beneficiarySsin);
	        request.setIssueInstant(ISSUE_INSTANT);
	        request.setId(generateId());
	        return request;
	    }
	    
	    private SearchUrgentMedicalAidAttestationRequest createSearchRequest(string beneficiarySsin, string attestationNumber) {
	        SearchUrgentMedicalAidAttestationRequest request = new SearchUrgentMedicalAidAttestationRequest();
	        CriteriaType criteria = new CriteriaType();
	        criteria.setAttestationNumber(attestationNumber);
	        criteria.setBeneficiarySsin(beneficiarySsin);
	        request.setCriteria(criteria);
	        request.setIssueInstant(ISSUE_INSTANT);
	        request.setId(generateId());
	        return request;
	    }
	    
	    private SendUrgentMedicalAidAttestationRequest createSendRequest(string authorNihii, string beneficiarySsin, string medicalCover) {
	        SendUrgentMedicalAidAttestationRequest request = new SendUrgentMedicalAidAttestationRequest();
	        request.setAuthor(createAuthor(authorNihii));
	        request.setBeneficiarySsin(beneficiarySsin);
	        request.setIssueInstant(ISSUE_INSTANT);
	        request.setMedicalCover(medicalCover);
	        request.setValidityPeriod(createPeriod());
	        request.setId(generateId());
	        return request;
	    }
	    
	    private Author createAuthor(string nihii) {
	        Author author = new Author();
	        ActorType actor = new ActorType();
	        actor.getFirstNames().add("Jean");
	        actor.setName("Neymar");
	        Id id = new Id();
	        id.setType("urn:be:fgov:ehealth:1.0:physician:nihii-number");
	        id.setValue(nihii);
	        actor.getIds().add(id);
	        author.getHcParties().add(actor);
	        return author;
	    }
	    
	    private PeriodType createPeriod() {
	        PeriodType period = new PeriodType();
	        period.setStartDate(ISSUE_INSTANT);
	        period.setEndDate(END_DATE);
	        return period;
	    }
	    
	    private string generateId() {
	        return "ID_" + UUID.randomUUID().toString();
	    }
	    
	}
	
}
