/*
 * Copyright (c) eHealth
 */
package be.ehealth.technicalconnector.config.impl;

import static be.ehealth.technicalconnector.config.impl.ConfigurationModuleSystemProps.SYSTEMPROP_ROOTKEY;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.junit.Assert.assertThat;

import java.io.File;
import java.util.UUID;

import be.ehealth.technicalconnector.shutdown.DeleteFileOnExitShutdownHook;
import org.apache.commons.io.FileUtils;
import org.junit.Rule;
import org.junit.Test;

import be.ehealth.technicalconnector.config.ConfigFactory;
import be.ehealth.technicalconnector.config.ConfigValidator;
import be.ehealth.technicalconnector.config.ConfigurationModule;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.RestoreSystemProperties;


/**
 * Test configuration module {@link ConfigurationModuleSystemProps}
 *
 * @author EHP
 */
public class ConfigurationModuleSystemPropsTest {

    private static final String LF = System.getProperty("line.separator");

    private ConfigValidator config = ConfigFactory.getConfigValidator();

    private ConfigurationModule module = new ConfigurationModuleSystemProps();

    @Rule
    public RestoreSystemProperties restoreSystemProperties = new RestoreSystemProperties();

    @Test
    public void newPropertyWithValue() throws Exception {
        String uuid = UUID.randomUUID()
                          .toString();

        initConfig("connector.uuid=" + uuid);

        module.init(config.getConfig());

        assertThat(System.getProperty("connector.uuid"), equalTo(uuid));

        module.unload();

        assertThat(System.getProperty("connector.uuid"), nullValue());

    }

    @Test
    public void newPropertyWithoutValue() throws Exception {
        initConfig("connector.uuid=");

        module.init(config.getConfig());

        assertThat(System.getProperty("connector.uuid"), equalTo(""));

        module.unload();

        assertThat(System.getProperty("connector.uuid"), nullValue());

    }

    @Test
    public void existingPropertyWithValue() throws Exception {

        String uuid_01 = UUID.randomUUID()
                             .toString();
        System.setProperty("connector.uuid", uuid_01);

        String uuid_02 = UUID.randomUUID()
                             .toString();

        initConfig("connector.uuid=" + uuid_02);

        module.init(config.getConfig());

        assertThat(System.getProperty("connector.uuid"), equalTo(uuid_02));

        module.unload();

        assertThat(System.getProperty("connector.uuid"), equalTo(uuid_01));

    }

    @Test
    public void existingPropertyWithoutValue() throws Exception {

        String uuid_01 = UUID.randomUUID()
                             .toString();
        System.setProperty("connector.uuid", uuid_01);

        initConfig("connector.uuid=");

        module.init(config.getConfig());

        assertThat(System.getProperty("connector.uuid"), equalTo(""));

        module.unload();

        assertThat(System.getProperty("connector.uuid"), equalTo(uuid_01));

    }

    @Test
    public void doubleDelimiter() throws Exception {
        initConfig("double.delimiter=6=5");
        module.init(config.getConfig());

        assertThat(System.getProperty("double.delimiter"), equalTo("6=5"));

        module.unload();


        assertThat(System.getProperty("double.delimiter"), nullValue());

    }


    @Test(expected = IllegalArgumentException.class)
    public void noDelimiter() throws Exception {
        initConfig("connector.nodelimiter");

        module.init(config.getConfig());

    }

    private static void initConfig(String... propertyBases) throws Exception {
        File tempfile = File.createTempFile("systemprops", ".properties");
        DeleteFileOnExitShutdownHook.deleteOnExit(tempfile);
        int counter = 1;
        for (int i = 0; i < propertyBases.length; i++) {
            FileUtils.write(tempfile, SYSTEMPROP_ROOTKEY + "." + counter + "=" + propertyBases[i] + LF, true);
            counter++;
        }
        ConfigFactory.setConfigLocation(tempfile.getAbsolutePath());
    }


}
