/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.dics.session;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import org.joda.time.DateTime;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.technicalconnector.idgenerator.IdGeneratorFactory;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.fgov.ehealth.dics.protocol.v1.FindCNKRequest;
import be.fgov.ehealth.dics.protocol.v1.FindCNKResponse;
import be.fgov.ehealth.dics.protocol.v1.FindParagraphRequest;
import be.fgov.ehealth.dics.protocol.v1.FindParagraphResponse;
import be.fgov.ehealth.dics.protocol.v1.FindParagraphTextRequest;
import be.fgov.ehealth.dics.protocol.v1.FindParagraphTextResponse;
import be.fgov.ehealth.dics.protocol.v1.FindReimbursementConditionsRequest;
import be.fgov.ehealth.dics.protocol.v1.FindReimbursementConditionsResponse;
import be.fgov.ehealth.dics.protocol.v1.GetAddedDocumentsRequest;
import be.fgov.ehealth.dics.protocol.v1.GetAddedDocumentsResponse;
import be.fgov.ehealth.dics.protocol.v1.GetParagraphExclusionsRequest;
import be.fgov.ehealth.dics.protocol.v1.GetParagraphExclusionsResponse;
import be.fgov.ehealth.dics.protocol.v1.GetParagraphIncludedSpecialtiesRequest;
import be.fgov.ehealth.dics.protocol.v1.GetParagraphIncludedSpecialtiesResponse;
import be.fgov.ehealth.dics.protocol.v1.GetProfessionalAuthorizationsRequest;
import be.fgov.ehealth.dics.protocol.v1.GetProfessionalAuthorizationsResponse;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;

/**
 * Integration tests for tarification.
 * 
 * @author eHealth Platform
 * 
 */
public final class DicsIntegrationTest {

    /**
     * @param oa
     */
    public DicsIntegrationTest() {
        super();
    }


    private static final Logger LOG = LoggerFactory.getLogger(DicsIntegrationTest.class);

    @BeforeClass
    public static void setupsession() throws Exception {
        LOG.debug("DicsIntegrationTest.loadSession:");
        SessionInitializer.init("/be.ehealth.businessconnector.dics.test.properties");
    }

    @AfterClass
    public static void teardownSession() throws Exception {
        Session.getInstance().unloadSession();
    }

    @Test
    public void findCNKRequestTest() throws Exception {
        FindCNKRequest request = new FindCNKRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setLang("FR");
        request.setProductName("PERD");
        request.setStartDate(new DateTime(2012, 10, 1, 0, 0, 0, 0));


        FindCNKResponse response = DicsSessionServiceFactory.getTarificationSession().findCNK(request);


        logResponse(response, FindCNKResponse.class);
        Assert.assertNotNull(response.getProducts());
        Assert.assertNotEquals(0, response.getProducts().size());
    }

    @Test
    public void findParagraphRequestTest() throws Exception {
        FindParagraphRequest request = new FindParagraphRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setAmppId(2447803L);

        FindParagraphResponse response = DicsSessionServiceFactory.getTarificationSession().findParagraph(request);

        logResponse(response, FindParagraphResponse.class);
        Assert.assertNotNull(response.getFindParagraphs());
        Assert.assertNotEquals(0, response.getFindParagraphs().size());
    }

    @Test
    public void findParagraphTextRequestTest() throws Exception {
        FindParagraphTextRequest request = new FindParagraphTextRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setLang("FR");
        request.setChapterName("IV");
        request.setParagraphName("30200");
        request.setStartDate(new DateTime(2012, 9, 1, 0, 0, 0, 0));

        FindParagraphTextResponse response = DicsSessionServiceFactory.getTarificationSession().findParagraphText(request);

        logResponse(response, FindParagraphTextResponse.class);
        Assert.assertNotNull(response.getParagraph());
    }

    @Test
    public void findReimbursementConditionsRequestTest() throws Exception {
        FindReimbursementConditionsRequest request = new FindReimbursementConditionsRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setLang("FR");
        request.setChapterName("IV");
        request.setParagraphName("170000");
        request.setStartDate(new DateTime(2012, 9, 1, 0, 0, 0, 0));

        FindReimbursementConditionsResponse response = DicsSessionServiceFactory.getTarificationSession().findReimbursementConditions(request);

        logResponse(response, FindReimbursementConditionsResponse.class);
        Assert.assertNotNull(response.getParagraphAndTherapies());
        Assert.assertNotEquals(0, response.getParagraphAndTherapies().size());
    }

    @Test
    public void getAddedDocumentsRequestTest() throws Exception {
        GetAddedDocumentsRequest request = new GetAddedDocumentsRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setLang("FR");
        request.setChapterName("IV");
        request.setParagraphName("1750000");

        GetAddedDocumentsResponse response = DicsSessionServiceFactory.getTarificationSession().getAddedDocuments(request);

        logResponse(response, GetAddedDocumentsResponse.class);
        Assert.assertNotNull(response.getAddedDocuments());
        Assert.assertNotEquals(0, response.getAddedDocuments().size());
    }

    @Test
    public void getParagraphExclusionsRequestTest() throws Exception {
        GetParagraphExclusionsRequest request = new GetParagraphExclusionsRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setLang("FR");
        request.setChapterName("IV");
        request.setParagraphName("1570000");
        request.setStartDate(new DateTime(2012, 10, 1, 0, 0, 0, 0));

        GetParagraphExclusionsResponse response = DicsSessionServiceFactory.getTarificationSession().getParagraphExclusions(request);

        logResponse(response, GetParagraphExclusionsResponse.class);
        Assert.assertNotNull(response.getExcludedParagraphs());
        Assert.assertNotEquals(0, response.getExcludedParagraphs().size());
    }

    @Test
    public void getParagraphIncludedSpecialtiesRequestTest() throws Exception {
        GetParagraphIncludedSpecialtiesRequest request = new GetParagraphIncludedSpecialtiesRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setLang("FR");
        request.setChapterName("IV");
        request.setParagraphName("40100");
        request.setStartDate(new DateTime(2012, 10, 1, 0, 0, 0, 0));

        GetParagraphIncludedSpecialtiesResponse response = DicsSessionServiceFactory.getTarificationSession().getParagraphIncludedSpecialities(request);

        logResponse(response, GetParagraphIncludedSpecialtiesResponse.class);
        Assert.assertNotNull(response.getAtmAndChildrens());
        Assert.assertNotEquals(0, response.getAtmAndChildrens().size());
    }

    @Test
    public void getProfessionalAuthorizationsRequestTest() throws Exception {
        GetProfessionalAuthorizationsRequest request = new GetProfessionalAuthorizationsRequest();
        request.setId(IdGeneratorFactory.getIdGenerator(IdGeneratorFactory.XSID).generateId());
        request.setIssueInstant(new DateTime());
        request.setLang("FR");
        request.setQualificationListId("121");

        GetProfessionalAuthorizationsResponse response = DicsSessionServiceFactory.getTarificationSession().getProfessionalAuthorizations(request);

        logResponse(response, GetProfessionalAuthorizationsResponse.class);
        Assert.assertNotNull(response.getProfessionalAuthorizations());
        Assert.assertNotEquals(0, response.getProfessionalAuthorizations().size());
    }


    /**
     * @param response
     * @throws Exception
     */
    @SuppressWarnings({
        "rawtypes", "unchecked"
    })
    private void logResponse(Object response, Class classToUnmarshall) throws Exception {
        MarshallerHelper<Object, Object> responseMarshaller;
        responseMarshaller = new MarshallerHelper(classToUnmarshall, classToUnmarshall);
        String unmarshalledString = responseMarshaller.toString(response);
        LOG.debug("response : " + unmarshalledString);
    }


    /**
     * @param result
     * @param fileName
     * @throws FileNotFoundException
     * @throws IOException
     */
    @SuppressWarnings("unused")
    private void saveFileValue(byte[] fileValue, String fileName) throws FileNotFoundException, IOException {
        FileOutputStream fos = new FileOutputStream(fileName);
        fos.write(fileValue);
        fos.close();
    }

}
