/*
 * Copyright (c) Smals
 */
package be.ehealth.businessconnector.test.ehbox.v3.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.junit.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import be.ehealth.businessconnector.ehbox.v3.builders.BuilderFactory;
import be.ehealth.businessconnector.ehbox.v3.session.EhealthBoxServiceV3;
import be.ehealth.businessconnector.ehbox.v3.session.ServiceFactory;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.DeleteMessageRequest;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.DeleteMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.DeleteOoOResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetAllEhboxesMessagesListResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetFullMessageResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetMessageAcknowledgmentsStatusResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetMessagesListResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.GetOoOListResponse;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.Message;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.OoO;
import be.fgov.ehealth.ehbox.core.v3.BoxIdType;
import be.fgov.ehealth.technicalconnector.tests.utils.TestPropertiesLoader;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * utility class to use in tests.
 *
 * @author EHP
 */
public final class EhboxTestUtilities {


    private static final Logger LOG = LoggerFactory.getLogger(EhboxTestUtilities.class);

    private static final String PROPERTIES_FILEPATH = "/be.ehealth.businessconnector.ehboxv3.test.properties";

    private static final int START = 1;

    private static final int END = 100;

    /**
     * only static methods needed.
     */
    private EhboxTestUtilities() {
        super();
    }

    public static DeleteMessageResponse deleteMessages(String source, String... messageIds) throws ConnectorException {
        DeleteMessageRequest request = BuilderFactory.getRequestBuilder().createDeleteMessageRequest(source, messageIds);
        EhealthBoxServiceV3 service = be.ehealth.businessconnector.ehbox.v3.session.ServiceFactory.getEhealthBoxServiceV3();
        DeleteMessageResponse response = service.deleteMessage(request);
        Assert.assertEquals("100", response.getStatus().getCode());
        return response;
    }

    public static DeleteMessageResponse deleteMessages(String source, BoxIdType boxId, String... messageIds) throws ConnectorException {
        DeleteMessageRequest request = BuilderFactory.getRequestBuilder().createDeleteMessageRequest(source, boxId, messageIds);
        EhealthBoxServiceV3 service = be.ehealth.businessconnector.ehbox.v3.session.ServiceFactory.getEhealthBoxServiceV3();
        DeleteMessageResponse response = service.deleteMessage(request);
        Assert.assertEquals("100", response.getStatus().getCode());
        return response;
    }

    /**
     * compares a jaxb object with the expected xml.
     *
     * @param fileLocation the file location of the expected xml file.
     * @param jaxbObject jaxbObject
     * @throws IOException
     * @throws SAXException
     * @throws TechnicalConnectorException
     */
    public static void compareXmlWithExpectedFile(String fileLocation, Object jaxbObject) throws SAXException, IOException, TechnicalConnectorException {
        Properties props = TestPropertiesLoader.getProperties(PROPERTIES_FILEPATH);
        Map<String, Object> ctx = new HashMap<String, Object>();
        ctx.put("senderId", props.getProperty("test.sender.id"));
        String expected = TemplateEngineUtils.generate(ctx, fileLocation);
        String actualXml = ConnectorXmlUtils.toString(jaxbObject);
        XmlAsserter.assertSimilar(expected, actualXml);

    }

    public static void logStackTrace(Throwable e) {
        StackTraceElement[] stackTrace = e.getStackTrace();
        StringBuilder sb = new StringBuilder();
        sb.append(e.getMessage()).append("\n");
        for (StackTraceElement stackTraceElement : stackTrace) {
            sb.append(stackTraceElement.getClassName()).append(".").append(stackTraceElement.getMethodName());
            sb.append(":").append(stackTraceElement.getLineNumber()).append("\n");
        }
        LOG.debug("logStackTrace : \n" + sb);
    }

    /**
     * @throws ConnectorException
     */
    public static void deleteAllOutOfOfficesForCurrentUser(BoxIdType boxId) throws ConnectorException {
        EhealthBoxServiceV3 service = ServiceFactory.getEhealthBoxServiceV3();
        GetOoOListResponse ooOList = service.getOoOList(BuilderFactory.getRequestBuilder().createGetOoOListRequest(boxId));
        List<String> oOoIdList = new ArrayList<String>();
        for (OoO oOo : ooOList.getOoOs()) {
            oOoIdList.add(oOo.getOoOId());
            LOG.debug("deleteAllOutOfOfficesForCurrentUser : adding delete id for oOo" + oOoToString(oOo));
        }
        if (!oOoIdList.isEmpty()) {
            DeleteOoOResponse deleteOoOResponse = service.deleteOoO(BuilderFactory.getRequestBuilder().createDeleteOoORequest(boxId, oOoIdList));
            Assert.assertEquals("delete oo should have succeeded", "100", deleteOoOResponse.getStatus().getCode());
        } else {
            LOG.debug("deleteAllOutOfOfficesForCurrentUser : no Out off offices found");
        }
    }

    /**
     * @param oOo
     * @return
     */
    private static String oOoToString(OoO oOo) {
        StringBuilder sb = new StringBuilder();
        sb.append("[ id:)").append(oOo.getOoOId());
        sb.append("StartDate").append(oOo.getStartDate());
        sb.append("EndDate").append(oOo.getEndDate());
        for (BoxIdType boxId : oOo.getSubstitutes()) {
            sb.append(" boxId:(").append(boxIdToString(boxId)).append(")");
        }
        return sb.toString();
    }

    /**
     * @param boxId
     * @return
     */
    private static Object boxIdToString(BoxIdType boxId) {
        StringBuilder sb = new StringBuilder();
        sb.append(" [").append("id:").append(boxId.getId());
        sb.append(", quality:").append(boxId.getQuality());
        sb.append(", type:").append(boxId.getType());
        sb.append(", subType:").append(boxId.getSubType());
        sb.append("]");
        return sb.toString();
    }

    /**
     * @return
     */
    public static List<String> getListOfAllSources() {
        List<String> sources = new ArrayList<String>();
        sources.add("INBOX");
        sources.add("SENTBOX");
        sources.add("BININBOX");
        sources.add("BINSENTBOX");
        return sources;
    }

    /**
     * checks properties to see if these are tests for persphysician.
     *
     * @param testConfigProperties
     * @return
     */
    public static boolean isDoctorTest(Properties testConfigProperties) {
        return "persphysician".equals(testConfigProperties.getProperty("session.professionType"));
    }

    public static void waitForProcessing(int waitPeriodInMilliSeconds) {
        LOG.debug("sleep " + waitPeriodInMilliSeconds + " milliseconds to allow ehbox to process the message");
        try {
            Thread.sleep(waitPeriodInMilliSeconds);
        } catch (InterruptedException e) {
            // just go on
        }
    }

    /**
     * tells if the given messageID is in the all ehboxes list of message retrieved from the given source of the given boxID
     *
     * @param messageId
     * @param source
     * @return
     * @throws Exception
     */
    public static boolean isMessageInAllEhBoxesList(String messageId, String source) throws Exception {
        GetAllEhboxesMessagesListResponse list = ServiceFactory.getEhealthBoxServiceV3().getAllEhboxesMessagesList(BuilderFactory.getRequestBuilder().createAllEhboxesMessagesListRequest(source));
        for (Message message : list.getMessages()) {
            if (message.getMessageId().equals(messageId)) {
                return true;
            }
        }
        return false;
    }

    /**
     * verifies if the given messageID is in the list of message retrieved from the given source of the given boxID
     *
     * @param messageId
     * @param source
     * @param boxId
     * @return
     * @throws Exception
     */
    public static boolean isMessageInList(String messageId, String source, BoxIdType boxId) throws Exception {
        GetMessagesListResponse list = ServiceFactory.getEhealthBoxServiceV3().getMessageList(BuilderFactory.getRequestBuilder().createGetMessagesListRequest(source, START, END, boxId));
        for (Message inboxMessage : list.getMessages()) {
            if (inboxMessage.getMessageId().equals(messageId)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Retrieves the full message corresponding to the given messageID from the given source of the given boxID
     *
     * @param messageId
     * @param source
     * @param boxId
     * @return
     * @throws Exception
     */
    public static GetFullMessageResponse getFullMessage(String messageId, String source, BoxIdType boxId) throws Exception {
        return ServiceFactory.getEhealthBoxServiceV3().getFullMessage(BuilderFactory.getRequestBuilder().createGetFullMessageRequest(messageId, source, boxId));
    }

    /**
     * Retrieves the acknowledgements of the message corresponding to the messageID from the given source of the given boxID
     *
     * @param messageId
     * @param boxId
     * @return
     * @throws Exception
     */
    public static GetMessageAcknowledgmentsStatusResponse getMessageAcknowledgements(String messageId, BoxIdType boxId) throws Exception {
        return ServiceFactory.getEhealthBoxServiceV3().getMessageAcknowledgmentsStatusRequest(BuilderFactory.getRequestBuilder().createGetMessageAcknowledgmentsStatusRequest(messageId, START, END, boxId));
    }

}
