/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.ocsp;

import be.fgov.ehealth.etee.crypto.ocsp.OCSPClient;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPConnector;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPException;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPRequestBuilder;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPURLReader;
import be.fgov.ehealth.etee.crypto.utils.Preconditions;
import be.fgov.ehealth.etee.crypto.utils.TimeFrameValidator;
import java.math.BigInteger;
import java.net.URI;
import java.security.cert.X509Certificate;
import java.util.Date;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.OCSPReq;
import org.bouncycastle.cert.ocsp.OCSPResp;
import org.bouncycastle.cert.ocsp.SingleResp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OCSPClientImpl
implements OCSPClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(OCSPClientImpl.class);
    private final OCSPConnector ocspRequestor;
    private final OCSPRequestBuilder ocspRequestBuilder;
    private final OCSPURLReader ocspUrlReader;
    private final TimeFrameValidator timeFrameValidator;

    OCSPClientImpl(OCSPConnector ocspRequestor, OCSPRequestBuilder ocspRequestBuilder, OCSPURLReader ocspUrlReader, long clockSkew) {
        Preconditions.checkNotNull(ocspRequestor, "ocspRequestor cannot be null");
        Preconditions.checkNotNull(ocspRequestBuilder, "ocspRequestBuilder cannot be null");
        this.ocspRequestor = ocspRequestor;
        this.ocspRequestBuilder = ocspRequestBuilder;
        this.ocspUrlReader = ocspUrlReader;
        this.timeFrameValidator = TimeFrameValidator.create(clockSkew);
    }

    @Override
    public BasicOCSPResp sendRequestAndCheckResponse(X509Certificate cert) throws OCSPException {
        URI ocspServiceURI = this.ocspUrlReader.getOcspServiceLocation(cert);
        LOGGER.debug("Sending OCSP request to " + ocspServiceURI);
        OCSPReq ocspRequest = this.ocspRequestBuilder.generateOCSPRequest(cert);
        OCSPResp ocspResp = this.ocspRequestor.send(ocspRequest, ocspServiceURI);
        this.checkOcspResponseStatus(ocspResp);
        try {
            BasicOCSPResp basicOcspResp = (BasicOCSPResp)ocspResp.getResponseObject();
            this.checkOcspResponseNonce(basicOcspResp, ocspRequest);
            this.validateContent(basicOcspResp.getResponses());
            return basicOcspResp;
        }
        catch (org.bouncycastle.cert.ocsp.OCSPException ocspException) {
            throw new OCSPException("The OCSPResponse could not be read.", ocspException);
        }
    }

    private void checkOcspResponseStatus(OCSPResp ocspResponse) throws OCSPException {
        switch (ocspResponse.getStatus()) {
            case 0: {
                LOGGER.info("OCSP Response Status is successful");
                break;
            }
            case 1: {
                throw new IllegalArgumentException("The OCSP request is malformed and not accepted by the OCSP service.");
            }
            case 6: {
                throw new OCSPException("The OCSP request is not authorized by the OCSP service.");
            }
            case 3: {
                throw new OCSPException("The OCSP service is busy and could not process the OCSP request now.");
            }
            case 2: {
                throw new OCSPException("An internal error occurred in the OCSP service.");
            }
            case 5: {
                throw new OCSPException("The OCSP request is declined by the OCSP service because it must be signed.");
            }
            default: {
                throw new OCSPException("The OCSP response contains an invalid Status code.");
            }
        }
    }

    private void checkOcspResponseNonce(BasicOCSPResp basicOcspResp, OCSPReq ocspRequest) throws OCSPException {
        Extension extension = basicOcspResp.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce);
        ASN1OctetString nonceString = extension.getExtnValue();
        byte[] respNonceBytes = nonceString.getOctets();
        BigInteger respNonce = new BigInteger(respNonceBytes);
        LOGGER.debug("OCSP response nonce : " + respNonce);
        extension = ocspRequest.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce);
        nonceString = extension.getExtnValue();
        byte[] reqNonceBytes = nonceString.getOctets();
        BigInteger reqNonce = new BigInteger(reqNonceBytes);
        LOGGER.debug("OCSP request nonce : " + reqNonce);
        if (!reqNonce.equals(respNonce)) {
            throw new OCSPException("The OCSP response is not to be trusted, a different nonce was found in the response.");
        }
        LOGGER.info("OCSP request nonce OK.");
    }

    private void validateContent(SingleResp[] singleResponses) throws OCSPException {
        if (singleResponses == null || singleResponses.length == 0) {
            throw new OCSPException("The OCSP Response did not return a single response on the certificate in question.");
        }
        Date now = new Date();
        for (SingleResp singleResp : singleResponses) {
            if (this.timeFrameValidator.validate(singleResp.getThisUpdate(), now, singleResp.getNextUpdate())) continue;
            throw new OCSPException("The timeframe of the OCSP Response is currently [" + now + "] acceptable [thisUpdate='" + singleResp.getThisUpdate() + "', nextUpdate='" + singleResp.getNextUpdate() + "'].");
        }
    }
}

