package be.ehealth.businessconnector.ehbox.v3.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.UUID;

import be.fgov.ehealth.ehbox.core.v3.CustomMetaType;
import org.hamcrest.core.StringContains;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.businessconnector.ehbox.api.domain.Addressee;
import be.ehealth.businessconnector.ehbox.api.domain.Document;
import be.ehealth.businessconnector.ehbox.api.domain.DocumentMessage;
import be.ehealth.businessconnector.ehbox.api.domain.NewsMessage;
import be.ehealth.businessconnector.ehbox.api.utils.QualityType;
import be.ehealth.businessconnector.ehbox.v3.EhboxTestDataBuilder;
import be.ehealth.businessconnector.ehbox.v3.builders.BuilderFactory;
import be.ehealth.businessconnector.ehbox.v3.builders.SendMessageBuilder;
import be.ehealth.businessconnector.ehbox.v3.session.Eh2eBoxServiceV3;
import be.ehealth.businessconnector.ehbox.v3.session.ServiceFactory;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.IdentifierType;
import be.fgov.ehealth.commons.core.v1.LocalisedString;
import be.fgov.ehealth.ehbox.consultation.protocol.v3.Message;
import be.fgov.ehealth.ehbox.publication.protocol.v3.SendMessageRequest;
import be.fgov.ehealth.ehbox.publication.protocol.v3.SendMessageResponse;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;
import be.fgov.ehealth.technicalconnector.tests.utils.TestPropertiesLoader;

/**
 * eHealthBox Service Integration Tests This test shows the use of the eHealthBox Service through the use of the Business Connector.
 * 
 * The tests below use the Session Management Service to manage the session and SAML/STS token. The eHealthBox of the test user specified in
 * the business connector property file is used, these tests expect that this is set to a valid INSS of a citizen.
 * 
 * @author EHP
 * 
 */
public class PublicationEh2EboxServiceIntegrationTest {

    private static Eh2eBoxServiceV3 service;

    @Rule
    public ExpectedException expectedEx = ExpectedException.none();

    @BeforeClass
    public static void initIntegrationTest() throws Exception {
        Properties props = TestPropertiesLoader.getProperties(FILE_AS_DOC);
        SessionInitializer.init(props, true);
        service = ServiceFactory.getEhealthBox2eBoxServiceV3();
    }

    @AfterClass
    public static void unloadSession() throws Exception {
        Session.getInstance().unloadSession();
    }

    /**
     * Properties file
     */
    private static final String FILE_AS_DOC = "/be.ehealth.businessconnector.ehboxv3.test.properties";

    private static final Logger LOG = LoggerFactory.getLogger(PublicationEh2EboxServiceIntegrationTest.class);

    /**
     * @throws Exception
     */
    @Test
    public void testSendDocumentWithAnnex() throws Exception {
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();
        documentMsg.setDestinations(getDestination());
        documentMsg.setImportant(false);
        documentMsg.generatePublicationId();

        documentMsg.setDocument(getDocument());
        documentMsg.getAnnexList().add(getAnnex());

        SendMessageRequest request = builder.buildMessage(documentMsg);
        request.getContentContext().getCustomMetas().add(getCustomMeta());

        verifyAnswer(service.sendMessage(request));
    }

    /**
     * @throws Exception
     */
    @Test
    public void testSendDocument() throws Exception {
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();
        documentMsg.setDestinations(getDestination());
        documentMsg.setImportant(false);
        documentMsg.generatePublicationId();

        documentMsg.setDocument(getDocument());

        SendMessageRequest request = builder.buildMessage(documentMsg);
        request.getContentContext().getCustomMetas().add(getCustomMeta());

        verifyAnswer(service.sendMessage(request));
    }

    /**
     * Test the sendMessage operation from the eHealthBox Service with encoding who results in a type of message error (Encryption not
     * allowed)
     * 
     * @throws Exception
     */
    @Test
    public void testSendEncryptedMessage() throws Exception {
        expectedEx.expect(TechnicalConnectorException.class);
        expectedEx.expectMessage(StringContains.containsString("does not support encryption"));

        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();
        documentMsg.setDestinations(getDestination());
        documentMsg.setImportant(false);
        documentMsg.setEncrypted(true);
        documentMsg.generatePublicationId();

        documentMsg.setDocument(getDocument());

        SendMessageRequest request = builder.buildMessage(documentMsg);
        service.sendMessage(request);
    }

    /**
     * Send a newsMessages who results in a type of message error (Document only allowed).
     * 
     * @throws Exception
     */
    @Test
    public void testSendNotDocumentMessage() throws Exception {
        expectedEx.expect(TechnicalConnectorException.class);
        expectedEx.expectMessage(StringContains.containsString("only supports DOCUMENT"));

        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        NewsMessage<Message> news = new NewsMessage<Message>();
        news.setDestinations(getDestination());
        news.setImportant(false);
        news.setEncrypted(false);

        news.getDocument().setTitle("Spamming with ehbox News " + new Date());
        news.getDocument().setContent("eureka".getBytes());
        news.getDocument().setFilename("test.txt");
        news.getDocument().setMimeType("text/xml");

        news.getCopyMailTo().add("dummy@ehealth.fgov.be");

        SendMessageRequest request = builder.buildMessage(news);
        service.sendMessage(request);
    }

    /**
     * Send a document with freetext who results in a type of message error (Freetext not allowed).
     * 
     * @throws Exception
     */
    @Test
    public void testSendDocumentWithFreeText() throws Exception {
        expectedEx.expect(TechnicalConnectorException.class);
        expectedEx.expectMessage(StringContains.containsString("does not support a FreeInformations element"));

        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();
        documentMsg.setDestinations(getDestination());
        documentMsg.setImportant(false);
        documentMsg.generatePublicationId();

        documentMsg.setFreeText("Free text:" + UUID.randomUUID());

        documentMsg.setDocument(getDocument());
        documentMsg.getAnnexList().add(getAnnex());

        SendMessageRequest request = builder.buildMessage(documentMsg);
        service.sendMessage(request);
    }

    /**
     * @throws Exception
     */
    @Test
    public void testSendDocumentToWrongQuality() throws Exception {
        expectedEx.expect(TechnicalConnectorException.class);
        expectedEx.expectMessage(StringContains.containsString("only support INSTITUTION or CITIZEN"));

        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();

        List<Addressee> addresseeList = new ArrayList<Addressee>();
        Addressee addressee = new Addressee(IdentifierType.SSIN);
        addressee.setId(EhboxTestDataBuilder.getInssUser1());
        addressee.setQuality(QualityType.DOCTOR_SSIN);
        addresseeList.add(addressee);
        documentMsg.setDestinations(addresseeList);

        documentMsg.setImportant(false);
        documentMsg.generatePublicationId();

        documentMsg.setDocument(getDocument());

        SendMessageRequest request = builder.buildMessage(documentMsg);
        service.sendMessage(request);
    }

    /**
     * @throws Exception
     */
    @Test
    public void testSendDocumentToTooMuchDestinations() throws Exception {
        expectedEx.expect(TechnicalConnectorException.class);
        expectedEx.expectMessage(StringContains.containsString("oes not support more than 5 Recipient"));
        
        SendMessageBuilder builder = BuilderFactory.getSendMessageBuilder();

        DocumentMessage<Message> documentMsg = new DocumentMessage<Message>();

        Addressee addressee = new Addressee(IdentifierType.SSIN);
        addressee.setId(EhboxTestDataBuilder.getInssUser1());
        addressee.setQuality(QualityType.DOCTOR_SSIN);
        
        documentMsg.setDestinations(Arrays.asList(addressee,addressee,addressee,addressee,addressee,addressee));

        documentMsg.setImportant(false);
        documentMsg.generatePublicationId();

        documentMsg.setDocument(getDocument());

        SendMessageRequest request = builder.buildMessage(documentMsg);
        service.sendMessage(request);
    }


    /**
     * Returns the destination
     * 
     * @return
     */
    private List<Addressee> getDestination() {
        Addressee addressee = new Addressee(IdentifierType.SSIN);
        addressee.setId(EhboxTestDataBuilder.getInssUser1());
        addressee.setQuality(QualityType.CITIZEN);

        return Arrays.asList(addressee);
    }

    /**
     * Returns the document
     * 
     * @return
     * @throws TechnicalConnectorException
     */
    private Document getDocument() throws TechnicalConnectorException {
        Document doc = new Document();
        doc.setTitle("Document " + new Date());
        doc.setContent(PublicationEh2EboxServiceIntegrationTest.class.getResourceAsStream(FILE_AS_DOC));
        doc.setMimeType("text/plain");
        doc.setFilename("be.ehealth.businessconnector.txt");
        return doc;
    }

    /**
     * Returns the custom meta
     * @return
     */
    private CustomMetaType getCustomMeta() {
        CustomMetaType meta = new CustomMetaType();
        meta.setKey("project");
        meta.setValue("multemediatt");
        return meta;
    }

    /**
     * Returns the document
     * 
     * @return
     * @throws TechnicalConnectorException
     */
    private Document getAnnex() throws TechnicalConnectorException {
        Document annex = new Document();
        annex.setTitle("properties");
        annex.setContent(PublicationEh2EboxServiceIntegrationTest.class.getResourceAsStream(FILE_AS_DOC));
        annex.setFilename("be.ehealth.technicalconnector.txt");
        annex.setMimeType("text/plain");
        return annex;
    }

    private void verifyAnswer(SendMessageResponse response) {
        // check if there are no errors
        Assert.assertEquals("100", response.getStatus().getCode());
        // check if a messageId has been returned
        Assert.assertNotNull(response.getId());

        LOG.debug("Message Sent: " + response.getId());
        for (LocalisedString localisedString : response.getStatus().getMessages()) {
            LOG.debug("statusMessage" + localisedString.getValue());
        }
        Assert.assertTrue(response.getStatus().getMessages().size() == 1);
        Assert.assertEquals("SUCCESS",response.getStatus().getMessages().get(0).getValue());
    }
}
