/*
 * Copyright (c) eHealth
 */
package be.fgov.ehealth.technicalconnector.bootstrap.bcp;

import javax.xml.ws.soap.SOAPFaultException;

import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import be.ehealth.technicalconnector.config.ConfigFactory;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.ws.ServiceFactory;
import be.ehealth.technicalconnector.ws.domain.GenericRequest;
import be.ehealth.technicalconnector.ws.impl.AbstractWsSender;
import be.ehealth.technicalconnector.ws.impl.strategy.RetryStrategy;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.EndpointUpdateRule;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.HttpServerStubRule;
import be.fgov.ehealth.technicalconnector.tests.server.EchoResponse;

/**
 * @author EH053
 */
public class GenericRequestTest {

    private static final String DUMMY_PAYLOAD = "<test>hello</test>";

    private static final String URL_BCP_ALTERNATIVE = "http://localhost:6666/error/403";

    private static final String URL_BCP_ERROR = "http://localhost:7777/error/404";

    private static final String URL_BCP_NO_ALTERNATIVE = "http://localhost:8888/error/403";

    private static final String URL_BCP_LOOP = "http://localhost:9999/error/403";

    @Rule
    public ExpectedException expect = ExpectedException.none();

    @Rule
    public HttpServerStubRule server6666 = new HttpServerStubRule(6666);

    @Rule
    public HttpServerStubRule server7777 = new HttpServerStubRule(7777);

    @Rule
    public HttpServerStubRule server8888 = new HttpServerStubRule(8888);

    @Rule
    public HttpServerStubRule server9999 = new HttpServerStubRule(9999);

    @Rule
    public EndpointUpdateRule rule = new EndpointUpdateRule("/tsl-samples/local/servicelist-local");

    @BeforeClass
    public static void before() {
        ConfigFactory.getConfigValidator().setProperty(AbstractWsSender.PROP_RETRY_STRATEGY, RetryStrategy.class.getName());
    }

    @Test
    public void invokeWithRetrySuccess() throws Exception {
        GenericRequest req = new GenericRequest();
        req.setPayload(DUMMY_PAYLOAD);
        req.setEndpoint(URL_BCP_ALTERNATIVE);
        Assert.assertTrue(ServiceFactory.getGenericWsSender().send(req).asObject(EchoResponse.class).getInbound().getFiles().get("postData").contains(DUMMY_PAYLOAD));
        Assert.assertTrue(EndpointDistributor.getInstance().mustPoll());
    }

    @Test
    public void invokeWithRetrySOAPFault() throws Exception {
        expect.expect(SOAPFaultException.class);
        expect.expectMessage("unsupported");
        GenericRequest req = new GenericRequest();
        req.setPayload(DUMMY_PAYLOAD);
        req.setEndpoint(URL_BCP_ERROR);
        ServiceFactory.getGenericWsSender().send(req).getSOAPException();
        Assert.assertTrue(EndpointDistributor.getInstance().mustPoll());
    }

    @Test
    public void invokeLoop() throws Exception {
        expect.expect(TechnicalConnectorException.class);
        expect.expectMessage("403");
        GenericRequest req = new GenericRequest();
        req.setPayload(DUMMY_PAYLOAD);
        req.setEndpoint(URL_BCP_LOOP);
        ServiceFactory.getGenericWsSender().send(req).getSOAPException();
        Assert.assertFalse(EndpointDistributor.getInstance().mustPoll());
    }

    @Test
    public void invokeNoAlternative() throws Exception {
        expect.expect(TechnicalConnectorException.class);
        expect.expectMessage("403");
        GenericRequest req = new GenericRequest();
        req.setPayload(DUMMY_PAYLOAD);
        req.setEndpoint(URL_BCP_NO_ALTERNATIVE);
        ServiceFactory.getGenericWsSender().send(req);
        Assert.assertFalse(EndpointDistributor.getInstance().mustPoll());
    }

}
