/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.ehbox.v3;

import org.joda.time.DateTime;

import be.ehealth.businessconnector.ehbox.api.domain.Addressee;
import be.ehealth.technicalconnector.utils.DateUtils;


/**
 * container for the parameters to build a message.
 * 
 * @author EH076
 * 
 * @since
 * 
 */
public final class MessageBuilder {

    private Addressee destination;

    private String mimeType;

    private String title;

    private String content;

    private boolean encrypted;

    private boolean important;

    private boolean receiveAck;

    private boolean publicationAck;

    private boolean readAck;


    /**
     * creates a MessageBuilder with the given addressee and content. the following default values are used
     * <ul>
     * <li>mimetType = text/plain</li>
     * <li>title : title + currentTime</li>
     * <li>encrypted = false</li>
     * <li>important = false</li>
     * <li>receiveAck = false</li>
     * <li>publicationAck = false</li>
     * <li>readAck = false</li>
     * </ul>
     * 
     * @param addressee
     * @param content
     * @return
     */
    public static MessageBuilder create(Addressee addressee, String content) {
        MessageBuilder builder = new MessageBuilder(addressee, content);
        builder.mimeType = "text/plain";
        builder.title = "title " + DateUtils.printDate(new DateTime());
        builder.encrypted = false;
        builder.important = false;
        builder.receiveAck = false;
        builder.publicationAck = false;
        builder.readAck = false;
        return builder;
    }

    public MessageBuilder mimeType(String mimeType) {
        this.mimeType = mimeType;
        return this;
    }

    public MessageBuilder title(String title) {
        this.title = title;
        return this;
    }

    public MessageBuilder encrypted(boolean encrypted) {
        this.encrypted = encrypted;
        return this;
    }

    public MessageBuilder important(boolean important) {
        this.important = important;
        return this;
    }

    public MessageBuilder receiveAck(boolean receiveAck) {
        this.receiveAck = receiveAck;
        return this;
    }

    public MessageBuilder publicationAck(boolean publicationAck) {
        this.publicationAck = publicationAck;
        return this;
    }

    public MessageBuilder readAck(boolean readAck) {
        this.readAck = readAck;
        return this;
    }


    /**
     * private constructor : static builder create method should be use to create an instance.
     * 
     * @param destination
     * @param content
     */
    private MessageBuilder(Addressee destination, String content) {
        super();
        this.destination = destination;
        this.content = content;
    }


    /**
     * @return the destination
     */
    public Addressee getDestination() {
        return destination;
    }


    /**
     * @return the mimeType
     */
    public String getMimeType() {
        return mimeType;
    }


    /**
     * @return the title
     */
    public String getTitle() {
        return title;
    }


    /**
     * @return the content
     */
    public String getContent() {
        return content;
    }


    /**
     * @return the encrypted
     */
    public boolean isEncrypted() {
        return encrypted;
    }


    /**
     * @return the important
     */
    public boolean isImportant() {
        return important;
    }


    /**
     * @return the receiveAck
     */
    public boolean isReceiveAck() {
        return receiveAck;
    }


    /**
     * @return the publicationAck
     */
    public boolean isPublicationAck() {
        return publicationAck;
    }


    /**
     * @return the readAck
     */
    public boolean isReadAck() {
        return readAck;
    }


}
