/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.mediprimav2.session;


import static be.ehealth.businessconnector.test.mediprimav2.builder.ConsultCarmedInterventionRequestBuilder.aRequest;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import be.ehealth.businessconnector.mediprimav2.session.MediprimaSessionServiceFactory;
import org.joda.time.DateTime;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import be.ehealth.technicalconnector.utils.DateUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionRequest;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;


/**
 * Integration tests for consultation mediprima v2 for orgpharmacy. The scenarios numbering match those of the MediPrima specification.
 *
 *  /!\ The configuration must have this Mediprima Consult specific saml attribute property:
 * sessionmanager.samlattributedesignator.8=urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:ehealth:1.0:pharmacy:nihii-number:recognisedpharmacy:nihii11
 * sessionmanager.samlattributedesignator.9=urn:be:fgov:person:ssin:ehealth:1.0:pharmacist:boolean,urn:be:fgov:ehealth:1.0:certificateholder:person:ssin:usersession:boolean

 *
 * @author EHP
 */
@RunWith(Parameterized.class)
public final class MediprimaConsultationOrgpharmacyWithoutSoaErrorIntegrationTest {


    private static final String NISS_16 = "68031061549";

    private static final String NISS_08 = "74032853407";

    private static final String NISS_15 = "90101351882";

    private static final String NISS_07 = "94000087334";

    private static final String NISS_6 = "93122961468";

    private static final String NISS_09 = "88081964510";

    private static final String NISS_03 = "68050555876";

    private static final String NISS_05 = "95090238957";

    private static final String NISS_02 = "60061861748";
    
    private static final String NISS_01 = "70410100375";

    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("orgpharmacy").forUser("maxime").build();


    private String scenarioId;
    private String scenarioName;
    private ConsultCarmedInterventionRequest request;


    public MediprimaConsultationOrgpharmacyWithoutSoaErrorIntegrationTest(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
        super();
        this.scenarioId = scenarioId;
        this.scenarioName = scenarioName;
        this.request = request;
    }


    @Parameterized.Parameters(name = "{0}: {1}")
    public static Collection<Object[]> data() throws Exception {
        Collection<Object[]> testData = new ArrayList<Object[]>();
        testData.add(new Object[]{"scenario05", "Unknown ssin", aRequest().withSsin(NISS_01).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario06", "Known ssin but dates nok", aRequest().withSsin(NISS_02).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario07", "No corresponding coverage", aRequest().withSsin(NISS_03).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario08", "Decision without required rights - coverage exists but nok", aRequest().withSsin(NISS_05).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario09", "Coverage ok, but unauthorized provider", aRequest().withSsin(NISS_09).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario10", "Card with rights ok but suspended", aRequest().withSsin(NISS_6).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario11", "Full SPP support, extra list", aRequest().withSsin(NISS_07).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario12", "Full SPP support, supplements via convention", aRequest().withSsin(NISS_15).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario13", "Partial SPP support, supplements = all", aRequest().withSsin(NISS_08).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario14", "Partial SPP support, supplements = nothing", aRequest().withSsin(NISS_16).withReferenceDate(new DateTime()).build()});

        return testData;
    }

    @Test
    public void testScenario() throws Exception {
        ConsultCarmedInterventionResponse response = MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);

        String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
        MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse> helper = new MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse>(ConsultCarmedInterventionResponse.class, ConsultCarmedInterventionResponse.class);
        XmlAsserter.assertSimilar(expected, helper.toString(response));
    }



    private static String expectedResponse(DateTime consultationDate, String requestId, String scenarioId) {
        Map<String, Object> templateContext = new HashMap<String, Object>();
        templateContext.put("requestId", requestId);
        templateContext.put("consultationDate", DateUtils.printDate(consultationDate));
        templateContext.put("startDate", DateUtils.printDate(new DateTime()));
        templateContext.put("endDate", DateUtils.printDate(new DateTime()));
        return TemplateEngineUtils.generate(templateContext, "/examples/mediprimav2/consultation/expected/orgpharmacy/" + scenarioId + "Response.xml");
    }
}
