/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.mediprimav2.session;


import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import be.ehealth.businessconnector.mediprimav2.session.MediprimaSessionServiceFactory;
import be.ehealth.businessconnector.test.mediprimav2.builder.ConsultCarmedInterventionRequestBuilder;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import be.ehealth.technicalconnector.utils.DateUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionRequest;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.RestoreSystemProperties;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * Integration tests for consultation mediprima v2 persphysician. The scenarios numbering match those of the MediPrima specification.
 *
 * @author EHP
 */
@RunWith(Parameterized.class)
public final class MediprimaConsultationPersphysicianWithoutSoaErrorIntegrationTest {

    private static final String NISS_08 = "74032853407";
    private static final String NISS_07 = "94000087334";
    private static final String NISS_6 = "93122961468";
    private static final String NISS_09 = "88081964510";
    private static final String NISS_5 = "95090238957";
    private static final String NISS_3 = "68050555876";
    private static final String NISS_2 = "60061861748";
    private static final String NISS_01 = "70410100375";

    @ClassRule
    public static RuleChain chain = RuleChain.outerRule(new RestoreSystemProperties()).around(SessionRule.withActiveSession().forEnvironment("acc").forProfession("persphysician").forUser("hannes").build());


    private String scenarioId;
    private String scenarioName;
    private ConsultCarmedInterventionRequest request;


    public MediprimaConsultationPersphysicianWithoutSoaErrorIntegrationTest(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
        super();
        this.scenarioId = scenarioId;
        this.scenarioName = scenarioName;
        this.request = request;
    }


    @Parameterized.Parameters(name = "{0}: {1}")
    public static Collection<Object[]> data() throws Exception {
        Collection<Object[]> testData = new ArrayList<Object[]>();
        testData.add(new Object[]{"scenario05", "No decision (SSIN unknown)", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_01).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario06", "No more cart (SSIN known, dates NOK)", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_2).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario07", "Decision without rights, reason no cover", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_3).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario08", "Decision without rights, reason cover present but nok", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_5).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario09", "Decision without right, reason cover ok but recipient nok", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_09).withReferenceDate(new DateTime()).build()});
        //Not compliant with expected document
        testData.add(new Object[]{"scenario10", "Decision ok, suspended, (total support SPP)", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_6).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario11", "Decision ok, active, SPP full support", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_07).withReferenceDate(new DateTime()).build()});
        testData.add(new Object[]{"scenario12", "Decision ok, active, SPP partial support", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin(NISS_08).withReferenceDate(new DateTime()).build()});

        return testData;
    }

    @Test
    public void testScenario() throws Exception {
        ConsultCarmedInterventionResponse response = MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);

        String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
        MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse> helper = new MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse>(ConsultCarmedInterventionResponse.class, ConsultCarmedInterventionResponse.class);
        XmlAsserter.assertSimilar(expected, helper.toString(response));
    }

    private static String expectedResponse(DateTime consultationDate, String requestId, String scenarioId) {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("requestId", requestId);
        velocityContext.put("consultationDate", DateUtils.printDate(consultationDate));
        String commentDate = DateTimeFormat.forPattern("dd/MM/yyyy").print(new DateTime());
        velocityContext.put("commentStartDate", commentDate);
        velocityContext.put("commentEndDate", commentDate);
        velocityContext.put("startDate", DateUtils.printDate(new DateTime()));
        velocityContext.put("endDate", DateUtils.printDate(new DateTime()));
        return TemplateEngineUtils.generate(velocityContext, "/examples/mediprimav2/consultation/expected/persphysician/" + scenarioId + "Response.xml");
    }
}
