/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.mediprimauma.session;

import be.ehealth.businessconnector.mediprimauma.exception.MediprimaUmaDeleteException;
import be.ehealth.businessconnector.mediprimauma.exception.MediprimaUmaSearchException;
import be.ehealth.businessconnector.mediprimauma.exception.MediprimaUmaSendException;
import be.ehealth.businessconnector.mediprimauma.session.MediprimaUmaService;
import be.ehealth.businessconnector.mediprimauma.session.MediprimaUmaSessionServiceFactory;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.commons.core.v2.ActorType;
import be.fgov.ehealth.commons.core.v2.Author;
import be.fgov.ehealth.commons.core.v2.Id;
import be.fgov.ehealth.mediprima.uma.core.v1.CriteriaType;
import be.fgov.ehealth.mediprima.uma.core.v1.PeriodType;
import be.fgov.ehealth.mediprima.uma.protocol.v1.DeleteUrgentMedicalAidAttestationRequest;
import be.fgov.ehealth.mediprima.uma.protocol.v1.SearchUrgentMedicalAidAttestationRequest;
import be.fgov.ehealth.mediprima.uma.protocol.v1.SendUrgentMedicalAidAttestationRequest;
import be.fgov.ehealth.mediprima.uma.protocol.v1.SendUrgentMedicalAidAttestationResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.joda.time.DateTime;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.FixMethodOrder;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runners.MethodSorters;

import javax.xml.ws.soap.SOAPFaultException;
import java.util.UUID;

/**
 * Integration tests for MediPrima UMA send, search and delete requests.
 *
 * @author EHP
 */
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public final class MediprimaUmaIntegrationTest {
    
    private static final DateTime ISSUE_INSTANT = new DateTime();

    private static final DateTime END_DATE = new DateTime().plusDays(1);

    @ClassRule
    public static SessionRule sessionRule = SessionRule.withActiveSession().baseOn("/be.ehealth.businessconnector.mediprimauma.test.properties").build();

    @Rule
    public ExpectedException expectedEx = ExpectedException.none();

    private static String attestationNumber;
    
    /* Public methods */
    
    @Test
    public void _001_sendUrgentMedicalAidAttestation() throws Exception {
        sendUrgentMedicalAidAttestation("19648339550", "80090116494", "hospitalization", "/examples/mediprimauma/expected/response/send.xml");
    }
    
    @Test
    public void _002_searchUrgentMedicalAidAttestation() throws Exception {
        searchUrgentMedicalAidAttestation("80090116494", attestationNumber, "/examples/mediprimauma/expected/response/search.xml");
    }
    
    @Test
    public void _003_deleteUrgentMedicalAidAttestation() throws Exception {
        deleteUrgentMedicalAidAttestation("19648339550", "80090116494", attestationNumber, "/examples/mediprimauma/expected/response/delete.xml");
    }
    
    @Test
    public void _004_invalidNihii_sendUrgentMedicalAidAttestation() throws Exception {
        sendUrgentMedicalAidAttestation("84022611609", "80090116494", "hospitalization", "/examples/mediprimauma/expected/error/send_invalid_nihii.xml");
    }
    
    @Test
    public void _005_invalidSsin_sendUrgentMedicalAidAttestation() throws Exception {
        sendUrgentMedicalAidAttestation("19648339550", "12345678901", "hospitalization", "/examples/mediprimauma/expected/error/send_invalid_niss.xml");
    }
    
    @Test
    public void _006_invalidMedicalCover_sendUrgentMedicalAidAttestation() throws Exception {
        sendUrgentMedicalAidAttestation("19648339550", "80090116494", "something_wrong", "/examples/mediprimauma/expected/error/send_invalid_medical_cover.xml");
    }
    
    @Test
    public void _007_noNihii_sendUrgentMedicalAidAttestation() throws Exception {
        sendUrgentMedicalAidAttestation(null, "80090116494", "hospitalization", "/examples/mediprimauma/expected/error/send_no_nihii.xml");
    }
    
    @Test
    public void _008_noSsin_sendUrgentMedicalAidAttestation() throws Exception {
        sendUrgentMedicalAidAttestation("19648339550", null, "hospitalization", "/examples/mediprimauma/expected/error/send_no_ssin.xml");
    }
    
    @Test
    public void _009_noMedicalCover_sendUrgentMedicalAidAttestation() throws Exception {
        sendUrgentMedicalAidAttestation("19648339550", "80090116494", null, "/examples/mediprimauma/expected/error/send_no_medical_cover.xml");
    }
    
    @Test
    public void _010_invalidNihii_searchUrgentMedicalAidAttestation() throws Exception {
        searchUrgentMedicalAidAttestation("84022611609", "690-HOSP-180000004548", "/examples/mediprimauma/expected/error/search_invalid_nihii.xml");
    }
    
    @Test
    public void _011_invalidAttestationNumber_searchUrgentMedicalAidAttestation() throws Exception {
        searchUrgentMedicalAidAttestation("80090116494", "SOME_WRONG_ATTEST_NBR", "/examples/mediprimauma/expected/error/search_invalid_attestation_number.xml");
    }
    
    @Test
    public void _012_noNihii_searchUrgentMedicalAidAttestation() throws Exception {
        searchUrgentMedicalAidAttestation(null, "690-HOSP-180000004548", "/examples/mediprimauma/expected/error/search_no_nihii.xml");
    }
    
    @Test
    public void _013_noAttestationNumber_searchUrgentMedicalAidAttestation() throws Exception {
        searchUrgentMedicalAidAttestation("80090116494", null, "/examples/mediprimauma/expected/error/search_no_attestation_number.xml");
    }
    
    @Test
    public void _014_invalidNihii_deleteUrgentMedicalAidAttestation() throws Exception {
        deleteUrgentMedicalAidAttestation("84022611609", "80090116494", "690-HOSP-180000004548", "/examples/mediprimauma/expected/error/delete_invalid_nihii.xml");
    }
    
    @Test
    public void _015_invalidSsin_deleteUrgentMedicalAidAttestation() throws Exception {
        deleteUrgentMedicalAidAttestation("19648339550", "12345678901", "690-HOSP-180000004548", "/examples/mediprimauma/expected/error/delete_invalid_ssin.xml");
    }
    
    @Test
    public void _016_invalidAttestationNumber_deleteUrgentMedicalAidAttestation() throws Exception {
        deleteUrgentMedicalAidAttestation("19648339550", "80090116494", "SOME_WRONG_ATTEST_NBR", "/examples/mediprimauma/expected/error/delete_invalid_attestation_number.xml");
    }
    
    @Test
    public void _017_noNihii_deleteUrgentMedicalAidAttestation() throws Exception {
        deleteUrgentMedicalAidAttestation(null, "80090116494", "690-HOSP-180000004548", "/examples/mediprimauma/expected/error/delete_no_nihii.xml");
    }
    
    @Test
    public void _018_noSsin_deleteUrgentMedicalAidAttestation() throws Exception {
        deleteUrgentMedicalAidAttestation("19648339550", null, "690-HOSP-180000004548", "/examples/mediprimauma/expected/error/delete_no_ssin.xml");
    }
    
    @Test
    public void _019_noAttestationNumber_deleteUrgentMedicalAidAttestation() throws Exception {
        deleteUrgentMedicalAidAttestation("19648339550", "80090116494", null, "/examples/mediprimauma/expected/error/delete_no_attestation_number.xml");
    }
    
    /* Private methods */
    
    private void deleteUrgentMedicalAidAttestation(String authorNihii, String beneficiarySsin, String attestationNumber, String expectedResponseTemplate) throws Exception {
        sendRequest(createDeleteRequest(authorNihii, beneficiarySsin, attestationNumber), expectedResponseTemplate, RequestType.DELETE);
    }
    
    private void searchUrgentMedicalAidAttestation(String beneficiarySsin, String attestationNumber, String expectedResponseTemplate) throws Exception {
        sendRequest(createSearchRequest(beneficiarySsin, attestationNumber), expectedResponseTemplate, RequestType.SEARCH);
    }
    
    private void sendUrgentMedicalAidAttestation(String authorNihii, String beneficiarySsin, String medicalCover, String expectedResponseTemplate) throws Exception {
        sendRequest(createSendRequest(authorNihii, beneficiarySsin, medicalCover), expectedResponseTemplate, RequestType.SEND);
    }
    
    private DeleteUrgentMedicalAidAttestationRequest createDeleteRequest(String authorNihii, String beneficiarySsin, String attestationNumber) {
        DeleteUrgentMedicalAidAttestationRequest request = new DeleteUrgentMedicalAidAttestationRequest();
        request.setAuthor(createAuthor(authorNihii));
        request.setAttestationNumber(attestationNumber);
        request.setBeneficiarySsin(beneficiarySsin);
        request.setIssueInstant(ISSUE_INSTANT);
        request.setId(generateId());
        return request;
    }
    
    private SearchUrgentMedicalAidAttestationRequest createSearchRequest(String beneficiarySsin, String attestationNumber) {
        SearchUrgentMedicalAidAttestationRequest request = new SearchUrgentMedicalAidAttestationRequest();
        CriteriaType criteria = new CriteriaType();
        criteria.setAttestationNumber(attestationNumber);
        criteria.setBeneficiarySsin(beneficiarySsin);
        request.setCriteria(criteria);
        request.setIssueInstant(ISSUE_INSTANT);
        request.setId(generateId());
        return request;
    }
    
    private SendUrgentMedicalAidAttestationRequest createSendRequest(String authorNihii, String beneficiarySsin, String medicalCover) {
        SendUrgentMedicalAidAttestationRequest request = new SendUrgentMedicalAidAttestationRequest();
        request.setAuthor(createAuthor(authorNihii));
        request.setBeneficiarySsin(beneficiarySsin);
        request.setIssueInstant(ISSUE_INSTANT);
        request.setMedicalCover(medicalCover);
        request.setValidityPeriod(createPeriod());
        request.setId(generateId());
        return request;
    }
    
    private PeriodType createPeriod() {
        PeriodType period = new PeriodType();
        period.setStartDate(ISSUE_INSTANT);
        period.setEndDate(END_DATE);
        return period;
    }
    
    private Author createAuthor(String nihii) {
        Author author = new Author();
        ActorType actor = new ActorType();
        actor.getFirstNames().add("Jean");
        actor.setName("Neymar");
        Id id = new Id();
        id.setType("urn:be:fgov:ehealth:1.0:physician:nihii-number");
        id.setValue(nihii);
        actor.getIds().add(id);
        author.getHcParties().add(actor);
        return author;
    }
    
    private String generateId() {
        return "ID_" + UUID.randomUUID().toString();
    }
    
    private void sendRequest(Object request, String expectedResponseTemplate, RequestType type) throws Exception {
        Object response;
        
        try {
            MediprimaUmaService service = MediprimaUmaSessionServiceFactory.getMediprimaUmaSession();
            
            switch(type) {
                case SEND:
                    SendUrgentMedicalAidAttestationResponse serviceResponse = service.sendUrgentMedicalAidAttestation((SendUrgentMedicalAidAttestationRequest) request);
                    attestationNumber = serviceResponse.getAttestion().getAttestationNumber();
                    response = serviceResponse;
                break;
                
                case SEARCH:
                    response = service.searchUrgentMedicalAidAttestation((SearchUrgentMedicalAidAttestationRequest) request);
                break;
                
                case DELETE:
                    response = service.deleteUrgentMedicalAidAttestation((DeleteUrgentMedicalAidAttestationRequest) request);
                break;
                
                default: 
                    throw new IllegalArgumentException("Unknown request type");
            }
            
            Assert.assertNotNull(response);
        } catch (MediprimaUmaSendException e1) {
            response = e1.getResponse();
            Assert.assertNotNull(e1.getRegistryStatus());
        } catch (MediprimaUmaSearchException e2) {
            response = e2.getResponse();
            Assert.assertNotNull(e2.getRegistryStatus());
        } catch (MediprimaUmaDeleteException e3) {
            response = e3.getResponse();
            Assert.assertNotNull(e3.getRegistryStatus());
        } catch (SOAPFaultException e4) {
            response = ConnectorXmlUtils.toString(e4.getMessage());
        } catch (TechnicalConnectorException e5) {
            response = ConnectorXmlUtils.toString(e5.getCause().getMessage());
        }
        
        String expectedResponse = ConnectorIOUtils.getResourceAsString(expectedResponseTemplate);
        String actualResponse = response instanceof String? (String)response : ConnectorXmlUtils.toString(response);
        XmlAsserter.assertSimilar(expectedResponse, actualResponse);
    }
    
    
    /* Private class */
    
    private enum RequestType {
        SEND, SEARCH, DELETE;
    }

}
