package be.ehealth.businessconnector.test.mycarenet.attestv3.session;

import be.ehealth.business.mycarenetcommons.domain.EncryptedRequestHolder;
import be.ehealth.business.mycarenetcommons.domain.SignedEncryptedResponseHolder;
import be.ehealth.business.mycarenetcommons.domain.SignedResponseHolder;
import be.ehealth.business.mycarenetdomaincommons.domain.Attribute;
import be.ehealth.business.mycarenetdomaincommons.domain.InputReference;
import be.ehealth.business.mycarenetdomaincommons.domain.Ssin;
import be.ehealth.businessconnector.mycarenet.attestv3.builders.CancelAttestationRequestInput;
import be.ehealth.businessconnector.mycarenet.attestv3.builders.RequestObjectBuilderFactory;
import be.ehealth.businessconnector.mycarenet.attestv3.builders.ResponseObjectBuilderFactory;
import be.ehealth.businessconnector.mycarenet.attestv3.builders.SendAttestationRequestInput;
import be.ehealth.businessconnector.mycarenet.attestv3.session.AttestSessionServiceFactory;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.fgov.ehealth.errors.core.v1.LocalisedStringType;
import be.fgov.ehealth.errors.soa.v1.BusinessError;
import be.fgov.ehealth.mycarenet.attest.protocol.v3.CancelAttestationRequest;
import be.fgov.ehealth.mycarenet.attest.protocol.v3.CancelAttestationResponse;
import be.fgov.ehealth.mycarenet.attest.protocol.v3.SendAttestationRequest;
import be.fgov.ehealth.mycarenet.attest.protocol.v3.SendAttestationResponse;
import be.fgov.ehealth.technicalconnector.signature.domain.SignatureVerificationResult;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.junit.Assert;
import org.junit.Test;

import javax.xml.soap.DetailEntry;
import javax.xml.ws.soap.SOAPFaultException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.junit.Assert.fail;

public abstract class AbstractAttestIntegrationTest  {

    private String emehID;

    private String enxsID;

    private String cmchID;

    private String cnxsID;

    public AbstractAttestIntegrationTest(String emehID, String enxsID, String cmchID, String cnxsID) {
        this.emehID = emehID;
        this.enxsID = enxsID;
        this.cmchID = cmchID;
        this.cnxsID = cnxsID;
    }

    @Test
    public void happySendAttestationWithEncryption() throws Exception {
        try {
            happySendAttestationTest(emehID);
        } catch (SOAPFaultException e) {
            if (e.getFault().getFaultString().equals("UNSUPPORTED_CAREPROVIDER: Not a supported care provider.")) {
                fail(e.getFault().getFaultString());
            } else {
                throw e;
            }
        }
    }

    @Test
    public void happyCancelAttestationWithoutEncryption() throws Exception {
        try {
            happyCancelAttestationTest(cmchID);
        } catch (SOAPFaultException e) {
            if (e.getFault().getFaultString().equals("UNSUPPORTED_CAREPROVIDER: Not a supported care provider")) {
                fail(e.getFault().getFaultString());
            } else {
                throw e;
            }
        }
    }

    @Test
    public void unhappySendAttestationWithEncryption() throws Exception {
        unhappySendAttestationTest(enxsID);
    }

    @Test
    public void unhappyCancelAttestationWithoutEncryption() throws Exception {
        unhappyCancelAttestationTest(cnxsID);
    }
    
    private void happySendAttestationTest(String inputReference) throws Exception {
        byte[] kmehrmessage = ConnectorIOUtils.getResourceAsString("/examples/mycarenet/attestv3/requests/mha-request-detail.xml").getBytes(StandardCharsets.UTF_8);
        
        SignedEncryptedResponseHolder attestResponse = sendAttestation(inputReference, kmehrmessage);

        SignatureVerificationResult signatureVerificationResult = attestResponse.getSignatureVerificationResult();
        Assert.assertTrue("Errors found in the signature verification", signatureVerificationResult.isValid());
        String expectedResponse = ConnectorIOUtils.getResourceAsString("/examples/mycarenet/attestv3/responses/mha-response-detail-" + inputReference + ".xml");
        XmlAsserter.assertSimilar(expectedResponse, new String(attestResponse.getBusinessResponse(), UTF_8.name()));
    }

    private void happyCancelAttestationTest(String inputReference) throws Exception {
        byte[] kmehrmessage = ConnectorIOUtils.getResourceAsString("/examples/mycarenet/attestv3/requests/mhca-request-detail.xml").getBytes(StandardCharsets.UTF_8);
        
        SignedResponseHolder attestResponse = cancelAttestation(inputReference, kmehrmessage);

        SignatureVerificationResult signatureVerificationResult = attestResponse.getSignatureVerificationResult();
        Assert.assertTrue("Errors found in the signature verification", signatureVerificationResult.isValid());
        String expectedResponse = ConnectorIOUtils.getResourceAsString("/examples/mycarenet/attestv3/responses/mhca-response-detail-" + inputReference + ".xml");
        XmlAsserter.assertSimilar(expectedResponse, new String(attestResponse.getBusinessResponse(), UTF_8.name()));
    }

    private void unhappySendAttestationTest(String inputReference) throws Exception {
        byte[] kmehrmessage = ConnectorIOUtils.getResourceAsString("/examples/mycarenet/attestv3/requests/ns-request-detail.xml").getBytes(StandardCharsets.UTF_8);
        try {
            sendAttestation(inputReference, kmehrmessage);
            Assert.fail("soap fault expected");
        } catch (SOAPFaultException sfe) {
            checkSOAPFault(sfe, "EMPTY_BLOB", "Blob is empty.");
        }
    }

    private void unhappyCancelAttestationTest(String inputReference) throws Exception {
        byte[] kmehrmessage = ConnectorIOUtils.getResourceAsString("/examples/mycarenet/attestv3/requests/ns-cancel-request-detail.xml").getBytes(StandardCharsets.UTF_8);
        try {
            cancelAttestation(inputReference, kmehrmessage);
            Assert.fail("soap fault expected");
        } catch (SOAPFaultException sfe) {
            checkSOAPFault(sfe, "INVALID_DETAIL_REQUEST", "Detail XML structure is not valid against schema.");
        }
    }

    private SignedEncryptedResponseHolder sendAttestation(String inputReference, byte[] kmehrmessage) throws ConnectorException {
        EncryptedRequestHolder<SendAttestationRequest> attestBuilderRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().buildSendAttestationRequest(
                SendAttestationRequestInput.builder()
                        .isTest(true)
                        .inputReference(new InputReference(inputReference))
                        .kmehrmessage(kmehrmessage)
                        .patientSsin(new Ssin("72070539942"))
                        .referenceDate(LocalDateTime.now())
                        .messageVersion("3.0")
                        .issuer("some issuer")
                        .commonInputAttributes(Arrays.asList(Attribute.builder()
                                        .key("urn:be:cin:nippin:purpose")
                                        .value("some purpose")
                                        .build(),
                                Attribute.builder()
                                        .key("urn:be:cin:nippin:attemptNbr")
                                        .value(1)
                                        .build()))
                        .build());
        
        SendAttestationResponse sendAttestationResponse = AttestSessionServiceFactory.getAttestService().sendAttestation(attestBuilderRequest.getSendAttestationRequest());

        return ResponseObjectBuilderFactory.getResponseObjectBuilder().handleSendAttestionResponse(sendAttestationResponse, attestBuilderRequest);
    }

    private SignedResponseHolder cancelAttestation(String inputReference, byte[] kmehrmessage) throws ConnectorException {
        CancelAttestationRequest cancelAttestationRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().buildCancelAttestationRequest(
                CancelAttestationRequestInput.builder()
                        .isTest(true)
                        .inputReference(new InputReference(inputReference))
                        .kmehrmessage(kmehrmessage)
                        .messageVersion("3.0")
                        .issuer("some issuer")
                        .commonInputAttributes(Arrays.asList(Attribute.builder()
                                        .key("urn:be:cin:nippin:purpose")
                                        .value("some purpose")
                                        .build(),
                                Attribute.builder()
                                        .key("urn:be:cin:nippin:attemptNbr")
                                        .value(1)
                                        .build()))
                        .build());
        
        CancelAttestationResponse cancelAttestationResponse = AttestSessionServiceFactory.getAttestService().cancelAttestation(cancelAttestationRequest);

        return ResponseObjectBuilderFactory.getResponseObjectBuilder().handleCancelAttestationResponse(cancelAttestationResponse, cancelAttestationRequest);
    }

    private void checkSOAPFault(SOAPFaultException sfe, String expectedErrorCode, String expectedMessage) {
        List<DetailEntry> entries = new ArrayList<DetailEntry>();
        Iterator it = sfe.getFault().getDetail().getDetailEntries();
        while (it.hasNext()) entries.add((DetailEntry) it.next());

        Assert.assertEquals(1, entries.size());

        MarshallerHelper<BusinessError, BusinessError> helper = new MarshallerHelper<BusinessError, BusinessError>(BusinessError.class, BusinessError.class);
        BusinessError error = helper.toObject(entries.get(0));

        Assert.assertEquals(expectedErrorCode, error.getCode());
        Assert.assertEquals("MYCARENET", error.getOrigin());

        List<LocalisedStringType> messages = error.getMessages();
        Assert.assertEquals(1, messages.size());

        LocalisedStringType message = messages.get(0);
        Assert.assertEquals(expectedMessage, message.getValue());
    }
}
