/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.mycarenet.attestv2.session;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import be.ehealth.businessconnector.mycarenet.attestv2.session.AttestSessionServiceFactory;
import be.ehealth.businessconnector.test.mycarenet.attestv2.helper.AttestIntegrationTestHelper;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import be.cin.encrypted.EncryptedKnownContent;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SendAttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedEncryptedBuilderResponse;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.messageservices.mycarenet.core.v1.SendTransactionResponse;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.SendAttestationResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * eAttest v2 Integration Test for send with profile dentist. The scenarios numbering matches one specified by mycarenet.
 *
 * @author EHP
 */
@RunWith(Parameterized.class)
public class SendAttestDentistIntegrationTest {

	private static final String NISS5_DENT_OA500 = "46111636603";

	private static final String NISS3_DENT_OA500 = "44042700342";

	private static final String NISS1_DENT_OA500 = "62110906574";

	private static final String NISS5_DENT_OA100 = "85020205931";

	private static final String NISS3_DENT_OA100 = "81010924034";

	private static final String NISS1_DENT_OA100 = "73052005540";

	@ClassRule
	public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persdentist").forUser("maxime").build();

	private AttestIntegrationTestHelper helper = new AttestIntegrationTestHelper();

	private String scenarioName;

	private String ssin;

	private String oa;

	private String profession;


	public SendAttestDentistIntegrationTest(String scenarioName, String ssin, String oa, String profession) {
		super();
		this.scenarioName = scenarioName;
		this.ssin = ssin;
		this.oa = oa;
		this.profession = profession;
	}


	@Parameterized.Parameters(name = "{0},{1},{2}")
	public static Collection<Object[]> data() {
		Collection<Object[]> testData = new ArrayList<Object[]>();

		testData.add(new Object[]{"DENT1", NISS1_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENT2", NISS1_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENT3a", NISS1_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENT3b", NISS1_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENT4", NISS1_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENT5", NISS3_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENT6", NISS5_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENT7", NISS5_DENT_OA100, "100", "DENT"});
		testData.add(new Object[]{"DENTFull", NISS1_DENT_OA100, "100", "DENT"});

		// OA 500
		testData.add(new Object[]{"DENT1", NISS1_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENT2", NISS1_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENT3a", NISS1_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENT3b", NISS1_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENT4", NISS1_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENT5", NISS3_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENT6", NISS5_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENT7", NISS5_DENT_OA500, "500", "DENT"});
		testData.add(new Object[]{"DENTFull", NISS1_DENT_OA500, "500", "DENT"});
		return testData;
	}

	@Test
	public void sendAttestationTest() throws Exception {
		InputReference inputReference = new InputReference();
		SendAttestBuilderRequest attestBuilderRequest = helper.buildRequest(inputReference, scenarioName, ssin, profession);
		SendAttestationResponse response = AttestSessionServiceFactory.getAttestService().sendAttestation(attestBuilderRequest.getSendAttestationRequest());
		SignedEncryptedBuilderResponse builderResponse = helper.getBuildSendAttestResponse(response, attestBuilderRequest);
		assertResult(inputReference, builderResponse, profession, oa, scenarioName);
	}

	private void assertResult(InputReference inputReference, SignedEncryptedBuilderResponse builderResponse, String profession, String oa, String scenarioName) throws Exception {
		assertEquals("SignatureVerificationResult should contain no error", 0, builderResponse.getSignatureVerificationResult().getErrors().size());
		XmlAsserter.assertSimilar(ConnectorXmlUtils.toObject(builderResponse.getSignedData(), EncryptedKnownContent.class), builderResponse.getRawDecryptedBlob());
		XmlAsserter.assertSimilar(ConnectorXmlUtils.toObject(builderResponse.getRawDecryptedBlob().getBusinessContent().getValue(),SendTransactionResponse.class), builderResponse.getSendTransactionResponse());
		assertNotNull("Missing xades", builderResponse.getXades());
		assertTrue("Xades different in blob and exposed xades", Arrays.equals(builderResponse.getRawDecryptedBlob().getXades(),builderResponse.getXades()));
		String expectedResponse = helper.buildExpectedSendAttestResponse(builderResponse, inputReference.getInputReference(), profession, oa, scenarioName);
		XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(builderResponse.getSendTransactionResponse()));
	}


}
