/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.mediprimav2.session;


import static be.ehealth.businessconnector.test.mediprimav2.builder.ConsultCarmedInterventionRequestBuilder.aRequest;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import be.ehealth.businessconnector.mediprimav2.session.MediprimaSessionServiceFactory;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.runners.Enclosed;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import be.ehealth.technicalconnector.exception.SoaErrorException;
import be.ehealth.technicalconnector.utils.DateUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionRequest;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * Integration tests for consultation mediprima v2 persphysician. The scenarios numbering match those of the MediPrima specification.
 *
 * @author EHP
 */
@RunWith(Enclosed.class)
public final class MediprimaConsultationPersphysicianIntegrationTest {


    @RunWith(Parameterized.class)
    public static class WithSoaErrorIntegrationTest {
        private static final String NISS_3 = "68050555876";
        private static final String NISS_2 = "60061861748";
        private static final String NISS_01 = "70410100375";
        private static final String NISS_10 = "85000132773";
        @ClassRule
        public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persphysician").forUser("hannes").build();
        private String scenarioId;
        private String scenarioName;
        private ConsultCarmedInterventionRequest request;


        public WithSoaErrorIntegrationTest(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
            super();
            this.scenarioId = scenarioId;
            this.scenarioName = scenarioName;
            this.request = request;
        }

        @Parameterized.Parameters(name = "{0}: {1}")
        public static Collection<Object[]> data() throws Exception {
            Collection<Object[]> testData = new ArrayList<Object[]>();
            testData.add(new Object[]{"scenario01", "SSIN invalid", aRequest().withSsin(NISS_10).withPeriod().build()});
            testData.add(new Object[]{"scenario02", "call doctor on period", aRequest().withSsin(NISS_01).withStartPeriod(new DateTime().minusDays(10)).withEndPeriod(new DateTime().plusDays(10)).build()});
            testData.add(new Object[]{"scenario03", "Consultation with date > today", aRequest().withSsin(NISS_01).withReferenceDate(new DateTime().plusDays(10)).build()});
            testData.add(new Object[]{"scenario04", "Consultation with date < today - 2 months", aRequest().withSsin(NISS_01).withReferenceDate(new DateTime().minusMonths(2)).build()});
            return testData;
        }

        @Test
        public void testScenario() throws Exception {
            try {
                MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);
                Assert.fail("should have thrown SoaErrorException");
            } catch (SoaErrorException e) {
                ConsultCarmedInterventionResponse response = (ConsultCarmedInterventionResponse) e.getResponseTypeV2();
                String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
                MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse> helper = new MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse>(ConsultCarmedInterventionResponse.class, ConsultCarmedInterventionResponse.class);
                XmlAsserter.assertSimilar(expected, helper.toString(response));
            }
        }
    }



    private static String generateXml(Object obj, Class objClass) throws JAXBException {
        JAXBContext jaxbContext = JAXBContext.newInstance(objClass);
        Marshaller jaxbMarshaller = jaxbContext.createMarshaller();
        jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
        StringWriter sw = new StringWriter();
        jaxbMarshaller.marshal(obj, sw);
        return sw.toString();
    }


    private static String expectedResponse(DateTime consultationDate, String requestId, String scenarioId) {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("requestId", requestId);
        velocityContext.put("consultationDate", DateUtils.printDate(consultationDate));
        String commentDate = DateTimeFormat.forPattern("dd/MM/yyyy").print(new DateTime());
        velocityContext.put("commentStartDate", commentDate);
        velocityContext.put("commentEndDate", commentDate);
        velocityContext.put("startDate", DateUtils.printDate(new DateTime()));
        velocityContext.put("endDate", DateUtils.printDate(new DateTime()));
        return TemplateEngineUtils.generate(velocityContext, "/examples/mediprimav2/consultation/expected/persphysician/" + scenarioId + "Response.xml");
    }
}
