/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.mycarenet.attestv2.session;

import static org.junit.Assert.assertEquals;

import java.util.Map;

import be.ehealth.businessconnector.mycarenet.attestv2.session.AttestSessionServiceFactory;
import be.ehealth.businessconnector.test.mycarenet.attestv2.helper.AttestIntegrationTestHelper;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;

import be.ehealth.businessconnector.mycarenet.attestv2.domain.CancelAttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SendAttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedBuilderResponse;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.CancelAttestationResponse;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.SendAttestationResponse;
import be.fgov.ehealth.standards.kmehr.mycarenet.schema.v1.Kmehrmessage;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * eAttest v2 Integration Test for cancel with profile physician. The scenarios numbering matches one specified by mycarenet.
 *
 * @author EHP
 */
public class CancelAttestPhysicianIntegrationTest {

    private static final String NISS1_MED_OA500 = "62110906574";

    private static final String NISS1_MED_OA100 = "71010639352";

    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persphysician").forUser("hannes").build();

    private AttestIntegrationTestHelper helper = new AttestIntegrationTestHelper();
    
    private static final String OA_100_INVOICE_NUMBER = "100-1-190226-9851229-88";

    private static final String OA_500_INVOICE_NUMBER = "500-1-190110-4567890-16";


    private static final String MED = "MED";

    private static final String OA_100 = "100";

    private static final String OA_500 = "500";

    
    @Test
    public void cancelAttestationOA100MED5a() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5a";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA100, MED, OA_100_INVOICE_NUMBER);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA500MED5a() throws Exception {

        InputReference inputReference = new InputReference();
        String scenarioName = "MED5a";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA500, MED, OA_500_INVOICE_NUMBER);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA100MED5b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5b";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA100, MED, OA_100_INVOICE_NUMBER);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA500MED5b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5b";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA500, MED, OA_500_INVOICE_NUMBER);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA100MED5c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5c";
        
        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA100, MED, OA_100_INVOICE_NUMBER);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA500MED5c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5c";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA500, MED, OA_500_INVOICE_NUMBER);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response, attestBuilderRequest);
    }
    
    @Test
    public void cancelAttestationOA100MED6() throws Exception {
        InputReference inputReference = new InputReference();
        SendAttestBuilderRequest builderRequest = helper.buildRequest(new InputReference(), "MED4", NISS1_MED_OA100, MED);
        SendAttestationResponse sendResponse = AttestSessionServiceFactory.getAttestService()
                .sendAttestation(builderRequest.getSendAttestationRequest());
        String invoiceNumber = helper.getInvoiceNumber(sendResponse, builderRequest);
        String scenarioName = "MED6";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA100, MED, invoiceNumber);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA500MED6() throws Exception {
        InputReference inputReference = new InputReference();
        SendAttestBuilderRequest builderRequest = helper.buildRequest(inputReference, "MED4", NISS1_MED_OA500, MED);
        SendAttestationResponse sendResponse = AttestSessionServiceFactory.getAttestService()
                .sendAttestation(builderRequest.getSendAttestationRequest());
        String invoiceNumber = helper.getInvoiceNumber(sendResponse, builderRequest);
        String scenarioName = "MED6";


        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_MED_OA500, MED, invoiceNumber);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response, attestBuilderRequest);
    }

    private Kmehrmessage getKmehrMessage(String scenarioName, String profession, Map<String, Object> velocityContext) {
        String kmerh = TemplateEngineUtils.generate(velocityContext, "/scenarios/requests/" + profession + "/" + scenarioName + "_request" + ".xml");

        MarshallerHelper<Kmehrmessage, Kmehrmessage> kmehrRequestMarshaller = new MarshallerHelper<Kmehrmessage, Kmehrmessage>(Kmehrmessage.class, Kmehrmessage.class);
        return kmehrRequestMarshaller.toObject(kmerh);
    }

    private void assertCancelResult(InputReference inputReference, String scenarioName, String oa, String profession,
                                    CancelAttestationResponse response, CancelAttestBuilderRequest cancelAttestBuilderRequest) throws Exception {
        SignedBuilderResponse builderResponse = helper.getBuildCancelAttestResponse(response, cancelAttestBuilderRequest.getCancelAttestationRequest());
        assertEquals("SignatureVerificationResult should contain no error", 0, builderResponse.getSignatureVerificationResult().getErrors().size());
        // check signedData is same as content of the business blob in the WS reponse
        Assert.assertEquals(response.getReturn().getDetail().getValue(), builderResponse.getSignedData());
        // check xadesT is same as the one from the WS reponse
        Assert.assertEquals(response.getReturn().getXadesT().getValue(), builderResponse.getXadesT());
        String expectedResponse = helper.buildExpectedCancelAttestResponse(builderResponse, inputReference.getInputReference(), scenarioName, oa, profession);
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(builderResponse.getSendTransactionResponse()));
    }
}
