/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.wsconsent.builders;

import java.util.ArrayList;
import java.util.List;

import be.ehealth.businessconnector.test.wsconsent.utils.TestUtils;
import be.ehealth.businessconnector.wsconsent.builders.RequestObjectBuilderFactory;
import org.joda.time.DateTime;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import be.ehealth.business.kmehrcommons.CDConsentBuilderUtil;
import be.ehealth.businessconnector.wsconsent.exception.WsConsentBusinessConnectorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.session.Session;
import be.fgov.ehealth.hubservices.core.v2.AuthorWithPatientAndPersonType;
import be.fgov.ehealth.hubservices.core.v2.ConsentType;
import be.fgov.ehealth.hubservices.core.v2.PatientIdType;
import be.fgov.ehealth.hubservices.core.v2.SelectGetPatientConsentType;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENT;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENTvalues;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDHCPARTYschemes;
import be.fgov.ehealth.standards.kmehr.id.v1.IDHCPARTYschemes;
import be.fgov.ehealth.standards.kmehr.id.v1.IDPATIENTschemes;
import be.fgov.ehealth.standards.kmehr.schema.v1.HcpartyType;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;
import be.fgov.ehealth.technicalconnector.tests.utils.TestPropertiesLoader;


/**
 * ConsentBuilderImplTest class test.
 * 
 * @author EHP
 * 
 * 
 * 
 */
public class ConsentBuilderImplTest {


    @BeforeClass
    public static void init() throws Exception {
        // only integration tests need to run for all professionTypes
        Assume.assumeTrue("persphysician".equals(TestPropertiesLoader.getProperties("/be.ehealth.businessconnector.wsconsent.test.properties").getProperty("session.professionType")));
        SessionInitializer.init("/be.ehealth.businessconnector.wsconsent.test.properties", false);
    }

    @AfterClass
    public static void tearDown() throws Exception {
        Session.getInstance().unloadSession();
    }

    @Test
    public void testCreateConsent() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {


        PatientIdType patient = TestUtils.createDummyPatient();

        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));

        // Author
        AuthorWithPatientAndPersonType author = RequestObjectBuilderFactory.getAuthorBuilder().createAuthor();

        ConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createNewConsent(patient, consentList, new DateTime(), author);

        // Patient
        // INSS
        Assert.assertEquals(IDPATIENTschemes.INSS, consentType.getPatient().getIds().get(0).getS());
        Assert.assertNotNull(consentType.getPatient().getIds().get(0).getValue());
        Assert.assertNotNull(consentType.getPatient().getFirstname());
        Assert.assertNotNull(consentType.getPatient().getFamilyname());
        // Eid
        Assert.assertEquals(IDPATIENTschemes.EID_CARDNO, consentType.getPatient().getIds().get(1).getS());
        Assert.assertNotNull(consentType.getPatient().getIds().get(1).getValue());


        // Author

        // NIHII
        HcpartyType hcpartyType = author.getHcparties().get(0);
        Assert.assertEquals(IDHCPARTYschemes.ID_HCPARTY, hcpartyType.getIds().get(0).getS());
        Assert.assertNotNull(hcpartyType.getIds().get(0).getValue());
        // INSS
        Assert.assertEquals(1, author.getHcparties().size());
        Assert.assertTrue(hcpartyType.getIds().size() > 1);
        Assert.assertEquals(IDHCPARTYschemes.INSS, hcpartyType.getIds().get(1).getS());
        Assert.assertNotNull(hcpartyType.getIds().get(1).getValue());
        // ProfessionType
        Assert.assertEquals(CDHCPARTYschemes.CD_HCPARTY, author.getHcparties().get(0).getCds().get(0).getS());
        Assert.assertNotNull(hcpartyType.getCds().get(0).getValue());

        Assert.assertNull(author.getPatient());
        Assert.assertNotNull(consentType.getSigndate());

    }

    @Test
    public void testCreateConsentWithNoPatient() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {

        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));

        // Author
        AuthorWithPatientAndPersonType author = RequestObjectBuilderFactory.getAuthorBuilder().createAuthor();

        ConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createNewConsent(null, consentList, new DateTime(), author);

        Assert.assertNotNull(consentType);
    }

    @Test
    public void testCreateConsentWithNoDate() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {

        PatientIdType patient = TestUtils.createDummyPatient();

        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));

        // Author
        AuthorWithPatientAndPersonType author = RequestObjectBuilderFactory.getAuthorBuilder().createAuthor();

        ConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createNewConsent(patient, consentList, null, author);

        Assert.assertNotNull(consentType);
    }

    @Test
    public void testCreateConsentWithNoAuthor() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {
        PatientIdType patient = TestUtils.createDummyPatient();

        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));


        ConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createNewConsent(patient, consentList, new DateTime(), null);

        Assert.assertNotNull(consentType);
    }


    @Test
    public void testCreateConsentForPut() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {

        // Patient
        PatientIdType patient = TestUtils.createDummyPatient();

        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));

        ConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createNewConsent(patient, consentList, null, null);


        // Patient
        // INSS
        Assert.assertEquals(IDPATIENTschemes.INSS, consentType.getPatient().getIds().get(0).getS());
        Assert.assertNotNull(consentType.getPatient().getIds().get(0).getValue());
        Assert.assertNotNull(consentType.getPatient().getFirstname());
        Assert.assertNotNull(consentType.getPatient().getFamilyname());
        // Eid
        Assert.assertEquals(IDPATIENTschemes.EID_CARDNO, consentType.getPatient().getIds().get(1).getS());
        Assert.assertNotNull(consentType.getPatient().getIds().get(1).getValue());

        Assert.assertNull(consentType.getSigndate());
        Assert.assertNull(consentType.getRevokedate());

    }

    @Test
    public void testCreateConsentForGet() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {

        // Patient
        PatientIdType patient = TestUtils.createDummyPatient();

        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));

        SelectGetPatientConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createSelectGetPatientConsent(patient, consentList);


        // Patient
        // INSS
        Assert.assertEquals(IDPATIENTschemes.INSS, consentType.getPatient().getIds().get(0).getS());
        Assert.assertNotNull(consentType.getPatient().getIds().get(0).getValue());
        Assert.assertNotNull(consentType.getPatient().getFirstname());
        Assert.assertNotNull(consentType.getPatient().getFamilyname());
        // Eid
        Assert.assertEquals(IDPATIENTschemes.EID_CARDNO, consentType.getPatient().getIds().get(1).getS());
        Assert.assertNotNull(consentType.getPatient().getIds().get(1).getValue());


    }

}
