﻿/*
 * Copyright (c) eHealth
 */
using System;
using be.ehealth.technicalconnector.session;
using ikvm.extensions;
using NUnit.Framework;
using lang = java.lang;
using java.util;
using joda = org.joda.time;
using be.ehealth.businessconnector.ehbox.api.domain;
using be.ehealth.businessconnector.ehbox.api.domain.exception;
using be.ehealth.businessconnector.ehbox.api.utils;
using be.ehealth.businessconnector.ehbox.v3.builders;
using be.ehealth.businessconnector.ehbox.v3.exception;
using be.ehealth.technicalconnector.exception;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.ehealth.technicalconnector.utils;
using publication = be.fgov.ehealth.ehbox.publication.protocol.v3;
using be.fgov.ehealth.ehbox.core.v3;
using be.fgov.ehealth.ehbox.publication.protocol.v3;
using be.ehealth.businessconnector.ehbox.v3.session;
using be.ehealth.businessconnector.ehbox.v3.util;
using be.fgov.ehealth.ehbox.consultation.protocol.v3;
using be.fgov.ehealth.businessconnector.tests.utils;
using be.fgov.ehealth.technicalconnector.tests.utils;

namespace be.ehealth.businessconnector.ehboxv3.session
{
    [TestFixture]
    public class EhboxV3WalkthroughIntegrationTest
    {    
        private static String MESSAGE_TITLE = "Connector integration test";
        private static String MESSAGE_CONTENT = "message test connector integration test with special characters {}()6§7^2²³ü*µ~`ù`´´][|çà&";
        private static String MESSAGE_ANNEX_TITLE = "Annex.pdf";
        private static String MESSAGE_ANNEX_MIME_TYPE = "application/pdf";
        private static String MESSAGE_ANNEX_CONTENT = ".//examples//test.pdf";
        private static String MESSAGE_FREE_TEXT = "Free information test integration test";
        private static String MESSAGE_CUSTOM_META_KEY = "CustomMeta";
        private static String MESSAGE_CUSTOM_META_VALUE = "[Connector]";
        private static String EXPECTED_RESPONSE_CODE = "100";
        private static String INBOX = "INBOX";
        private static String BININBOX = "BININBOX";
        private static String SENTBOX = "SENTBOX";
        
        private String currentMessageId;
        
        [SetUp]
        public void init() {
            java.util.Properties props = new java.util.Properties();
            props.put("session.environment", "acc");
            props.put("session.professionType", "persphysician");
            props.put("session.username", "olivier");
            SessionInitializer.init(props, true, SupportedLanguages.NET.getAbbreviation());
        }

        [TearDown]
        public void tearDown() {
            SessionDestroyer.destroy();
        }

        static object[] testCases =
        {
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/plain", MessageEncryption.NOT_ENCRYPTED},
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/plain", MessageEncryption.ENCRYPTED},
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/plain", MessageEncryption.NOT_ENCRYPTED},
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/plain", MessageEncryption.ENCRYPTED},
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/html", MessageEncryption.NOT_ENCRYPTED},
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.DOCTOR_SSIN, "text/html", MessageEncryption.ENCRYPTED},
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/html", MessageEncryption.NOT_ENCRYPTED},
            new Object[]{"74123045193", IdentifierType.SSIN, QualityType.PHARMACIST_SSIN, "text/html", MessageEncryption.ENCRYPTED}
        };
        
        [Test, TestCaseSource("testCases")]
        public void walkThrough(String destinationId,
                         IdentifierType destinationType,
                         QualityType destinationQuality,
                         String mimeType,
                         MessageEncryption messageEncryption) {
        	
            sendMessage(destinationId, destinationType, destinationQuality, mimeType, messageEncryption);           
            getMessageAcknowledgements(destinationId, destinationType, destinationQuality, mimeType, messageEncryption);
            getMessageList(destinationId, destinationType, destinationQuality, mimeType, messageEncryption);
            getAllEhboxesMessagesList(destinationId, destinationType, destinationQuality, mimeType, messageEncryption);
            getFullMessage(destinationId, destinationType, destinationQuality, mimeType, messageEncryption);
            moveMessage(destinationId, destinationType, destinationQuality, mimeType, messageEncryption);
            deleteMessage(destinationId, destinationType, destinationQuality, mimeType, messageEncryption);
        }

        private void sendMessage(String destinationId,
                                 IdentifierType destinationType,
                                 QualityType destinationQuality,
                                 String mimeType,
                                 MessageEncryption messageEncryption) {
            Addressee destination = EhboxTestDataBuilder.generateDestination(destinationId, destinationType, destinationQuality);
            MessageBuilder builder = MessageBuilder.create(destination, MESSAGE_CONTENT)
                    .withTitle(MESSAGE_TITLE)
                    .withMimeType(mimeType)
                    .withEncrypted(isEncrypted(messageEncryption))
                    .withImportant(true)
                    .withReceiveAck(true)
                    .withPublicationAck(true)
                    .withReadAck(true);
            DocumentMessage message = EhboxTestDataBuilder.generateDocumentMessage(builder);

            message.getAnnexList().add(EhboxTestDataBuilder
                    .generateDocument(MESSAGE_ANNEX_TITLE, MESSAGE_ANNEX_MIME_TYPE, ConnectorIOUtils.getBytes(ConnectorIOUtils.getResourceAsStream(MESSAGE_ANNEX_CONTENT))));

            message.setFreeText(MESSAGE_FREE_TEXT);

            message.getCustomMetas().put(MESSAGE_CUSTOM_META_KEY, MESSAGE_CUSTOM_META_VALUE);

            SendMessageRequest request = BuilderFactory.getSendMessageBuilder().buildMessage(message);
            SendMessageResponse response = ServiceFactory.getEhealthBoxServiceV3().sendMessage(request);
            currentMessageId = response.getId();

            Assert.AreEqual(EXPECTED_RESPONSE_CODE, response.getStatus().getCode());
            Assert.IsNotNull("Message ID should not be null", currentMessageId);

            EhboxTestUtilities.waitForProcessing();
        }

        private void getMessageAcknowledgements(String destinationId,
                     IdentifierType destinationType,
                     QualityType destinationQuality,
                     String mimeType,
                     MessageEncryption messageEncryption) {
            GetMessageAcknowledgmentsStatusResponse acks = EhboxTestUtilities.getMessageAcknowledgements(currentMessageId, EhboxTestDataBuilder.getEhboxSender());
            Iterator it = acks.getAcknowledgmentsStatus().getRows().iterator();
            while (it.hasNext()) {
                be.fgov.ehealth.ehbox.consultation.protocol.v3.Row currentAcknowledgementsStatus = (be.fgov.ehealth.ehbox.consultation.protocol.v3.Row) it.next();
                Assert.IsNull(currentAcknowledgementsStatus.getRead(), "The message should not have been read yet");
                Assert.IsNull(currentAcknowledgementsStatus.getReceived(), "The message should not have been received");
            }
        }

        private void getMessageList(String destinationId,
                     IdentifierType destinationType,
                     QualityType destinationQuality,
                     String mimeType,
                     MessageEncryption messageEncryption) {
            Assert.IsTrue(EhboxTestUtilities.isMessageInList(currentMessageId, INBOX,
                    boxId(destinationId, destinationType, destinationQuality)), "The message with id " + currentMessageId + " should be in the list");
        }

        private void getAllEhboxesMessagesList(String destinationId,
                     IdentifierType destinationType,
                     QualityType destinationQuality,
                     String mimeType,
                     MessageEncryption messageEncryption) {
            Assert.IsTrue(EhboxTestUtilities.isMessageInAllEhBoxesList(currentMessageId, INBOX), "The message with id " + currentMessageId + " should be in the list");
        }

        private void getFullMessage(String destinationId,
                     IdentifierType destinationType,
                     QualityType destinationQuality,
                     String mimeType,
                     MessageEncryption messageEncryption) {
            GetFullMessageResponse fullMessage = EhboxTestUtilities.getFullMessage(currentMessageId, INBOX,
                    boxId(destinationId, destinationType, destinationQuality));

            be.ehealth.businessconnector.ehbox.api.domain.Message retrievedFullMessage = BuilderFactory.getConsultationMessageBuilder().buildFullMessage(fullMessage);

            DocumentMessage inboxMessage = (DocumentMessage) retrievedFullMessage;

            String messageTitle = inboxMessage.getDocumentTitle();
            Assert.AreEqual(MESSAGE_TITLE, messageTitle, "Message title should be \"" + MESSAGE_TITLE + "\" but was \"" + messageTitle + "\"");
            Assert.AreEqual(System.Text.Encoding.UTF8.GetString(inboxMessage.getBody().getContent()), System.Text.Encoding.UTF8.GetString(System.Text.Encoding.Default.GetBytes(MESSAGE_CONTENT)));
            Assert.AreEqual(mimeType, ((GetFullMessageResponse) inboxMessage.getOriginal()).getMessage().getContentContext().getContent().getDocument().getMimeType());
            Assert.AreEqual(isEncrypted(messageEncryption), inboxMessage.isEncrypted(), "The message should " + (isEncrypted(messageEncryption) ? "be encrypted" : "not be encryted"));
            Assert.IsTrue(inboxMessage.isImportant(), "The message should be important");
            Assert.IsTrue(inboxMessage.isHasAnnex(), "The message should have an annex");
            Assert.AreEqual(1, inboxMessage.getAnnexList().size(), "The message should have 1 annex");
            Assert.IsTrue(inboxMessage.isHasFreeInformations(), "The message should have some free information");
            Assert.AreEqual(1, inboxMessage.getCustomMetas().size(), "The message should have a custom metadata");

            String annexTitle = ((Document) inboxMessage.getAnnexList().get(0)).getTitle();
            Assert.AreEqual(MESSAGE_ANNEX_TITLE, annexTitle, "The message annex title should be \"" + MESSAGE_ANNEX_TITLE + "\" but was \"" + annexTitle + "\"");
            Assert.AreEqual(MESSAGE_ANNEX_MIME_TYPE, ((Document) inboxMessage.getAnnexList().get(0)).getMimeType());

            String customMeta = (String) inboxMessage.getCustomMetas().get(MESSAGE_CUSTOM_META_KEY);
            Assert.AreEqual(MESSAGE_CUSTOM_META_VALUE, customMeta, "The message custom metadata should be \"" + MESSAGE_CUSTOM_META_VALUE + "\" but was \"" + customMeta + "\"");
        }

        private void moveMessage(String destinationId,
                     IdentifierType destinationType,
                     QualityType destinationQuality,
                     String mimeType,
                     MessageEncryption messageEncryption) {
            MoveMessageResponse response = ServiceFactory.getEhealthBoxServiceV3().moveMessage(BuilderFactory.getRequestBuilder().createMoveMessageRequest(INBOX, BININBOX,
                    boxId(destinationId, destinationType, destinationQuality), currentMessageId));
            Assert.AreEqual(EXPECTED_RESPONSE_CODE, response.getStatus().getCode(), "The message should have been moved");
        }

        private void deleteMessage(String destinationId,
                     IdentifierType destinationType,
                     QualityType destinationQuality,
                     String mimeType,
                     MessageEncryption messageEncryption) {
            Set messageSet = new HashSet();
            messageSet.add(currentMessageId);
            EhboxTestUtilities.deleteMessages(BININBOX, messageSet, boxId(destinationId, destinationType, destinationQuality));
            EhboxTestUtilities.deleteMessages(SENTBOX, messageSet, EhboxTestDataBuilder.getEhboxSender());
        }

        private BoxIdType boxId(String destinationId,
                 IdentifierType destinationType,
                 QualityType destinationQuality) {
            return EhboxTestDataBuilder.generateBoxId(destinationId, destinationType.getType(IdentifierType.EHBOX), destinationQuality.getQuality());
        }

        private bool isEncrypted(MessageEncryption messageEncryption) {
        	return MessageEncryption.ENCRYPTED.Equals(messageEncryption);
        }
        
        public enum MessageEncryption {
            ENCRYPTED, NOT_ENCRYPTED
        }
    }
}

