/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.mediprima.session;

import be.ehealth.business.kmehrcommons.HcPartyUtil;
import be.ehealth.business.mycarenetcommons.mapper.SendRequestMapper;
import be.ehealth.business.mycarenetdomaincommons.builders.BlobBuilderFactory;
import be.ehealth.business.mycarenetdomaincommons.domain.Blob;
import be.ehealth.business.mycarenetdomaincommons.domain.CareReceiverId;
import be.ehealth.business.mycarenetdomaincommons.domain.Routing;
import be.ehealth.businessconnector.mediprima.session.MediprimaSessionServiceFactory;
import be.ehealth.businessconnector.mediprima.session.MediprimaTarificationSessionService;
import be.ehealth.businessconnector.tarification.builder.RequestBuilder;
import be.ehealth.businessconnector.tarification.builder.TarificationRequestBuilderFactory;
import be.ehealth.businessconnector.tarification.helper.ResponseHelper;
import be.ehealth.technicalconnector.config.ConfigFactory;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.idgenerator.IdGeneratorFactory;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.SessionUtil;
import be.fgov.ehealth.messageservices.core.v1.RetrieveTransactionResponse;
import be.fgov.ehealth.mycarenet.commons.core.v2.BlobType;
import be.fgov.ehealth.mycarenet.commons.protocol.v2.TarificationConsultationRequest;
import be.fgov.ehealth.mycarenet.commons.protocol.v2.TarificationConsultationResponse;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDERRORMYCARENET;
import be.fgov.ehealth.standards.kmehr.schema.v1.ErrorMyCarenetType;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.joda.time.DateTime;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Integration tests for tarification mediprima. The scenarios numbering match those of the MediPrima specification.
 *
 * @author EHP
 */
@RunWith(Parameterized.class)
public final class MediprimaTarificationIntegrationTest {

    private static final Logger LOG = LoggerFactory.getLogger(MediprimaTarificationIntegrationTest.class);


    @Parameters(name = "{0}: {1}")
    public static Collection<Object[]> data() throws Exception {
        Collection<Object[]> testData = new ArrayList<Object[]>();
        testData.add(new Object[]{"scenario01", "Unexisting nomenclature code", "28033135008", new DateTime(), Arrays.asList("101075"), new ExpectedResult("130")});
        testData.add(new Object[]{"scenario02", "Non general practitioner nomenclature code", "92010442932", new DateTime(), Arrays.asList("560011"), new ExpectedResult("171")});
        testData.add(new Object[]{"scenario03", "Medical specialist nomenclature code", "92010442932", new DateTime(), Arrays.asList("102034"), new ExpectedResult("171")});
        testData.add(new Object[]{"scenario04", "Consultation date < today - 2 months", "92010442932",  new DateTime().minusMonths(3), Arrays.asList("101032"), new ExpectedResult("166")});
        testData.add(new Object[]{"scenario05", "Consultation date > today", "28033135008", new DateTime().plusDays(2), Arrays.asList("101032"), new ExpectedResult("146")});
        testData.add(new Object[]{"scenario06", "No decision", "28033135008", new DateTime(), Arrays.asList("101032"), new ExpectedResult("202")});
        testData.add(new Object[]{"scenario07", "No decision for date", "35061540420", new DateTime(), Arrays.asList("101032"), new ExpectedResult("201")});
        testData.add(new Object[]{"scenario08", "Decision without required rights - missing coverage", "55051059009", new DateTime(), Arrays.asList("101010"), new ExpectedResult("207")});
        testData.add(new Object[]{"scenario09", "Decision without required rights - missing coverage with type physician", "76091051964", new DateTime(), Arrays.asList("101032"), new ExpectedResult("207")});
        testData.add(new Object[]{"scenario10", "Decision without required rights - coverage exists but nok", "61011060097", new DateTime(), Arrays.asList("101032"), new ExpectedResult("206")});
        testData.add(new Object[]{"scenario11", "Decision without required rights - coverage ok but care provider nok", "98032463545", new DateTime(), Arrays.asList("101032"), new ExpectedResult("204")});
        testData.add(new Object[]{"scenario12", "Decision pending", "69082054866", new DateTime(), Arrays.asList("101032"), new ExpectedResult("203")});
        testData.add(new Object[]{"scenario13", "Decision is active - full support", "64101567102", new DateTime(), Arrays.asList("101032"), new ExpectedResult(null)});
        testData.add(new Object[]{"scenario14", "Decision is active - partial support", "79030750196", new DateTime(), Arrays.asList("101032"), new ExpectedResult(null)});
        testData.add(new Object[]{"scenario15", "Decision is active - full support - multiple codes provided", "64101567102", new DateTime(), Arrays.asList("101032", "475075"), new ExpectedResult(null)});

        return testData;
    }


    @Rule
    public SessionRule sessionRule = SessionRule.withActiveSession().baseOn("/be.ehealth.businessconnector.mediprima.test.properties").build();

    private String scenarioId;
    private String scenarioName;
    private String niss;
    private DateTime dateTime;
    private List<String> claims;
    private ExpectedResult expectedResult;


    public MediprimaTarificationIntegrationTest(String scenarioId, String scenarioName, String niss, DateTime dateTime, List<String> claims, ExpectedResult expectedResult) {
        super();
        this.scenarioId = scenarioId;
        this.scenarioName = scenarioName;
        this.niss = niss;
        this.dateTime = dateTime;
        this.claims = claims;
        this.expectedResult = expectedResult;
    }


    @Test
    public void testScenario() throws Exception {
        TarificationConsultationRequest request = givenRequestWith(niss, dateTime, claims);

        byte[] response = whenConsultingTarification(request);

        then(expectedResult, response);
    }


    private TarificationConsultationRequest givenRequestWith(String niss, DateTime dateTime, List<String> claims) throws TechnicalConnectorException {
        RequestBuilder requestBuilder = TarificationRequestBuilderFactory.getRequestObjectBuilder();
        InputStream businessXmlContentStream = buildRequest(claims);
        XmlTestRequest xmlTestRequest = replaceVariableTagsWithContent(niss, dateTime, claims, new String(ConnectorIOUtils.getBytes(businessXmlContentStream)));
        String businessXml = xmlTestRequest.textContent;
        LOG.debug("used content : \n" + businessXml);
        String inputReference = IdGeneratorFactory.getIdGenerator("xsid").generateId();
        TarificationConsultationRequest request = requestBuilder.buildConsultationRequest(getRoutingForNiss(niss), businessXml.getBytes(), inputReference);
        request.getCommonInput().setInputReference(xmlTestRequest.kmehrId);
        request.getRouting().setReferenceDate(xmlTestRequest.time);
        return request;
    }

    private byte[] whenConsultingTarification(TarificationConsultationRequest request) throws ConnectorException {
        MediprimaTarificationSessionService session = MediprimaSessionServiceFactory.getTarificationMediprimaSession();
        TarificationConsultationResponse consultTarificationResponse = session.consultTarification(request);
        BlobType detail = consultTarificationResponse.getReturn().getDetail();
        Blob blob = SendRequestMapper.mapBlobTypeToBlob(detail);
        byte[] content = BlobBuilderFactory.getBlobBuilder("mediprima.tarification").checkAndRetrieveContent(blob);
        String responseContent = new String(content);
        LOG.debug("content : \n" + responseContent);
        return content;
    }

    private void then(ExpectedResult expectedResult, byte[] content) throws Exception {
        if (expectedResult.errorCode != null) {
            assertErrorCode(expectedResult, content);
        } else {
            XmlAsserter.assertSimilar(ConnectorIOUtils.getResourceAsString("/tarification/expected/" + scenarioId + "Response.xml"), new String(content));
        }
    }

    private void assertErrorCode(ExpectedResult expectedResult, byte[] content) throws TechnicalConnectorException {
        MarshallerHelper<RetrieveTransactionResponse, RetrieveTransactionResponse> helper = new MarshallerHelper<RetrieveTransactionResponse, RetrieveTransactionResponse>(RetrieveTransactionResponse.class, RetrieveTransactionResponse.class);
        RetrieveTransactionResponse commonInputResponse = helper.toObject(content);
        ResponseHelper.validateResponse(commonInputResponse);
        Boolean flagFound = false;
        for (ErrorMyCarenetType error : commonInputResponse.getAcknowledge().getErrors()) {
            for (CDERRORMYCARENET code : error.getCds()) {
                if (expectedResult.errorCode.equals(code.getValue())) {
                    flagFound = true;
                }
            }
        }
        if (!flagFound) {
            Assert.fail("Wait for error code " + expectedResult.errorCode + " but not found in response");
        }
    }

    private InputStream buildRequest(List<String> claims) throws TechnicalConnectorException {
        return ConnectorIOUtils.getResourceAsStream("/tarification/request/" + (claims.size() > 1 ? "multipleCodesRequestTemplate.xml" : "requestTemplate.xml"));
    }

    private XmlTestRequest replaceVariableTagsWithContent(String niss, DateTime dateTime, List<String> claims, String testFileContent) {
        String khmerId = dateTime.toString("YYYYddhhmmssSS");
        testFileContent = testFileContent.replaceAll("\\$\\{currentTimestamp\\}", khmerId);
        String date = dateTime.toString("YYYY-MM-dd");
        testFileContent = testFileContent.replaceAll("\\$\\{currentDate\\}", date);
        String time = dateTime.toString("hh:mm:ss");
        testFileContent = testFileContent.replaceAll("\\$\\{currentTime\\}", time);
        String userNihiiNumber = retrieveCareProviderNihii();
        testFileContent = testFileContent.replaceAll("\\$\\{nihiiNumber\\}", userNihiiNumber);
        testFileContent = testFileContent.replaceAll("\\$\\{lastName\\}", SessionUtil.getLastname());
        testFileContent = testFileContent.replaceAll("\\$\\{firstName\\}", SessionUtil.getFirstname());
        String nissNumber = retrieveCareProviderNiss();
        testFileContent = testFileContent.replaceAll("\\$\\{niss\\}", nissNumber);
        testFileContent = testFileContent.replaceAll("\\$\\{professionType\\}", HcPartyUtil.getAuthorKmehrQuality());
        testFileContent = testFileContent.replaceAll("\\$\\{patientNiss\\}", niss);
        Iterator<String> it = claims.iterator();
        for (int i = 0; it.hasNext(); i++) {
            testFileContent = testFileContent.replaceAll("\\$\\{claim" + i +"\\}", it.next());
        }

        LOG.trace("xml content after replacing tags : \n" + testFileContent);
        return new XmlTestRequest(testFileContent, khmerId, dateTime);
    }


    private String retrieveCareProviderNiss() {
        return ConfigFactory.getConfigValidator().getProperty("user.inss");
    }


    private String retrieveCareProviderNihii() {
        return ConfigFactory.getConfigValidator().getProperty("user.nihii");
    }

    private static Routing getRoutingForNiss(String niss) {
        CareReceiverId careReceiver = new CareReceiverId(niss);
        return new Routing(careReceiver, new DateTime());
    }



    private final class XmlTestRequest {
        private String textContent;
        private String kmehrId;
        private DateTime time;

        private XmlTestRequest(String textContent, String kmehrId, DateTime time) {
            super();
            this.textContent = textContent;
            this.kmehrId = kmehrId;
            this.time = time;
        }
    }


    private static final class ExpectedResult {
        private String errorCode;

        private ExpectedResult(String errorCode) {
            super();
            this.errorCode = errorCode;
        }
    }
}
