/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.mycarenet.attestv2.session;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import be.ehealth.businessconnector.mycarenet.attestv2.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedEncryptedBuilderResponse;
import be.ehealth.businessconnector.mycarenet.attestv2.helper.AttestIntegrationTestHelper;
import be.fgov.ehealth.messageservices.mycarenet.core.v1.SendTransactionResponse;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import be.cin.encrypted.EncryptedKnownContent;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SendAttestBuilderRequest;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.SendAttestationResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * eAttest v2 Integration Test for send with profile physician. The scenarios numbering matches one specified by mycarenet.
 *
 * @author EHP
 */
@RunWith(Parameterized.class)
public class SendAttestPhysicianIntegrationTest {

    private static final String NISS4_MED_OA500 = "58031245635";

    private static final String NISS5_MED_OA500 = "46111636603";

    private static final String NISS1_MED_OA500 = "62110906574";

    private static final String NISS4_MED_OA100 = "29041433972";

    private static final String NISS5_MED_OA100 = "85020205931";

    private static final String NISS1_MED_OA100 = "71010639352";

    private static final String MED = "MED";

    private static final String OA_500 = "500";

    private static final String OA_100 = "100";

    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persphysician").forUser("hannes").build();

    private AttestIntegrationTestHelper helper = new AttestIntegrationTestHelper();

    private String scenarioName;
    private String ssin;
    private String oa;
    private String profession;


    public SendAttestPhysicianIntegrationTest(String scenarioName, String ssin, String oa, String profession) {
        super();
        this.scenarioName = scenarioName;
        this.ssin = ssin;
        this.oa = oa;
        this.profession = profession;
    }


    @Parameterized.Parameters(name = "{0},{1},{2}")
    public static Collection<Object[]> data() {


        Collection<Object[]> testData = new ArrayList<Object[]>();

        // OA 100
        testData.add(new Object[]{"MED1a", NISS1_MED_OA100, OA_100, MED});
        testData.add(new Object[]{"MED1b", NISS1_MED_OA100, OA_100, MED});
        testData.add(new Object[]{"MED2", NISS5_MED_OA100, OA_100, MED});
        testData.add(new Object[]{"MED3", NISS5_MED_OA100, OA_100, MED});
        testData.add(new Object[]{"MED4", NISS1_MED_OA100, OA_100, MED});
        testData.add(new Object[]{"MEDFullOA100", NISS4_MED_OA100, OA_100, MED});

        // OA 500
        testData.add(new Object[]{"MED1a", NISS1_MED_OA500, OA_500, MED});
        testData.add(new Object[]{"MED1b", NISS1_MED_OA500, OA_500, MED});
        testData.add(new Object[]{"MED2", NISS5_MED_OA500, OA_500, MED});
        testData.add(new Object[]{"MED3", NISS5_MED_OA500, OA_500, MED});
        testData.add(new Object[]{"MED4", NISS1_MED_OA500, OA_500, MED});
        testData.add(new Object[]{"MEDFullOA500", NISS4_MED_OA500, OA_500, MED});
        return testData;
    }


    @Test
    public void sendAttestationTest() throws Exception {
        InputReference inputReference = new InputReference();
        SendAttestBuilderRequest attestBuilderRequest = helper.buildRequest(inputReference, scenarioName, ssin, profession);
        SendAttestationResponse response = AttestSessionServiceFactory.getAttestService().sendAttestation(attestBuilderRequest.getSendAttestationRequest());
        SignedEncryptedBuilderResponse builderResponse = helper.getBuildSendAttestResponse(response, attestBuilderRequest);
        assertResult(inputReference, builderResponse, profession, oa, scenarioName);
    }


    private void assertResult(InputReference inputReference, SignedEncryptedBuilderResponse builderResponse, String profession, String oa, String scenarioName) throws Exception {
        assertEquals("SignatureVerificationResult should contain no error", 0, builderResponse.getSignatureVerificationResult().getErrors().size());
        XmlAsserter.assertSimilar(ConnectorXmlUtils.toObject(builderResponse.getSignedData(), EncryptedKnownContent.class), builderResponse.getRawDecryptedBlob());
        XmlAsserter.assertSimilar(ConnectorXmlUtils.toObject(builderResponse.getRawDecryptedBlob().getBusinessContent().getValue(),
                SendTransactionResponse.class), builderResponse.getSendTransactionResponse());
        assertNotNull("Missing xades", builderResponse.getXades());
        assertTrue("Xades different in blob and exposed xades", Arrays.equals(builderResponse.getRawDecryptedBlob().getXades(), builderResponse.getXades()));
        String expectedResponse = helper.buildExpectedSendAttestResponse(builderResponse, inputReference.getInputReference(), profession, oa, scenarioName);
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(builderResponse.getSendTransactionResponse()));
    }


}
