/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.chapterIV.integration;

import be.ehealth.businessconnector.chapterIV.builders.BuilderFactory;
import be.ehealth.businessconnector.chapterIV.builders.ResponseBuilder;
import be.ehealth.businessconnector.chapterIV.exception.ChapterIVBusinessConnectorException;
import be.ehealth.technicalconnector.exception.SoaErrorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.fgov.ehealth.chap4.core.v1.DetailType;
import be.fgov.ehealth.chap4.core.v1.FaultType;
import be.fgov.ehealth.errors.core.v1.LocalisedStringType;
import be.fgov.ehealth.errors.soa.v1.SOAErrorType;
import be.fgov.ehealth.standards.kmehr.schema.v1.FolderType;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.SAMLTokenUtils;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.List;


/**
 * abstract super class for ChapterIv integration tests.
 *
 * @author EHP
 */
public abstract class AbstractChapterIVTestBase {

    private static ResponseBuilder responseBuilder;

    private static final Logger LOG = LoggerFactory.getLogger(ChapterIVConsultationIntegrationTest.class);

    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().baseOn("/be.ehealth.businessconnector.chapterIV.test.properties").build();


    @BeforeClass
    public static void setup() throws Exception {
        responseBuilder = BuilderFactory.getBuilderFactoryForSession().getResponseBuilder();
    }


    protected void process(String service, ChapterIVBusinessConnectorException e) {
        StringBuilder sb = new StringBuilder();
        appendAndLog(sb, e.getMessage());

        SOAErrorType soaError = e.getSOAError();
        if (soaError != null) {
            String errorCode = soaError.getCode();
            appendAndLog(sb, "ChapterIVIntegrationTest.test" + service + "Request: error : " + errorCode);
            appendAndLog(sb, "contact :" + soaError.getContact());
            appendAndLog(sb, "environment : " + soaError.getEnvironment());
            appendAndLog(sb, "id :" + soaError.getId());
            appendAndLog(sb, "Origin " + soaError.getOrigin());
            appendAndLog(sb, "isRetrye : " + soaError.isRetry());
            for (LocalisedStringType localisedString : soaError.getMessages()) {
                appendAndLog(sb, "message : lang=" + localisedString.getLang() + " and value = " + localisedString.getValue());
            }
        }
        Assert.fail(sb.toString());
    }


    protected void process(String service, SoaErrorException e) {
        StringBuilder sb = new StringBuilder();
        String errorCode = e.getErrorCode();
        appendAndLog(sb, "ChapterIVIntegrationTest.test" + service + "Request: error : " + errorCode);
        FaultType fault = responseBuilder.retrieveReturnInfo(e.getResponseType());
        String errorMessage = "";
        if (fault.getMessage() != null) {
            errorMessage = fault.getMessage().getValue();
        }
        appendAndLog(sb, "errorMessage :" + errorMessage);
        for (DetailType detail : fault.getDetails()) {
            appendAndLog(sb, "detail : " + detail.getDetailCode() + " " + detail.getDetailSource() + " " + detail.getLocation() + " " + (detail.getMessage() == null ? "no messages" : detail.getMessage().getValue()));
        }
        Assert.fail(sb.toString());
    }

    protected void process(String service, Exception e) throws Exception {
        throw e;
    }


    protected FolderType parseFolderType(InputStream xmlResourceInputStream) throws TechnicalConnectorException {
        MarshallerHelper<FolderType, FolderType> folderTypeMarshaller = new MarshallerHelper<FolderType, FolderType>(FolderType.class, FolderType.class);
        return folderTypeMarshaller.toObject(xmlResourceInputStream);
    }


    protected void appendAndLog(StringBuilder sb, String message) {
        LOG.error(message);
        sb.append(message).append("\n");
    }


    public static ResponseBuilder getResponseBuilder() {
        return responseBuilder;
    }


    protected static boolean isDoctor() throws TechnicalConnectorException {
        List<String> result = SAMLTokenUtils.getAttributeValue(Session.getInstance().getSession().getSAMLToken().getAssertion(), "doctor");
        return !result.isEmpty();
    }

    protected static boolean isPharmacy() throws TechnicalConnectorException {
        List<String> result = SAMLTokenUtils.getAttributeValue(Session.getInstance().getSession().getSAMLToken().getAssertion(), "pharmacy");
        return !result.isEmpty();
    }
}
