/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.vsbnetinsurability.session;

import be.ehealth.businessconnector.testcommons.utils.FileTestUtils;
import be.ehealth.technicalconnector.idgenerator.IdGeneratorFactory;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import be.fgov.ehealth.vsb.insurability.protocol.v1.GetInsurabilityRequest;
import be.fgov.ehealth.vsb.insurability.protocol.v1.GetInsurabilityResponse;
import be.vlaanderen.mercurius.insurability.schemas.v1.WZCMHDF001DetermineInsurabilityResponse;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import javax.xml.ws.soap.SOAPFaultException;
import java.util.HashMap;
import java.util.Map;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

/**
 * Integration tests for GetInsurability
 *
 * @author EHP
 *
 */
public final class VSBNetInsurabilityIntegrationTest {

    @ClassRule
    public static SessionRule sessionRule = SessionRule.withActiveSession().baseOn("/be.ehealth.businessconnector.vsbnetinsurability.test.properties").build();

    @Rule
    public ExpectedException expectedEx = ExpectedException.none();

    @Test
    public void getInsurability_Success() throws Exception {
        GetInsurabilityResponse response = createGetInsurabilityRequest("/examples/request/GetInsurability-success.xml");
        WZCMHDF001DetermineInsurabilityResponse businessResponse = ConnectorXmlUtils.toObject(response.getReturn().getDetail().getValue(), WZCMHDF001DetermineInsurabilityResponse.class);
        assertEquals("35021198120", businessResponse.getPerson().getSsin());
        XmlAsserter.assertSimilar(ConnectorIOUtils.getResourceAsString("/examples/response/GetInsurability-success.xml"), ConnectorXmlUtils.toString(response));
    }

    @Test
    public void getInsurability_IncorrectNihii() throws Exception {
        try {
            createGetInsurabilityRequest("/examples/request/GetInsurability-IncorrectNihii.xml");
            fail("This test should send an exception");
        } catch (SOAPFaultException e) {
            assertEquals("INCORRECT_NIHII_XXX_SAML: For trussmaker the NIHII '70127777123' in the CareProvider element must correspond to the 'urn:be:fgov:person:ssin:ehealth:1.0:nihii:trussmaker:nihii11' attribute in the SAML '62141168200'.", e.getMessage());
        }
    }

    @Test
    public void getInsurability_IncorrectQuality() throws Exception {
        try {
            createGetInsurabilityRequest("/examples/request/GetInsurability-IncorrectQuality.xml");
            fail("This test should send an exception");
        } catch (SOAPFaultException e) {
            assertEquals("UNSUPPORTED_CAREPROVIDER: Not a supported care provider.", e.getMessage());
        }
    }
    
    private GetInsurabilityResponse createGetInsurabilityRequest(String testRequestFileLocation) throws Exception {
        Map<String, Object> templateContext = new HashMap<String, Object>();
        templateContext.put("issueInstant", DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'+02:00'").print(new DateTime()));
        templateContext.put("requestId", IdGeneratorFactory.getIdGenerator("uuid").generateId());
        templateContext.put("currentDate", DateTimeFormat.forPattern("yyyy-MM-dd").print(new DateTime()));
        
        GetInsurabilityRequest request = FileTestUtils.toObject(templateContext, testRequestFileLocation, GetInsurabilityRequest.class);
        return VSBNetInsurabilitySessionServiceFactory.getVSbNestInsurabilityService().getInsurability(request);
    }
}
