/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.mycarenet.attestv2.session;

import static org.junit.Assert.assertEquals;

import java.io.UnsupportedEncodingException;

import be.ehealth.businessconnector.mycarenet.attestv2.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedBuilderResponse;
import be.ehealth.businessconnector.mycarenet.attestv2.helper.AttestIntegrationTestHelper;
import org.junit.ClassRule;
import org.junit.Test;

import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.CancelAttestationResponse;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.SendAttestationResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * eAttest v2 Integration Test for cancel with profile physician. The scenarios numbering matches one specified by mycarenet.
 *
 * @author EHP
 */
public class CancelAttestPhysicianIntegrationTest {

    private static final String NISS1_MED_OA500 = "62110906574";

    private static final String NISS1_MED_OA100 = "71010639352";

    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persphysician").forUser("hannes").build();

    private AttestIntegrationTestHelper helper = new AttestIntegrationTestHelper();
    
    private static final String OA_100_ACK = "100-1-190226-9851229-88";

    private static final String OA_500_ACK = "500-1-190110-4567890-16";


    private static final String MED = "MED";

    private static final String OA_100 = "100";

    private static final String OA_500 = "500";


    @Test
    public void cancelAttestationOA100MED5a() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5a";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_MED_OA100, MED, OA_100_ACK);

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response);
    }

    @Test
    public void cancelAttestationOA500MED5a() throws Exception {

        InputReference inputReference = new InputReference();
        String scenarioName = "MED5a";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_MED_OA500, MED, OA_500_ACK);

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response);
    }

    @Test
    public void cancelAttestationOA100MED5b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5b";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_MED_OA100, MED, OA_100_ACK);

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response);
    }

    @Test
    public void cancelAttestationOA500MED5b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5b";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_MED_OA500, MED, OA_500_ACK);

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response);
    }

    @Test
    public void cancelAttestationOA100MED5c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5c";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_MED_OA100, MED, OA_100_ACK);

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response);
    }

    @Test
    public void cancelAttestationOA500MED5c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED5c";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_MED_OA500, MED, OA_500_ACK);

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response);
    }


    @Test
    public void cancelAttestationOA100MED6() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED6";
        SendAttestationResponse sendResponse = helper.sendAttestation(inputReference, "MED4", NISS1_MED_OA100, MED);
        String invoiceNumber = helper.getInvoiceNumber(sendResponse);

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_MED_OA100, MED, invoiceNumber);

        assertCancelResult(inputReference, scenarioName, OA_100, MED, response);
    }

    @Test
    public void cancelAttestationOA500MED6() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "MED6";
        String ssin = NISS1_MED_OA500;
        SendAttestationResponse sendResponse = helper.sendAttestation(inputReference, "MED4", ssin, MED);
        String invoiceNumber = helper.getInvoiceNumber(sendResponse);

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, ssin, MED, invoiceNumber);

        assertCancelResult(inputReference, scenarioName, OA_500, MED, response);
    }

    private void assertCancelResult(InputReference inputReference, String scenarioName, String oa, String profession, CancelAttestationResponse response) throws TechnicalConnectorException, UnsupportedEncodingException, Exception {
        SignedBuilderResponse builderResponse = helper.getBuildCancelAttestResponse(response);
        assertEquals("SignatureVerificationResult should contain no error", 0, builderResponse.getSignatureVerificationResult().getErrors().size());
        XmlAsserter.assertSimilar(ConnectorXmlUtils.toObject(builderResponse.getSignedData(), CancelAttestationResponse.class), response);
        String expectedResponse = helper.buildExpectedCancelAttestResponse(builderResponse, inputReference.getInputReference(), scenarioName, oa, profession);
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(builderResponse.getSendTransactionResponse()));
    }
}
