/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.mycarenet.attestv2.session;

import static org.junit.Assert.assertEquals;

import java.io.UnsupportedEncodingException;

import be.ehealth.businessconnector.mycarenet.attestv2.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedBuilderResponse;
import be.ehealth.businessconnector.mycarenet.attestv2.helper.AttestIntegrationTestHelper;
import org.junit.ClassRule;
import org.junit.Test;

import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.CancelAttestationResponse;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.SendAttestationResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * eAttest v2 Integration Test for cancel with profile dentist. The scenarios numbering matches one specified by mycarenet.
 *
 * @author EHP
 */
public class CancelAttestDentistIntegrationTest {

    private static final String NISS1_DENT_OA500 = "62110906574";

    private static final String NISS1_DENT_OA100 = "73052005540";

    private static final String OA_100_ACK = "100-1-190226-9851229-88";

    private static final String OA_500_ACK = "500-1-190110-4567890-16";

    private static final String DENT = "DENT";

    private static final String OA_100 = "100";

    private static final String OA_500 = "500";
    
    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persdentist").forUser("maxime").build();

    private AttestIntegrationTestHelper helper = new AttestIntegrationTestHelper();


    @Test
    public void cancelAttestationOA100DENT8a() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8a";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, OA_100_ACK);

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response);
    }


    @Test
    public void cancelAttestationOA500DENT8a() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8a";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, OA_500_ACK);

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response);
    }


    @Test
    public void cancelAttestationOA100DENT8b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8b";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, OA_100_ACK);

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response);
    }

    @Test
    public void cancelAttestationOA500DENT8b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8b";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, OA_500_ACK);

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response);
    }


    @Test
    public void cancelAttestationOA100DENT8c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8c";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, OA_100_ACK);

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response);
    }

    @Test
    public void cancelAttestationOA500DENT8c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8c";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, OA_500_ACK);

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response);
    }

    @Test
    public void cancelAttestationOA100DENT9() throws Exception {
        InputReference inputReference = new InputReference();
        SendAttestationResponse sendResponse = helper.sendAttestation(inputReference, "DENT4", NISS1_DENT_OA100, DENT);
        String ack = helper.getInvoiceNumber(sendResponse);
        String scenarioName = "DENT9";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, ack);

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response);
    }

    @Test
    public void cancelAttestationOA500DENT9() throws Exception {
        InputReference inputReference = new InputReference();
        SendAttestationResponse sendResponse = helper.sendAttestation(inputReference, "DENT9Send", NISS1_DENT_OA500, DENT);
        String ack = helper.getInvoiceNumber(sendResponse);
        String scenarioName = "DENT9";

        CancelAttestationResponse response = helper.cancelAttestation(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, ack);

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response);
    }

    private void assertCancelResult(InputReference inputReference, String scenarioName, String oa, String profession, CancelAttestationResponse response) throws TechnicalConnectorException, UnsupportedEncodingException, Exception {
        SignedBuilderResponse builderResponse = helper.getBuildCancelAttestResponse(response);
        assertEquals("SignatureVerificationResult should contain no error", 0, builderResponse.getSignatureVerificationResult().getErrors().size());
        XmlAsserter.assertSimilar(ConnectorXmlUtils.toObject(builderResponse.getSignedData(), CancelAttestationResponse.class), response);
        String expectedResponse = helper.buildExpectedCancelAttestResponse(builderResponse, inputReference.getInputReference(), scenarioName, oa, profession);
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(builderResponse.getSendTransactionResponse()));
    }
}
