package be.fgov.ehealth.technicalconnector.tests.junit.rule;

import java.util.Properties;

import org.junit.rules.ExternalResource;

/**
 * The {@code RestoreSystemProperties} rule undoes changes of system
 * properties when the test finishes (whether it passes or fails).
 * <p>Let's assume the system property {@code YourProperty} has the
 * value {@code YourValue}. Now run the test
 * <pre>
 *   public void YourTest {
 *     &#064;Rule
 *     public final TestRule restoreSystemProperties = new RestoreSystemProperties();
 *
 *     &#064;Test
 *     public void overrideProperty() {
 *       System.setProperty("YourProperty", "other value");
 *       assertEquals("other value", System.getProperty("YourProperty"));
 *     }
 *   }
 * </pre>
 * After running the test, the system property {@code YourProperty} has
 * the value {@code YourValue} again.
 */
public class RestoreSystemProperties extends ExternalResource {

    private Properties originalProperties;

    protected void before() throws Throwable {
        this.originalProperties = System.getProperties();
        System.setProperties(new Properties(this.originalProperties));
    }

    protected void after() {
        System.setProperties(this.originalProperties);
    }
}