/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.mediprima.session;


import be.ehealth.technicalconnector.exception.SoaErrorException;
import be.ehealth.technicalconnector.utils.DateUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mediprima.protocol.v1.ConsultCarmedInterventionRequest;
import be.fgov.ehealth.mediprima.protocol.v1.ConsultCarmedInterventionResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.RestoreSystemProperties;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.joda.time.DateTime;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.runners.Enclosed;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import static be.ehealth.businessconnector.mediprima.builder.ConsultCarmedInterventionRequestBuilder.aRequest;

/**
 * Integration tests for consultation mediprima for orgpharmacy. The scenarios numbering match those of the MediPrima specification.
 *
 * @author EHP
 */
@RunWith(Enclosed.class)
public final class MediprimaConsultationOrgpharmacyIntegrationTest {

    private static final Logger LOG = LoggerFactory.getLogger(MediprimaConsultationOrgpharmacyIntegrationTest.class);


    @RunWith(Parameterized.class)
    public static class WithSoaErrorIntegrationTest {
        @ClassRule
        public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("orgpharmacy").forUser("hannes").build();
        private String scenarioId;
        private String scenarioName;
        private ConsultCarmedInterventionRequest request;


        public WithSoaErrorIntegrationTest(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
            super();
            this.scenarioId = scenarioId;
            this.scenarioName = scenarioName;
            this.request = request;
        }

        @Parameterized.Parameters(name = "{0}: {1}")
        public static Collection<Object[]> data() throws Exception {
            Collection<Object[]> testData = new ArrayList<Object[]>();
            testData.add(new Object[]{"scenario01", "Invalid ssin", aRequest().withSsin("85000132773").withPeriod().build()});
            testData.add(new Object[]{"scenario02", "Consult with physician and period", aRequest().withSsin("86081523513").withPeriod().build()});
            testData.add(new Object[]{"scenario03", "Consultation date > today", aRequest().withSsin("86081523513").withReferenceDate(new DateTime().plusDays(10)).build()});
            testData.add(new Object[]{"scenario04", "Consultation date > today", aRequest().withSsin("86081523513").withReferenceDate(new DateTime().minusMonths(3)).build()});

            return testData;
        }

        @Test
        public void testScenario() throws Exception {
            try {
                MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);
                Assert.fail("should have thrown SoaErrorException");
            } catch (SoaErrorException e) {
                ConsultCarmedInterventionResponse response = (ConsultCarmedInterventionResponse) e.getResponseTypeV2();
                String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
                MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse> helper = new MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse>(ConsultCarmedInterventionResponse.class, ConsultCarmedInterventionResponse.class);
                XmlAsserter.assertSimilar(expected, helper.toString(response));
            }
        }
    }

    @RunWith(Parameterized.class)
    public static class WithoutSoaErrorIntegrationTest {

        @ClassRule
        public static RuleChain chain = RuleChain.outerRule(new RestoreSystemProperties()).around(SessionRule.withActiveSession().forEnvironment("acc").forProfession("orgpharmacy").forUser("hannes").build());


        private String scenarioId;
        private String scenarioName;
        private ConsultCarmedInterventionRequest request;


        public WithoutSoaErrorIntegrationTest(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
            super();
            this.scenarioId = scenarioId;
            this.scenarioName = scenarioName;
            this.request = request;
        }


        @Parameterized.Parameters(name = "{0}: {1}")
        public static Collection<Object[]> data() throws Exception {
         Collection<Object[]> testData = new ArrayList<Object[]>();
            testData.add(new Object[]{"scenario05", "Unknown ssin", aRequest().withSsin("86081523513").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario06", "Known ssin but dates nok", aRequest().withSsin("82041562197").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario07a", "Decision without required rights - missing coverage", aRequest().withSsin("68050555876").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario07b", "Decision without required rights - missing coverage", aRequest().withSsin("76091051964").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario08", "Decision without required rights - coverage exists but nok", aRequest().withSsin("61011060097").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario09", "Decision without required rights - coverage ok but care provider nok", aRequest().withSsin("98032463545").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario10", "Decision pending", aRequest().withSsin("69082054866").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario11", "Decision is active - full support - extras via list", aRequest().withSsin("64101567102").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario12", "Decision is active - full support - extras via convention", aRequest().withSsin("79053045251").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario13", "Decision is active - partial support - all extras", aRequest().withSsin("79030750196").withReferenceDate(new DateTime()).build()});
            testData.add(new Object[]{"scenario14", "Decision is active - partial support - no extras", aRequest().withSsin("87102148011").withReferenceDate(new DateTime()).build()});

           return testData;
        }

        @Test
        public void testScenario() throws Exception {
            ConsultCarmedInterventionResponse response = MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);

            String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
            MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse> helper = new MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse>(ConsultCarmedInterventionResponse.class, ConsultCarmedInterventionResponse.class);
            XmlAsserter.assertSimilar(expected, helper.toString(response));
        }
    }
    

    private static String expectedResponse(DateTime consultationDate, String requestId, String scenarioId) {
        Map<String, Object> templateContext = new HashMap<String, Object>();
        templateContext.put("requestId", requestId);
        templateContext.put("consultationDate", DateUtils.printDate(consultationDate));
        templateContext.put("startDate", DateUtils.printDate(new DateTime()));
        templateContext.put("endDate", DateUtils.printDate(new DateTime()));
        return TemplateEngineUtils.generate(templateContext, "/consultation/expected/orgpharmacy/" + scenarioId + "Response.xml");
    }
}
